/*
 * JBoss, Home of Professional Open Source
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */ 
package org.jboss.ha.framework.interfaces;

import org.jboss.util.NestedException;

/**
 * Generic clustering exception that can be used to replace other exceptions
 * that occur on the server. Thus, only this class is needed on the client side
 * and some specific server side exceptions classes are not needed on the client side
 * (such as JMX exceptions for example). 
 * Furthermore, it has support for "COMPLETED" status flag a la IIOP.
 * <p>
 * When an invocation reaches a node, it may be invoked on the actual
 * target or not (or not completely). If {@link #COMPLETED_NO} and working in 
 * a clustered environment, the client proxy is allowed to invoke
 * the same invocation on a different node. Otherwise, it will depend
 * if the target method is idempotent: this is no more the problem of
 * this class but rather the meta-data of the business environment
 * that can give this information
 * </p>
 *
 * @author  <a href="mailto:sacha.labourey@cogito-info.ch">Sacha Labourey</a>.
 * @version $Revision: 74878 $
 *
 * <p><b>Revisions:</b>
 *
 * <p><b>8 avril 2002 Sacha Labourey:</b>
 * <ul>
 * <li> First implementation </li>
 * </ul>
 */

public class GenericClusteringException
   extends NestedException
{
   // BES - 2007/11/12: This wasn't set in 4.x.  If this causes a problem in a 
   // compatibility test, just switch it to the old value; I didn't change
   // the serialized form of the class at all.   
   /** The serialVersionUID */
   private static final long serialVersionUID = 3030093807459327632L;
   
   /** Means the invocation that generated this exception was invoked on the server-side target */
   public final static int COMPLETED_YES = 0;
   /** Means the invocation that generated this exception was *not* invoked on the server-side target */
   public final static int COMPLETED_NO = 1;
   /** 
    * Means it is unclear whether the invocation that generated this exception 
    * was invoked on the server-side target 
    */
   public final static int COMPLETED_MAYBE = 2;
   
   public boolean isDefinitive = true;
   
   // if yes, then the invocation may be retried against another node
   // because the state has not been modified by the current invocation
   //
   public int completed;
   
   public GenericClusteringException ()
   {
      this.completed = COMPLETED_MAYBE;
   }
   
   public GenericClusteringException (int CompletionStatus)
   {
      this.completed = CompletionStatus;
   }

   public GenericClusteringException (int CompletionStatus, String s)
   {
      super(s);
      this.completed = CompletionStatus;
   }

   public GenericClusteringException (int CompletionStatus, String s, boolean isDefinitive)
   {
      super(s);
      this.completed = CompletionStatus;
      this.isDefinitive = isDefinitive;
   }

   public GenericClusteringException (int CompletionStatus, Throwable nested, boolean isDefinitive)
   {
      super(nested);
      this.completed = CompletionStatus;
      this.isDefinitive = isDefinitive;
   }
   
   public GenericClusteringException (int CompletionStatus, Throwable nested)
   {
      super(nested);
      this.completed = CompletionStatus;
   }
   
   /** 
    * Gets whether the invocation that generated this exception was completely
    * invoked on the server side target.
    * <p>
    * When an invocation reaches a node, it may be invoked on the actual
    * target or not (or not completely). If COMPLETED_NO and working in 
    * a clustered environment, the client proxy is allowed to invoke
    * the same invocation on a different node. Otherwise, it will depend
    * if the target method is idempotent: this is not the problem of
    * this class but rather the meta-data of the business environment
    * that can give this information
    * </p>
    * 
    * @return {@link #COMPLETED_YES}, {@link #COMPLETED_NO} or 
    *         {@link #COMPLETED_MAYBE}.
    */
   public int getCompletionStatus ()
   {
      return this.completed;
   }
   
   public void setCompletionStatus (int completionStatus)
   {
      this.completed = completionStatus;
   }
   
   /**
    * Indicates if the exception will most probably be repetitive (definitive) 
    * or if it is just a temporary failure and new attempts should be tried.
    */
   public boolean isDefinitive ()
   {
      return this.isDefinitive;      
   }
   
   public void setIsDefinitive (boolean definitive)
   {
      this.isDefinitive = definitive;
   }
}
