/*
 * JBoss, Home of Professional Open Source
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.profileservice.management.upload;

import java.net.URL;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.ResourceBundle;

import org.jboss.deployers.spi.management.deploy.DeploymentManager;
import org.jboss.deployers.spi.management.deploy.DeploymentProgress;
import org.jboss.deployers.spi.management.deploy.DeploymentTarget;
import org.jboss.deployers.spi.management.deploy.DeploymentStatus.CommandType;
import org.jboss.logging.Logger;
import org.jboss.managed.api.ManagedDeployment.DeploymentPhase;
import org.jboss.profileservice.management.upload.remoting.DeployHandler;
import org.jboss.profileservice.management.upload.remoting.StreamingDeploymentTarget;
import org.jboss.profileservice.spi.DeploymentRepository;
import org.jboss.profileservice.spi.DeploymentRepositoryFactory;
import org.jboss.profileservice.spi.NoSuchProfileException;
import org.jboss.profileservice.spi.Profile;
import org.jboss.profileservice.spi.ProfileKey;
import org.jboss.profileservice.spi.ProfileService;
import org.jboss.remoting.InvokerLocator;

/**
 * The remoting base DeploymentManager implementation
 * @author Scott.Stark@jboss.org
 * @version $Revision: 77864 $
 */
public class DeploymentManagerImpl implements DeploymentManager
{
   private static Logger log = Logger.getLogger(DeploymentManagerImpl.class);
   private static final String BUNDLE_NAME = "org.jboss.profileservice.management.upload.messages"; //$NON-NLS-1$
   /** */
   private ProfileService ps;
   /** The currently loaded profile */
   private Profile activeProfile;
   private DeploymentRepositoryFactory repositoryFactory;
   private DeployHandler deployHandler;

   /** */
   private ResourceBundle i18n;
   /** */
   private Locale currentLocale;
   /** */
   private MessageFormat formatter = new MessageFormat("");
   private InvokerLocator locator;
   private String remotingSubsystem = "DeploymentManager";

   public DeploymentManagerImpl()
   {
      currentLocale = Locale.getDefault();
      formatter.setLocale(currentLocale);
      i18n = ResourceBundle.getBundle(BUNDLE_NAME, currentLocale);
   }

   
   public ProfileService getProfileService()
   {
      return ps;
   }
   public void setProfileService(ProfileService ps)
   {
      this.ps = ps;
   }

   public DeploymentRepositoryFactory getRepositoryFactory()
   {
      return repositoryFactory;
   }
   public void setRepositoryFactory(DeploymentRepositoryFactory repositoryFactory)
   {
      this.repositoryFactory = repositoryFactory;
   }

   public DeployHandler getDeployHandler()
   {
      return deployHandler;
   }
   public void setDeployHandler(DeployHandler deployHandler)
   {
      this.deployHandler = deployHandler;
   }

   public InvokerLocator getLocator()
   {
      return locator;
   }
   public void setLocator(InvokerLocator locator)
   {
      this.locator = locator;
   }

   public String getRemotingSubsystem()
   {
      return remotingSubsystem;
   }
   public void setRemotingSubsystem(String remotingSubsystem)
   {
      this.remotingSubsystem = remotingSubsystem;
   }

   public DeploymentProgress distribute(String name, DeploymentPhase phase, URL contentURL)
   {
      return distribute(name, phase, contentURL, true);
   }

   public DeploymentProgress distribute(String name, DeploymentPhase phase, URL contentURL, boolean copyContent)
   {
      List<DeploymentTarget> targets = getDeploymentTargets();
      SerializableDeploymentID deployment = new SerializableDeploymentID(name, phase, contentURL.toString());
      deployment.setContentURL(contentURL);
      deployment.setCopyContent(copyContent);
      return new DeploymentProgressImpl(targets, deployment, CommandType.DISTRIBUTE);
   }

   public String[] getRepositoryNames(String[] names, DeploymentPhase phase) throws Exception
   {
      List<DeploymentTarget> targets = getDeploymentTargets();
      return targets.get(0).getRepositoryNames(names, phase);
   }

   public boolean isRedeploySupported()
   {
      return false;
   }

   public void loadProfile(ProfileKey key, boolean allowHotDeployments) throws Exception
   {
      activeProfile = ps.getProfile(key);
      if( activeProfile == null )
      {
         formatter.applyPattern(i18n.getString("DeploymentManager.NoSuchProfileException")); //$NON-NLS-1$
         Object[] args = {key};
         String msg = formatter.format(args);
         throw new NoSuchProfileException(msg);
      }
      activeProfile.enableModifiedDeploymentChecks(allowHotDeployments);
      // Set the deployment repository on the handler
      DeploymentRepository repository = repositoryFactory.getDeploymentRepository(key);
      deployHandler.setDeploymentRepository(repository);
   }

   public void releaseProfile(ProfileKey key, boolean allowHotDeployments)
         throws Exception
   {
      activeProfile = ps.getProfile(key);
      if( activeProfile == null )
      {
         formatter.applyPattern(i18n.getString("DeploymentManager.NoSuchProfileException")); //$NON-NLS-1$
         Object[] args = {key};
         String msg = formatter.format(args);
         throw new NoSuchProfileException(msg);
      }
      activeProfile.enableModifiedDeploymentChecks(allowHotDeployments);
      deployHandler.setDeploymentRepository(null);
   }

   public DeploymentProgress redeploy(String name, DeploymentPhase phase, URL contentURL)
   {
      List<DeploymentTarget> targets = getDeploymentTargets();
      SerializableDeploymentID deployment = new SerializableDeploymentID(name, phase, null);
      return new DeploymentProgressImpl(targets, deployment, CommandType.REDEPLOY);
   }

   public DeploymentProgress prepare(DeploymentPhase phase, String... names) throws Exception
   {
      return doProgress(CommandType.PREPARE, phase, names);
   }

   public DeploymentProgress start(DeploymentPhase phase, String... names) throws Exception
   {
      return doProgress(CommandType.START, phase, names);
   }

   public DeploymentProgress stop(DeploymentPhase phase, String... names) throws Exception
   {
      return doProgress(CommandType.STOP, phase, names);
   }

   public DeploymentProgress undeploy(DeploymentPhase phase, String... names) throws Exception
   {
      return doProgress(CommandType.UNDEPLOY, phase, names);
   }

   protected DeploymentProgress doProgress(CommandType type, DeploymentPhase phase, String... names)
   {
      if (names == null || names.length == 0)
         log.warn("Null or empty names.");

      List<DeploymentTarget> targets = getDeploymentTargets();
      SerializableDeploymentID deployment = new SerializableDeploymentID(names, phase, null);
      return new DeploymentProgressImpl(targets, deployment, type);
   }

   /**
    * TODO: should the targets include cluster info
    * @param name
    * @return
    */
   protected List<DeploymentTarget> getDeploymentTargets()
   {
      String targetName = locator.getHost();
      List<DeploymentTarget> targets = new ArrayList<DeploymentTarget>();
      StreamingDeploymentTarget hostTarget = new StreamingDeploymentTarget(locator, targetName, remotingSubsystem);
      targets.add(hostTarget);
      return targets;
   }
   
}
