/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.system.microcontainer;

import java.util.List;

import javax.management.ObjectName;

import org.jboss.dependency.spi.ControllerState;
import org.jboss.system.ServiceConfigurator;
import org.jboss.system.ServiceContext;
import org.jboss.system.ServiceController;
import org.jboss.system.metadata.ServiceAttributeMetaData;
import org.jboss.system.metadata.ServiceMetaData;

/**
 * ConfigureAction. Called when installing to/uninstalling from the 
 * {@link ControllerState#CONFIGURED} state. Configures the MBean instance
 * with values from the {@link ServiceMetaData} on install.
 * 
 * @see ServiceControllerContextActions
 * @author <a href="adrian@jboss.com">Adrian Brock</a>
 * @author <a href="ales.justin@jboss.org">Ales Justin</a>
 */
public class ConfigureAction extends ServiceControllerContextAction
{
   /**
    * Configures the MBean instance with values from the ServiceMetaData.
    * 
    * @param context The ServiceControllerContext we are installing
    */
   public void installAction(ServiceControllerContext context) throws Throwable
   {
      // Configure any attributes
      ServiceMetaData metaData = context.getServiceMetaData();
      if (metaData != null)
      {
         ServiceController controller = context.getServiceController();
         ObjectName objectName = context.getObjectName();

         List<ServiceAttributeMetaData> attrs = metaData.getAttributes();
         if (attrs.size() > 0)
         {
            ObjectName classLoaderName = metaData.getClassLoaderName(); 
            ServiceConfigurator.configure(null, controller, objectName, classLoaderName, attrs);
         }
      }
      
      context.getServiceContext().state = ServiceContext.CONFIGURED;
   }

   /**
    * Unconfigure the mbean.
    * Unget values - releasing potential injections.
    * 
    * @param context The ServiceControllerContext we are uninstalling
    */
   public void uninstallAction(ServiceControllerContext context)
   {
      // Unconfigure any attributes
      ServiceMetaData metaData = context.getServiceMetaData();
      if (metaData != null)
      {
         ServiceController controller = context.getServiceController();
         ObjectName objectName = context.getObjectName();

         List<ServiceAttributeMetaData> attrs = metaData.getAttributes();
         if (attrs.size() > 0)
         {
            ObjectName classLoaderName = metaData.getClassLoaderName();
            try
            {
               ServiceConfigurator.unconfigure(null, controller, objectName, classLoaderName, attrs);
            }
            catch (Exception ignored)
            {
            }
         }
      }
   }
}
