/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.system.microcontainer;

import org.jboss.dependency.spi.ControllerState;
import org.jboss.dependency.spi.tracker.ContextRegistry;
import org.jboss.kernel.Kernel;
import org.jboss.system.ServiceController;
import org.jboss.system.ServiceCreator;
import org.jboss.system.ServiceInstance;
import org.jboss.system.metadata.ServiceMetaData;

import javax.management.MBeanServer;
import javax.management.ObjectName;

/**
 * InstantiateAction. Called when installing to/uninstalling from the 
 * {@link ControllerState#INSTANTIATED} state. Creates an instance of
 * the MBean and registers it in the MBeanServer on ininstall, and unregisters
 * it from the MBeanServer on uninstall.
 * 
 * @see ServiceControllerContextActions
 * @author <a href="adrian@jboss.com">Adrian Brock</a>
 * @author <a href="ales.justin@jboss.org">Ales Justin</a>
 * @version $Revision: 100961 $
 */
public class InstantiateAction extends ServiceControllerContextAction
{
   /**
    * Get context registry.
    *
    * @param context the current context
    * @return the context registry
    */
   protected ContextRegistry getRegistry(ServiceControllerContext context)
   {
      ServiceController serviceController = context.getServiceController();
      Kernel kernel = serviceController.getKernel();
      return kernel.getController();
   }

   /**
    * Creates an instance of the MBean from the metadata stored in the context,
    * and registers it in the MBeanServer.
    * 
    * @param context The ServiceControllerContext we are installing
    */
   public void installAction(ServiceControllerContext context) throws Throwable
   {
      MBeanServer server = context.getServiceController().getMBeanServer();
      ObjectName objectName = context.getObjectName();
      ServiceMetaData metaData = context.getServiceMetaData();
      Object mbean = context.getTarget();
      ServiceInstance instance = ServiceCreator.install(server, objectName, metaData, mbean);
      context.setTarget(instance.getResource());

      ContextRegistry registry = getRegistry(context);
      registry.addInstantiatedContext(context);
   }

   /**
    * Unregisters the MBean from the the MBeanServer.
    * 
    * @param context The ServiceControllerContext we are uninstalling
    */
   public void uninstallAction(ServiceControllerContext context)
   {
      ContextRegistry registry = getRegistry(context);
      registry.removeInstantiatedContext(context);

      MBeanServer server = context.getServiceController().getMBeanServer();
      ObjectName objectName = context.getObjectName();
      ServiceCreator.uninstall(server, objectName);
      ServiceMetaData metaData = context.getServiceMetaData();
      if (metaData != null)
         context.setTarget(null);
   }
}
