/*
 * #%L
 * JBossOSGi VFS VFS30
 * %%
 * Copyright (C) 2010 - 2012 JBoss by Red Hat
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 2.1 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-2.1.html>.
 * #L%
 */
/*
 * JBoss, Home of Professional Open Source
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.osgi.vfs30;

import static org.jboss.osgi.vfs.VFSMessages.MESSAGES;

import java.io.IOException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.jboss.vfs.VirtualFile;
import org.jboss.vfs.VirtualFileVisitor;
import org.jboss.vfs.VisitorAttributes;
import org.jboss.vfs.util.MatchAllVirtualFileFilter;

/**
 * An enumeration of VFS entries.
 * 
 * @author <a href="adrian@jboss.com">Adrian Brock</a>
 * @author thomas.diesler@jboss.com
 * @author <a href="david@redhat.com">David Bosschaert</a>
 * @version $Revision: 1.1 $
 */
class VFSFindEntriesEnumeration implements Enumeration<URL> {

    /** The paths */
    private Iterator<URL> paths;

    /**
     * Create a new VFSFindEntriesEnumeration.
     *
     * @param root the root file
     * @param file the file to enumerate
     * @param filePattern the file pattern
     * @param recurse whether to recurse
     * @throws IOException for any error
     */
    public VFSFindEntriesEnumeration(VirtualFile root, VirtualFile file, String filePattern, boolean recurse) throws IOException {
        if (root == null)
            throw MESSAGES.illegalArgumentNull("root");
        if (file == null)
            throw MESSAGES.illegalArgumentNull("file");

        String rootPath = root.getPathName();
        VisitorAttributes attributes = new VisitorAttributes();
        attributes.setIncludeRoot(false);
        attributes.setLeavesOnly(false);
        if (recurse)
            attributes.setRecurseFilter(MatchAllVirtualFileFilter.INSTANCE);

        VisitorImpl visitor = new VisitorImpl(rootPath, filePattern, attributes);
        file.visit(visitor);

        this.paths = visitor.paths.iterator();
    }

    public boolean hasMoreElements() {
        return paths.hasNext();
    }

    public URL nextElement() {
        return paths.next();
    }

    static class VisitorImpl implements VirtualFileVisitor {

        ArrayList<URL> paths = new ArrayList<URL>();

        Pattern filter;
        String rootPath;
        VisitorAttributes attributes;

        VisitorImpl(String rootPath, String filter, VisitorAttributes attributes) {
            this.rootPath = rootPath;
            this.filter = convertToPattern(filter);
            this.attributes = attributes;
        }

        public VisitorAttributes getAttributes() {
            return attributes;
        }

        public void visit(VirtualFile virtualFile) {
            // See if the filter matches
            Matcher matcher = filter.matcher(virtualFile.getName());
            if (matcher.find() == false)
                return;

            try {
                paths.add(virtualFile.toURL());
            } catch (Exception e) {
                throw MESSAGES.runtimeErrorVistingFile(e, virtualFile);
            }
        }

        // Convert file pattern (RFC 1960-based Filter) into a RegEx pattern
        private static Pattern convertToPattern(String filePattern) {
            filePattern = filePattern.replace("*", ".*");
            return Pattern.compile("^" + filePattern + "$");
        }

    }
}
