/******************************************************************************
 * JBoss, a division of Red Hat                                               *
 * Copyright 2006, Red Hat Middleware, LLC, and individual                    *
 * contributors as indicated by the @authors tag. See the                     *
 * copyright.txt in the distribution for a full listing of                    *
 * individual contributors.                                                   *
 *                                                                            *
 * This is free software; you can redistribute it and/or modify it            *
 * under the terms of the GNU Lesser General Public License as                *
 * published by the Free Software Foundation; either version 2.1 of           *
 * the License, or (at your option) any later version.                        *
 *                                                                            *
 * This software is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU           *
 * Lesser General Public License for more details.                            *
 *                                                                            *
 * You should have received a copy of the GNU Lesser General Public           *
 * License along with this software; if not, write to the Free                *
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA         *
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.                   *
 ******************************************************************************/
package org.jboss.unit;

import java.io.Serializable;
import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.Collection;

/**
 * A test id made of a sequence of non null names. As the name indicates it uniquely identity a test in the
 * of a test driver.
 *
 * @author <a href="mailto:julien@jboss.org">Julien Viet</a>
 * @version $Revision: 1.1 $
 */
public final class TestId implements Serializable
{
   
   /** . */
   private final String[] names;

   public TestId()
   {
      this.names = new String[0];
   }

   /**
    * @param prefix the prefix
    * @param name the name
    * @throws IllegalArgumentException if any argument is null
    */
   public TestId(TestId prefix, String name) throws IllegalArgumentException
   {
      if (prefix == null)
      {
         throw new IllegalArgumentException();
      }
      if (name == null)
      {
         throw new IllegalArgumentException();
      }

      //
      String[] names = new String[prefix.names.length + 1];
      System.arraycopy(prefix.names, 0, names, 0, prefix.names.length);
      names[prefix.names.length] = name;

      //
      this.names = names;
   }

   /**
    *
    * @param name the name
    * @param suffix the suffix
    * @throws IllegalArgumentException if any argument is null
    */
   public TestId(String name, TestId suffix) throws IllegalArgumentException
   {
      if (suffix == null)
      {
         throw new IllegalArgumentException();
      }
      if (name == null)
      {
         throw new IllegalArgumentException();
      }

      //
      String[] names = new String[1 + suffix.names.length];
      names[0] = name;
      System.arraycopy(suffix.names, 0, names, 1, suffix.names.length);

      //
      this.names = names;
   }


   /**
    * @param names if the names array is null or any composite of this name is null
    */
   public TestId(String... names) throws IllegalArgumentException
   {
      if (names == null)
      {
         throw new IllegalArgumentException();
      }

      //
      String[] tmp = names.clone();

      //
      for (String name : tmp)
      {
         if (name == null)
         {
            throw new IllegalArgumentException();
         }
      }

      //
      this.names = tmp;
   }

   /**
    * @param names the collection of names
    * @throws IllegalArgumentException if the collection argument is null or any value in the collection is null
    */
   public TestId(Collection<String> names) throws IllegalArgumentException
   {
      if (names == null)
      {
         throw new IllegalArgumentException();
      }

      //
      String[] tmp = names.toArray(new String[names.size()]);

      //
      for (String name : tmp)
      {
         if (name == null)
         {
            throw new IllegalArgumentException();
         }
      }

      //
      this.names = tmp;
   }

   /**
    * Returns an iterator over the names composing this id.
    *
    * @return the name iterator
    */
   public Iterator<String> iterator()
   {
      return new Iterator<String>()
      {
         /** . */
         int index = 0;

         public boolean hasNext()
         {
            return index < names.length;
         }

         public String next()
         {
            if (hasNext())
            {
               return names[index++];
            }
            throw new NoSuchElementException();
         }

         public void remove()
         {
            throw new UnsupportedOperationException();
         }
      };
   }

   /**
    * Return a name in the id.
    *
    * @param index the name index to return
    * @return the specified name
    * @throws IllegalArgumentException if the index is lower than zero or greater than the value returned by <code>getLength()</code>
    */
   public String getName(int index) throws IllegalArgumentException
   {
      if (index < 0)
      {
         throw new IllegalArgumentException("Index cannot be negative");
      }
      if (index >= names.length)
      {
         throw new IllegalArgumentException("Index cannot be greater than the length " + names.length);
      }
      return names[index];
   }

   /**
    * Return the number of names composing this id.
    *
    * @return the length
    */
   public int getLength()
   {
      return names.length;
   }

   /**
    * Create a new id by using the names from the specified index up to the last one.
    * @param from the specified index
    * @return a new id
    * throws IllegalArgumentException if the specified index is negative or greater than the length
    */
   public TestId range(int from) throws IllegalArgumentException
   {
      if (from < 0)
      {
         throw new IllegalArgumentException();
      }
      if (from > names.length)
      {
         throw new IllegalArgumentException();
      }

      //
      String[] subNames = new String[names.length - from];
      System.arraycopy(names, from, subNames, 0, subNames.length);
      return new TestId(subNames);
   }

   public int hashCode()
   {
      int hashCode = 0;

      //
      for (String name : names)
      {
         hashCode = hashCode * 41 + name.hashCode();
      }

      //
      return hashCode;
   }

   public boolean equals(Object object)
   {
      if (this == object)
      {
         return true;
      }

      //
      if (object instanceof TestId)
      {
         TestId that = (TestId)object;

         //
         if (this.names.length != that.names.length)
         {
            return false;
         }

         //
         for (int i = 0;i < this.names.length;i++)
         {
            String thisName = this.names[i];
            String thatName = that.names[i];

            //
            if (!thisName.equals(thatName))
            {
               return false;
            }
         }

         //
         return true;
      }

      //
      return false;
   }

   public String toString()
   {
      StringBuffer tmp = new StringBuffer("TestId[");

      //
      for (int i = 0;i < names.length;i++)
      {
         if (i > 0)
         {
            tmp.append(',');
         }

         //
         tmp.append("\"").append(names[i]).append("\"");
      }

      //
      tmp.append(']');

      //
      return tmp.toString();
   }
}
