/******************************************************************************
 * JBoss, a division of Red Hat                                               *
 * Copyright 2006, Red Hat Middleware, LLC, and individual                    *
 * contributors as indicated by the @authors tag. See the                     *
 * copyright.txt in the distribution for a full listing of                    *
 * individual contributors.                                                   *
 *                                                                            *
 * This is free software; you can redistribute it and/or modify it            *
 * under the terms of the GNU Lesser General Public License as                *
 * published by the Free Software Foundation; either version 2.1 of           *
 * the License, or (at your option) any later version.                        *
 *                                                                            *
 * This software is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU           *
 * Lesser General Public License for more details.                            *
 *                                                                            *
 * You should have received a copy of the GNU Lesser General Public           *
 * License along with this software; if not, write to the Free                *
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA         *
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.                   *
 ******************************************************************************/
package org.jboss.unit;

import org.jboss.unit.util.CollectionTools;

import java.util.List;
import java.util.ArrayList;

/**
 * @author <a href="mailto:julien@jboss.org">Julien Viet</a>
 * @version $Revision: 1.1 $
 */
public abstract class TestIdFormat
{

   /**
    * Parse a fqn representation of a test id.
    *
    * @param fqn the fqn to parse
    * @return the test id value
    * @throws IllegalArgumentException if the argument is not valid
    */
   public abstract TestId parse(String fqn) throws IllegalArgumentException;

   public abstract String toString(TestId testId) throws IllegalArgumentException;

   public static final TestIdFormat CLASSIC = new TestIdFormat()
   {
      public TestId parse(String fqn)
      {
         if (fqn == null)
         {
            throw new IllegalArgumentException("No null fqn accepted");
         }

         //
         int from = 0;
         List<String> names = new ArrayList<String>();

         //
         do
         {
            int pos = fqn.indexOf('.', from);

            //
            if (pos == -1)
            {
               names.add(fqn.substring(from));
               break;
            }

            //
            names.add(fqn.substring(from, pos));

            //
            from = pos + 1;
         }
         while (true);

         //
         return new TestId(names);
      }

      public String toString(TestId testId)
      {
         if (testId == null)
         {
            throw new IllegalArgumentException("No null test id accepted");
         }

         //
         if (testId.getLength() == 0)
         {
            return "";
         }

         //
         StringBuilder builder = new StringBuilder();
         for (String name : CollectionTools.iterable(testId.iterator()))
         {
            builder.append(name);
            builder.append('.');
         }

         //
         return builder.substring(0, builder.length() - 1);
      }
   };

}
