/******************************************************************************
 * JBoss, a division of Red Hat                                               *
 * Copyright 2006, Red Hat Middleware, LLC, and individual                    *
 * contributors as indicated by the @authors tag. See the                     *
 * copyright.txt in the distribution for a full listing of                    *
 * individual contributors.                                                   *
 *                                                                            *
 * This is free software; you can redistribute it and/or modify it            *
 * under the terms of the GNU Lesser General Public License as                *
 * published by the Free Software Foundation; either version 2.1 of           *
 * the License, or (at your option) any later version.                        *
 *                                                                            *
 * This software is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU           *
 * Lesser General Public License for more details.                            *
 *                                                                            *
 * You should have received a copy of the GNU Lesser General Public           *
 * License along with this software; if not, write to the Free                *
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA         *
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.                   *
 ******************************************************************************/
package org.jboss.unit.api.pojo.junit;

import org.jboss.unit.spi.pojo.TestLifeCycle;
import org.jboss.unit.spi.pojo.TestCaseLifeCycleException;

import java.util.Map;
import java.lang.reflect.InvocationTargetException;

/**
 * @author <a href="mailto:julien@jboss.org">Julien Viet</a>
 * @version $Revision: 1.1 $
 */
class JUnitLifeCycle implements TestLifeCycle
{

   /** . */
   private final JUnitProvider provider;

   public JUnitLifeCycle(JUnitProvider testCase)
   {
      this.provider = testCase;
   }

   public org.jboss.unit.spi.pojo.TestCase newTestCase(String name) throws TestCaseLifeCycleException
   {
      JUnitTestMethod method = provider.descriptor.testMap.get(name);
      if (method == null)
      {
         throw new TestCaseLifeCycleException("No test case " + name + " found");
      }

      try
      {
         TestCase testCase = (TestCase)provider.testClass.newInstance();
         return new JUnitTestMethodLifeCycle(testCase, method);
      }
      catch (InstantiationException e)
      {
         throw new TestCaseLifeCycleException(e);
      }
      catch (IllegalAccessException e)
      {
         throw new TestCaseLifeCycleException(e);
      }
   }

   public void testCaseParametrize(org.jboss.unit.spi.pojo.TestCase testCase, Map<String, String> parametrization) throws TestCaseLifeCycleException
   {
      // No op
   }

   public void testCaseCreate(org.jboss.unit.spi.pojo.TestCase testCase) throws TestCaseLifeCycleException
   {
      try
      {
         JUnitTestMethodLifeCycle lifeCycle = (JUnitTestMethodLifeCycle)testCase;
         lifeCycle.pojo.setUp();
      }
      catch (Throwable e)
      {
         throw unwrap(e);
      }
   }

   public void testCaseInvoke(org.jboss.unit.spi.pojo.TestCase testCase) throws TestCaseLifeCycleException
   {
      JUnitTestMethodLifeCycle lifeCycle = (JUnitTestMethodLifeCycle)testCase;

      // Invoke test joinpoint
      try
      {
         lifeCycle.testMethod.method.invoke(lifeCycle.pojo);
      }
      catch (IllegalAccessException e)
      {
         throw unwrap(e);
      }
      catch (InvocationTargetException e)
      {
         throw unwrap(e.getCause());
      }
   }

   private TestCaseLifeCycleException unwrap(Throwable throwable)
   {
      // Look if it's an assertion failure from junit API
      for (Class clazz = throwable.getClass();clazz != null;clazz = clazz.getSuperclass())
      {
         if ("junit.framework.AssertionFailedError".equals(clazz.getName()))
         {
            return new TestCaseLifeCycleException(new AssertionError(throwable));
         }
      }

      //
      return new TestCaseLifeCycleException(throwable);
   }

   public void testCaseDestroy(org.jboss.unit.spi.pojo.TestCase testCase)
   {
      try
      {
         JUnitTestMethodLifeCycle lifeCycle = (JUnitTestMethodLifeCycle)testCase;
         lifeCycle.pojo.tearDown();
      }
      catch (Throwable e)
      {
         e.printStackTrace();
      }
   }
}
