/******************************************************************************
 * JBoss, a division of Red Hat                                               *
 * Copyright 2006, Red Hat Middleware, LLC, and individual                    *
 * contributors as indicated by the @authors tag. See the                     *
 * copyright.txt in the distribution for a full listing of                    *
 * individual contributors.                                                   *
 *                                                                            *
 * This is free software; you can redistribute it and/or modify it            *
 * under the terms of the GNU Lesser General Public License as                *
 * published by the Free Software Foundation; either version 2.1 of           *
 * the License, or (at your option) any later version.                        *
 *                                                                            *
 * This software is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU           *
 * Lesser General Public License for more details.                            *
 *                                                                            *
 * You should have received a copy of the GNU Lesser General Public           *
 * License along with this software; if not, write to the Free                *
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA         *
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.                   *
 ******************************************************************************/
package org.jboss.unit.driver.impl.pojo;

import org.jboss.unit.spi.pojo.TestProvider;
import org.jboss.unit.spi.pojo.TestProviderSupport;
import org.jboss.unit.spi.pojo.TestCaseLifeCycleException;
import org.jboss.unit.spi.pojo.TestLifeCycle;
import org.jboss.unit.spi.pojo.TestCase;
import org.jboss.unit.spi.pojo.annotations.Provider;
import org.jboss.unit.driver.DriverResponse;
import org.jboss.unit.driver.DriverCommand;
import org.jboss.unit.driver.AbstractTestDriver;
import org.jboss.unit.driver.response.EndTestResponse;
import org.jboss.unit.driver.response.FailureResponse;
import org.jboss.unit.driver.command.StartTestCommand;
import org.jboss.unit.info.TestInfo;
import org.jboss.unit.info.TestSuiteInfo;
import org.jboss.unit.TestId;
import org.jboss.unit.Failure;

import java.util.Map;
import java.lang.reflect.AnnotatedElement;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.annotation.Annotation;

/**
 * @author <a href="mailto:julien@jboss.org">Julien Viet</a>
 * @version $Revision: 1.1 $
 */
public class POJOTestSuiteDriver extends AbstractTestDriver
{

   /** . */
   private TestProvider provider;

   /** . */
   private TestSuiteInfo info;

   public POJOTestSuiteDriver(Class testClass) throws InstantiationException, IllegalAccessException
   {
      // Get provider annotation
      Provider provider = ((AnnotatedElement)testClass).getAnnotation(Provider.class);

      // Scan meta annotations
      if (provider == null)
      {
         for (Annotation annotation : testClass.getAnnotations())
         {
            Class blah = annotation.annotationType();
            provider = ((AnnotatedElement)blah).getAnnotation(Provider.class);
            if (provider != null)
            {
               break;
            }
         }
      }

      if (provider != null)
      {
         try
         {
            Class<? extends TestProvider> providerClass = provider.value();
            Constructor<? extends TestProvider> providerCtor = providerClass.getConstructor(Class.class);
            this.provider = providerCtor.newInstance(testClass);
         }
         catch (NoSuchMethodException e)
         {
            throw new IllegalArgumentException(e);
         }
         catch (InvocationTargetException e)
         {
            throw new IllegalArgumentException(e);
         }
      }
//      else if (TestHandler.class.isAssignableFrom(testClass))
//      {
//         handler = (TestHandler)testClass.newInstance();
//      }
      else
      {
         this.provider = new TestProviderSupport(testClass);
      }

      //
      info = new POJOTestSuiteInfo(this.provider.getDescriptor());
   }

   public TestInfo getInfo()
   {
      return info;
   }

   public DriverResponse invoke(TestId id, DriverCommand command)
   {
      if (command instanceof StartTestCommand)
      {
         StartTestCommand start = (StartTestCommand)command;

         //
         if (id.getLength() == 0)
         {
            throw new IllegalArgumentException();
         }

         //
         TestLifeCycle lifeCycle = provider.getLifeCycle();

         //
         String name = id.getName(0);

         //
         TestCase testCase;
         try
         {
            testCase = lifeCycle.newTestCase(name);
         }
         catch (Exception e)
         {
            return createResponse(e);
         }

         //
         Map<String,String> parametrization = start.getParametrization();

         //
         try
         {
            lifeCycle.testCaseParametrize(testCase, parametrization);
         }
         catch (Exception e)
         {
            return createResponse(e);
         }

         //
         try
         {
            lifeCycle.testCaseCreate(testCase);

            //
            lifeCycle.testCaseInvoke(testCase);
         }
         catch (Exception e)
         {
            return createResponse(e);
         }
         finally
         {
            lifeCycle.testCaseDestroy(testCase);
         }

         //
         return new EndTestResponse();
      }
      else
      {
         throw new IllegalArgumentException(); // convert to response style
      }
   }

   private DriverResponse createResponse(Throwable t)
   {
      if (t instanceof TestCaseLifeCycleException)
      {
         if (t.getCause() != null)
         {
            t = t.getCause();
         }
      }

      //
      return new FailureResponse(Failure.createFailure(t));
   }
}
