/******************************************************************************
 * JBoss, a division of Red Hat                                               *
 * Copyright 2006, Red Hat Middleware, LLC, and individual                    *
 * contributors as indicated by the @authors tag. See the                     *
 * copyright.txt in the distribution for a full listing of                    *
 * individual contributors.                                                   *
 *                                                                            *
 * This is free software; you can redistribute it and/or modify it            *
 * under the terms of the GNU Lesser General Public License as                *
 * published by the Free Software Foundation; either version 2.1 of           *
 * the License, or (at your option) any later version.                        *
 *                                                                            *
 * This software is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU           *
 * Lesser General Public License for more details.                            *
 *                                                                            *
 * You should have received a copy of the GNU Lesser General Public           *
 * License along with this software; if not, write to the Free                *
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA         *
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.                   *
 ******************************************************************************/
package org.jboss.unit.info.impl;

import org.jboss.unit.info.TestInfo;
import org.jboss.unit.info.ParameterInfo;
import org.jboss.unit.info.TestSuiteInfo;
import org.jboss.unit.info.TestCaseInfo;

import java.util.Map;
import java.util.Set;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Collections;

/**
 * @author <a href="mailto:julien@jboss.org">Julien Viet</a>
 * @version $Revision: 1.1 $
 */
public abstract class SimpleTestInfo implements TestInfo
{

   /** . */
   protected final String name;

   /** . */
   protected String description;

   /** . */
   protected Map<String, SimpleParameterInfo> parameters;

   /** . */
   protected Set<String> keywords;

   public SimpleTestInfo(String name)
   {
      if (name == null)
      {
         throw new IllegalArgumentException();
      }

      //
      this.name = name;
      this.description = null;
      this.parameters = new HashMap<String, SimpleParameterInfo>();
      this.keywords = new HashSet<String>();
   }

   public SimpleTestInfo(TestInfo that)
   {
      if (that == null)
      {
         throw new IllegalArgumentException();
      }

      //
      Map<String, SimpleParameterInfo> parameters = new HashMap<String, SimpleParameterInfo>();
      for (Map.Entry<String, ? extends ParameterInfo> entry : that.getParameters().entrySet())
      {
         parameters.put(entry.getKey(), new SimpleParameterInfo(entry.getValue()));
      }

      //
      this.name = that.getName();
      this.description = that.getDescription();
      this.keywords = new HashSet<String>(that.getKeywords());
      this.parameters = parameters;
   }

   public String getName()
   {
      return name;
   }

   public String getDescription()
   {
      return description;
   }

   public void setDescription(String description)
   {
      this.description = description;
   }

   public Map<String, SimpleParameterInfo> getParameters()
   {
      return Collections.unmodifiableMap(parameters);
   }

   public Set<String> getKeywords()
   {
      return Collections.unmodifiableSet(keywords);
   }

   public void setKeywords(Set<String> keywords)
   {
      if (keywords == null)
      {
         throw new IllegalArgumentException();
      }
      this.keywords = new HashSet<String>(keywords);
   }

   /**
    * Factory method for cloning the state of an existing test info.
    *
    * @param testInfo the inf oto clone
    * @return the clone
    * @throws IllegalArgumentException if the test info is null or has an incorrect state
    */
   public static SimpleTestInfo create(TestInfo testInfo) throws IllegalArgumentException
   {
      if (testInfo == null)
      {
         throw new IllegalArgumentException();
      }
      if (testInfo instanceof TestSuiteInfo)
      {
         TestSuiteInfo testSuite = (TestSuiteInfo)testInfo;
         return new SimpleTestSuiteInfo(testSuite);
      }
      else
      {
         TestCaseInfo testCase = (TestCaseInfo)testInfo;
         return new SimpleTestCaseInfo(testCase);
      }
   }
}
