/******************************************************************************
 * JBoss, a division of Red Hat                                               *
 * Copyright 2006, Red Hat Middleware, LLC, and individual                    *
 * contributors as indicated by the @authors tag. See the                     *
 * copyright.txt in the distribution for a full listing of                    *
 * individual contributors.                                                   *
 *                                                                            *
 * This is free software; you can redistribute it and/or modify it            *
 * under the terms of the GNU Lesser General Public License as                *
 * published by the Free Software Foundation; either version 2.1 of           *
 * the License, or (at your option) any later version.                        *
 *                                                                            *
 * This software is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU           *
 * Lesser General Public License for more details.                            *
 *                                                                            *
 * You should have received a copy of the GNU Lesser General Public           *
 * License along with this software; if not, write to the Free                *
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA         *
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.                   *
 ******************************************************************************/
package org.jboss.unit.report.impl.junit;

import org.jboss.unit.Failure;
import org.jboss.unit.TestId;
import org.jboss.unit.info.TestInfo;
import org.jboss.unit.runner.TestResult;
import org.jboss.unit.runner.TestRunnerEvent;
import org.jboss.unit.runner.TestRunnerEventListener;
import org.jboss.unit.runner.event.EndRunnerEvent;
import org.jboss.unit.runner.event.EndTestCaseEvent;
import org.jboss.unit.runner.event.EndTestSuiteEvent;
import org.jboss.unit.runner.event.RunnerFailureEvent;
import org.jboss.unit.runner.event.StartRunnerEvent;
import org.jboss.unit.runner.event.StartTestCaseEvent;
import org.jboss.unit.runner.event.StartTestSuiteEvent;
import org.jboss.unit.tooling.SystemOutputManager;
import org.jboss.unit.tooling.SystemOutputView;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * @author <a href="mailto:julien@jboss.org">Julien Viet</a>
 * @version $Revision: 1.1 $
 */
public abstract class AbstractJUnitReport implements TestRunnerEventListener
{

   /** The tested classes stack. */
   private TestSuite current;

   public void onEvent(TestRunnerEvent event)
   {
      if (event instanceof StartRunnerEvent)
      {
         handle((StartRunnerEvent) event);
      }
      else if (event instanceof EndRunnerEvent)
      {
         handle((EndRunnerEvent)event);
      }
      else if (event instanceof RunnerFailureEvent)
      {
         handle((RunnerFailureEvent) event);
      }
      else if (event instanceof StartTestSuiteEvent)
      {
         handle((StartTestSuiteEvent)event);
      }
      else if (event instanceof EndTestSuiteEvent)
      {
         handle((EndTestSuiteEvent)event);
      }
      else if (event instanceof StartTestCaseEvent)
      {
         handle((StartTestCaseEvent) event);
      }
      else if (event instanceof EndTestCaseEvent)
      {
         handle((EndTestCaseEvent) event);
      }
   }

   protected abstract void startRunner();

   protected abstract void endRunner();

   protected abstract void runnerFailure(TestSuite testSuite, Failure failure);

   protected abstract void endTestCase(TestSuite testSuite, String testName, TestResult result);

   protected abstract void startTestSuite(TestSuite testSuite);

   protected abstract void endTestSuite(TestSuite testSuite);

   private void handle(StartRunnerEvent event)
   {
      current = new TestSuite(null, "", new HashMap<String, String>());

      //
      startTestSuite(current);

      //
      startRunner();
   }

   private void handle(EndRunnerEvent event)
   {
      TestSuite tmp = current;
      current = null;

      //
      endTestSuite(tmp);

      //
      endRunner();
   }

   private void handle(StartTestSuiteEvent event)
   {
      current = new TestSuite(current, event.getTestInfo().getName(), event.getProperties());

      //
      startTestSuite(current);
   }

   private void handle(EndTestSuiteEvent event)
   {
      TestSuite testSuite = current;
      current = current.parent;

      //
      endTestSuite(testSuite);
   }

   private void handle(StartTestCaseEvent event)
   {
      current.startedTests.put(event.getTestId(), event.getTestInfo());
   }

   private void handle(RunnerFailureEvent event)
   {
      current.tests++;
      current.errors++;

      //
      runnerFailure(current, event.getFailure());
   }

   private void handle(EndTestCaseEvent event)
   {
      TestId testId = event.getTestId();
      TestResult result = event.getTestResult();
      TestInfo info = current.startedTests.get(testId);

      // remove test from started List
      current.startedTests.remove(testId);

      //
      current.durationMillis += result.getDurationMillis();
      current.tests++;

      //
      endTestCase(current, info.getName(), result);
   }

   public class TestSuite
   {

      /** . */
      final TestSuite parent;

      /** . */
      final String name;

      /** . */
      Map<String, String> properties;

      /** . */
      long durationMillis;

      /** The test count */
      int tests;

      /** The error count */
      int errors;

      /** The failed count */
      int failures;

      /** The started Tests */
      Map<TestId, TestInfo> startedTests = new HashMap<TestId, TestInfo>();

      /** The view of the system output. */
      SystemOutputView view = SystemOutputManager.getInstance().createView(false);

      /**
       * Todo: set properties
       *
       * @param name
       */
      TestSuite(TestSuite parent, String name, Map<String, String> properties)
      {
         this.parent = parent;
         this.name = name;
         this.properties = properties;
      }

      public StringBuffer getFQN()
      {
         if (parent == null)
         {
            return new StringBuffer("_none_");
         }
         else
         {
            List<TestSuite> tmp = new ArrayList<TestSuite>();
            for (TestSuite current = this;current.parent != null;current = current.parent)
            {
               tmp.add(current);
            }

            //
            Collections.reverse(tmp);

            // Skip the first top that will not participate in the name
            StringBuffer nb = new StringBuffer();
            for (Iterator<TestSuite> i = tmp.iterator();i.hasNext();)
            {
               TestSuite testSuite = i.next();
               nb.append(testSuite.name);
               if (i.hasNext())
               {
                  nb.append(".");
               }
            }
            return nb;
         }
      }
   }
}
