/******************************************************************************
 * JBoss, a division of Red Hat                                               *
 * Copyright 2006, Red Hat Middleware, LLC, and individual                    *
 * contributors as indicated by the @authors tag. See the                     *
 * copyright.txt in the distribution for a full listing of                    *
 * individual contributors.                                                   *
 *                                                                            *
 * This is free software; you can redistribute it and/or modify it            *
 * under the terms of the GNU Lesser General Public License as                *
 * published by the Free Software Foundation; either version 2.1 of           *
 * the License, or (at your option) any later version.                        *
 *                                                                            *
 * This software is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU           *
 * Lesser General Public License for more details.                            *
 *                                                                            *
 * You should have received a copy of the GNU Lesser General Public           *
 * License along with this software; if not, write to the Free                *
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA         *
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.                   *
 ******************************************************************************/
package org.jboss.unit.report.impl.junit;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.io.Writer;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.text.NumberFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Map;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Result;
import javax.xml.transform.Source;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.jboss.unit.Failure;
import org.jboss.unit.FailureType;
import org.jboss.unit.TestId;
import org.jboss.unit.info.TestInfo;
import org.jboss.unit.runner.TestResult;
import org.jboss.unit.runner.event.EndRunnerEvent;
import org.jboss.unit.runner.event.EndTestCaseEvent;
import org.jboss.unit.runner.event.RunnerFailureEvent;
import org.jboss.unit.runner.event.StartRunnerEvent;
import org.jboss.unit.runner.event.StartTestCaseEvent;
import org.jboss.unit.runner.results.TestFailure;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

/**
 * @author <a href="mailto:emuckenh@redhat.com">Emanuel Muckenhuber</a>
 * @version $Revision$
 */
public class JUnitTestReport
{

   /** The test suite name */
   private String testSuiteName;

   public static final String DEFAULT_TEST_SUITE_NAME = "JBossUnitTestSuite";

   public static final String TEST_FILE_NAME_START = "TEST-";

   /** The document */
   private Document document;
   
   /** The root element */
   protected Element rootElement;
   
   /** The started Tests */
   private Map<TestId, TestInfo> startedTests = new HashMap<TestId, TestInfo>();
   
   /** The system error */
   private StringBuffer systemErr = new StringBuffer();
   
   /** The system output */
   private StringBuffer systemOut = new StringBuffer();
   
   /** The test count */
   private int tests;
  
   /** The error count */
   private int errors;
   
   /** The failed count */
   private int failed;

   /** The timeStamp */
   private String timeStamp;
   
   /** The tested classes Stack */
   private LinkedList<String> testSuiteNames = new LinkedList<String>();

   /** .*/
   private NumberFormat FORMAT = NumberFormat.getInstance();

   public JUnitTestReport()
   {
   }

   public JUnitTestReport(String testSuiteName)
   {
      this.testSuiteName = testSuiteName;
   }
   
   
   public void startJUnitTestSuite(StartRunnerEvent event)
   {
      /** Create root element */
      this.document = getDocumentBuilder().newDocument();
      this.rootElement = this.document.createElement(XMLConstants.TESTSUITE);
      this.rootElement.setAttribute(XMLConstants.TIMESTAMP, getTimestamp());
      this.rootElement.setAttribute(XMLConstants.HOSTNAME, getHostName());
      this.rootElement.setAttribute(XMLConstants.ATTR_NAME, this.testSuiteName);
      /** Create properties element */
      Element properties = document.createElement(XMLConstants.PROPERTIES);
      this.rootElement.appendChild(properties);
      /**
       * TODO - set properties 
       */
   }
   
   // add tested class to stack (starttestsuiteevent)
   public void addTestedSuiteName(String testedClassName)
   {
      this.testSuiteNames.add(testedClassName);
   }
   
   // remove last from stack (endtestsuiteevent)
   public void removeLastTestedSuite()
   {
      this.testSuiteNames.removeLast();
   }
   
   public void startTest(StartTestCaseEvent event)
   { 
      this.startedTests.put(event.getTestId(), event.getTestInfo());
      this.tests++;
   }
   
   public void runnerFailed(RunnerFailureEvent event)
   {
      this.tests++;
      this.errors++;
      StringBuffer nb = new StringBuffer();
      /** The current tested Class */
      Element e = document.createElement(XMLConstants.ERROR);
      e.setAttribute(XMLConstants.ATTR_TYPE, event.getFailure().getType().name());
      e.setAttribute(XMLConstants.ATTR_MESSAGE, event.getFailure().getMessage());
      e.appendChild(document.createCDATASection(stackToString(event.getFailure().getStackTrace())));
      rootElement.appendChild(e);
   }
   
   public void endTest(EndTestCaseEvent event)
   {
      TestId testId = event.getTestId();
      TestResult result = event.getTestResult();
      Element test = document.createElement(XMLConstants.TESTCASE);
      TestInfo info = this.startedTests.get(testId);
      
      /** The testcase name */
      StringBuffer nb = new StringBuffer();
      /** The current tested Class */
      for(String testSuiteName : this.testSuiteNames)
      {
         nb.append(testSuiteName);
         nb.append(".");
      }
      /** Adding testcase name */
      nb.append(info.getName());
      
      String testName = nb.toString();
      
      /** The testcase parameters */
      StringBuffer parameters = new StringBuffer();
      for(Iterator<String> i = result.getParametrization().keySet().iterator(); i.hasNext(); )
      {
         String name = i.next();
         String value = result.getParametrization().get(name);
         parameters.append( name + "=" + value );
         if(i.hasNext())
         {
            parameters.append("&");
         }
      }
      // append parameters
      if ( parameters.length() > 0)
         testName = testName + "?" + parameters.toString();
      
      /** Test case attributes */
      test.setAttribute(XMLConstants.ATTR_NAME, testName);
      test.setAttribute(XMLConstants.ATTR_TIME, FORMAT.format(result.getDurationMillis() / 1000.0));
      /** Check if test was started */
      if ( info != null)
      {
         // remove test from started List
         this.startedTests.remove(testId);
      }
      else 
      {
         // If test is not started - add error message
         Element errorNotStarted = document.createElement(XMLConstants.ERROR);
         errorNotStarted.setAttribute(XMLConstants.ATTR_MESSAGE, "TestCase: " + testName + " was not started properly. (No StartTestEvent)");
         test.appendChild(errorNotStarted);
      }
     
      /** If testcase failed */
      if (result instanceof TestFailure)
      {
         TestFailure testFailure = (TestFailure) result;
         Failure failure = testFailure.getFailure();
         
         String failureType;
         // Check failure type (failed, error)
         if(FailureType.ASSERTION == failure.getType())
         {
            this.failed++;
            failureType = XMLConstants.FAILURE;
         }
         else 
         {
            this.errors++;
            failureType = XMLConstants.ERROR;
         }

         test.appendChild(addError(failureType, failure.getStackTrace()));
      }

      this.rootElement.appendChild(test);
   }
   
   public void endTestSuite(EndRunnerEvent event)
   {
      if( this.startedTests.size() > 0 )
      {
         for(TestId id : this.startedTests.keySet())
         {
            Element errorNotEnded = document.createElement(XMLConstants.ERROR);
            errorNotEnded.setAttribute(XMLConstants.ATTR_MESSAGE, "Testcase: " + id + " was not ended properly. (No EndTestEvent)");
            this.rootElement.appendChild(errorNotEnded);
         }
      }
      
      // Testsuite counts (tests, errors, failures)
      this.rootElement.setAttribute(XMLConstants.ATTR_TESTS, Integer.toString(this.tests));
      this.rootElement.setAttribute(XMLConstants.ATTR_ERRORS, Integer.toString(this.errors));
      this.rootElement.setAttribute(XMLConstants.ATTR_FAILURES, Integer.toString(this.failed));
      
      // Testsuite system-err
      Element systemError = document.createElement(XMLConstants.SYSTEM_ERR);
      systemError.appendChild(document.createCDATASection(systemErr.toString()));
      // Testsuite system-out
      Element systemOutElement = document.createElement(XMLConstants.SYSTEM_OUT);
      systemOutElement.appendChild(document.createCDATASection(systemOut.toString()));
      
      this.rootElement.appendChild(systemError);
      this.rootElement.appendChild(systemOutElement);
   }
   
   public void appendSystemOutput(String output)
   {
      this.systemOut.append(output);
   }
   
   public void appendSystemError(String error)
   {
      this.systemErr.append(error);
   }
   
   public void exportXML(String toDir)
   {
      try
      {
         toDir = checkDir(toDir);
         StringBuffer filePath = new StringBuffer();

         // If no test suite name is specified use default and add timestamp
         filePath.append(toDir)
            .append(File.separator)
            .append(TEST_FILE_NAME_START)
            .append(getTestSuiteName())
            .append(getTestSuiteName().equals(DEFAULT_TEST_SUITE_NAME) ? "-" + getTimestamp() : "" )
            .append(".xml");


         File file = new File(filePath.toString());
         
         Source source = new DOMSource(this.rootElement);
         Transformer xtrans = TransformerFactory.newInstance().newTransformer();
         xtrans.setOutputProperty(OutputKeys.INDENT, "yes");
         xtrans.setOutputProperty(OutputKeys.METHOD, "xml");
         xtrans.setOutputProperty(OutputKeys.STANDALONE, "yes");
         Result result = new StreamResult(new FileOutputStream(file));
         xtrans.transform(source, result);
      }
      catch(FileNotFoundException e)
      {
         e.printStackTrace();
      }
      catch(TransformerException e)
      {
         e.printStackTrace();
      }
      catch(IllegalArgumentException e)
      {
         e.printStackTrace();
      }
   }
   
   private String checkDir(String toDir)
   {
      File dir = new File(toDir);
      if (! dir.exists())
      {
         dir.mkdir();
      }
      return dir.toString();
   }
   
   private Element addError(String type, Throwable t)
   {
      Element error = document.createElement(type);
      // Testcase error message
      error.setAttribute(XMLConstants.ATTR_MESSAGE, t.getMessage());
      // Testcase error type
      error.setAttribute(XMLConstants.ATTR_TYPE, t.getClass().getName());
      // Testcase stacktrace
      error.appendChild(document.createCDATASection(stackToString(t)));
      return error;
   }
  
   private static String stackToString(Throwable t)
   {
      final Writer writer = new StringWriter();
      final PrintWriter printWriter = new PrintWriter(writer);
      t.printStackTrace(printWriter);
      printWriter.flush();
      printWriter.close();
      return writer.toString();
   }
   
   private String getTimestamp()
   {
      if (timeStamp == null)
      {
         SimpleDateFormat date = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss");
         timeStamp = date.format(new Date());

      }
      return timeStamp; 
   }
   
   private static String getHostName()
   {
      try
      {
         return InetAddress.getLocalHost().getHostName();
      }
      catch(UnknownHostException e)
      {
         return "localhost";
      }
   }
   
   private static DocumentBuilder getDocumentBuilder()
   {
      try
      {
         return DocumentBuilderFactory.newInstance().newDocumentBuilder();
      }
      catch(ParserConfigurationException e)
      {
         e.printStackTrace();
         return null;
      }
   }

   public String getTestSuiteName()
   {
      if (testSuiteName == null)
      {
         return DEFAULT_TEST_SUITE_NAME;
      }
      return testSuiteName;
   }

   public void setTestSuiteName(String testSuiteName)
   {
      this.testSuiteName = testSuiteName;
   }
}