/******************************************************************************
 * JBoss, a division of Red Hat                                               *
 * Copyright 2006, Red Hat Middleware, LLC, and individual                    *
 * contributors as indicated by the @authors tag. See the                     *
 * copyright.txt in the distribution for a full listing of                    *
 * individual contributors.                                                   *
 *                                                                            *
 * This is free software; you can redistribute it and/or modify it            *
 * under the terms of the GNU Lesser General Public License as                *
 * published by the Free Software Foundation; either version 2.1 of           *
 * the License, or (at your option) any later version.                        *
 *                                                                            *
 * This software is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU           *
 * Lesser General Public License for more details.                            *
 *                                                                            *
 * You should have received a copy of the GNU Lesser General Public           *
 * License along with this software; if not, write to the Free                *
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA         *
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.                   *
 ******************************************************************************/
package org.jboss.unit.report.impl.junit;

import org.jboss.unit.Failure;
import org.jboss.unit.FailureType;
import org.jboss.unit.TestId;
import org.jboss.unit.runner.TestResult;
import org.jboss.unit.runner.results.TestFailure;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Result;
import javax.xml.transform.Source;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.io.Writer;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.text.NumberFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * @author <a href="mailto:emuckenh@redhat.com">Emanuel Muckenhuber</a>
 * @version $Revision$
 */
public class JUnitTestsuiteReport extends AbstractJUnitReport
{

   /** */
   public static final String TEST_FILE_NAME_PREFIX = "TEST-";

   /** The formatter for timestamps.*/
   private static final SimpleDateFormat TIMESTAMP_FORMAT = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss");

   /** The formatter for durations.*/
   private static final NumberFormat DURATION_FORMAT = NumberFormat.getInstance();

   /** The document */
   private Document testsuitesDocument;

   /** . */
   private File htmlReportDir;

   /** . */
   private File xmlReportDir;

   /** . */
   private Map<TestSuiteKey, TestSuiteReport> reportMap = new HashMap<TestSuiteKey, TestSuiteReport>();

   public JUnitTestsuiteReport(File xmlReportDir, File htmlReportDir)
   {
      this.xmlReportDir = xmlReportDir;
      this.htmlReportDir = htmlReportDir;
   }

   protected void startRunner()
   {
      testsuitesDocument = createDocument();
   }

   protected void endRunner()
   {
      Element testsuitesElement = testsuitesDocument.createElement(XMLConstants.TESTSUITES);
      testsuitesDocument.appendChild(testsuitesElement);

      //
      int counter = 0;

      //
      for (Map.Entry<TestSuiteKey, TestSuiteReport> entry : reportMap.entrySet())
      {
         TestSuiteReport report = entry.getValue();

         // We filter and keep only meaningful results
         if (report.elements.size() > 0)
         {
            report.elements.add(testsuitesDocument.createElement(XMLConstants.PROPERTIES));
            Element systemError = testsuitesDocument.createElement(XMLConstants.SYSTEM_ERR);
            systemError.appendChild(testsuitesDocument.createCDATASection(report.err.toString()));
            report.elements.add(systemError);
            Element systemOutElement = testsuitesDocument.createElement(XMLConstants.SYSTEM_OUT);
            systemOutElement.appendChild(testsuitesDocument.createCDATASection(report.out.toString()));
            report.elements.add(systemOutElement);

            //
            Element testSuiteElt = testsuitesDocument.createElement(XMLConstants.TESTSUITE);

            //
            testSuiteElt.setAttribute(XMLConstants.HOSTNAME, getHostName());
            testSuiteElt.setAttribute(XMLConstants.ATTR_PACKAGE, entry.getKey().packageName);
            testSuiteElt.setAttribute(XMLConstants.ATTR_NAME, appendURI(new StringBuffer(), entry.getKey().className, entry.getKey().properties).toString());
            testSuiteElt.setAttribute(XMLConstants.ATTR_TESTS, Integer.toString(report.tests));
            testSuiteElt.setAttribute(XMLConstants.ATTR_ERRORS, Integer.toString(report.errors));
            testSuiteElt.setAttribute(XMLConstants.ATTR_FAILURES, Integer.toString(report.failures));
            testSuiteElt.setAttribute(XMLConstants.ATTR_TIME, DURATION_FORMAT.format(report.durationMillis / 1000.0));

            //
            for (Element elt : report.elements)
            {
               testSuiteElt.appendChild(elt);
            }

            //
            if (xmlReportDir != null)
            {
               if (xmlReportDir.exists())
               {
                  Document tmp = createDocument();
                  Element copy = (Element)tmp.importNode(testSuiteElt, true);
                  tmp.appendChild(copy);
                  String a = copy.getAttribute("package");
                  String b = copy.getAttribute("name");
                  String c = a.length() == 0 ? b : a + "." + b;
                  copy.setAttribute("name", c);
                  copy.removeAttribute("package");
                  File f = new File(xmlReportDir, TEST_FILE_NAME_PREFIX + (counter++) + ".xml");
                  try
                  {
                     if (!f.exists())
                     {
                        f.createNewFile();
                     }
                     serializeNode(tmp, f);
                  }
                  catch (IOException e)
                  {
                     System.out.println("Could not create file " + f.getAbsolutePath());
                     e.printStackTrace();
                  }
               }
               else
               {
                  System.out.println("Cannot create report because parent dir " + xmlReportDir.getAbsolutePath() + " does not exist");
               }
            }

            //
            testsuitesElement.appendChild(testSuiteElt);
         }
      }

      //
      if (htmlReportDir != null)
      {
         if (htmlReportDir.exists())
         {
            try
            {
               InputStream xslIn = getClass().getResourceAsStream("/org/jboss/unit/report/impl/junit/junit-frames.xsl");
               Source xmlSource = new DOMSource(testsuitesElement);
               Source xslSource = new StreamSource(xslIn);
               TransformerFactory transFact = TransformerFactory.newInstance();
               Transformer trans = transFact.newTransformer(xslSource);
               OutputStream os = new ByteArrayOutputStream();
               Result result = new StreamResult(os);
               trans.setParameter("output.dir", htmlReportDir.getAbsolutePath());
               trans.transform(xmlSource, result);
            }
            catch (TransformerException e)
            {
               e.printStackTrace();
            }

            //
            File tmp = new File(htmlReportDir, "TESTS-TestSuites.xml");
            serializeNode(testsuitesElement, tmp);
         }
         else
         {
            System.out.println("Cannot create report because parent dir " + htmlReportDir.getAbsolutePath() + " does not exist");
         }
      }
   }

   protected void runnerFailure(TestSuite testSuite, Failure failure)
   {
      Element errorElt = testsuitesDocument.createElement(XMLConstants.ERROR);
      errorElt.setAttribute(XMLConstants.ATTR_TYPE, failure.getType().name());
      errorElt.setAttribute(XMLConstants.ATTR_MESSAGE, failure.getMessage());
      errorElt.appendChild(testsuitesDocument.createCDATASection(stackToString(failure.getStackTrace())));

      //
      get(testSuite).elements.add(errorElt);
   }

   protected void startTestSuite(TestSuite testSuite)
   {
   }

   private TestSuiteReport get(TestSuite testSuite)
   {
      TestSuiteKey key = new TestSuiteKey(testSuite);
      TestSuiteReport report = reportMap.get(key);
      if (report == null)
      {
         report = new TestSuiteReport();
         reportMap.put(key, report);
      }
      return report;
   }

   protected void endTestSuite(TestSuite testSuite)
   {
      TestSuiteReport report = get(testSuite);

      //
      if( testSuite.startedTests.size() > 0 )
      {
         for(TestId id : testSuite.startedTests.keySet())
         {
            Element errorNotEnded = testsuitesDocument.createElement(XMLConstants.ERROR);
            errorNotEnded.setAttribute(XMLConstants.ATTR_MESSAGE, "Testcase: " + id + " was not ended properly. (No EndTestEvent)");
            report.elements.add(errorNotEnded);
         }
      }

      //
      try
      {
         testSuite.view.writeTo(report.out, report.err);
      }
      catch (IOException e)
      {
         e.printStackTrace();
      }

      //
      report.tests += testSuite.tests;
      report.failures += testSuite.failures;
      report.errors += testSuite.errors;
      report.durationMillis += testSuite.durationMillis;
   }

   private void serializeNode(Node node, File dest)
   {
      try
      {
         Source source = new DOMSource(node);
         Transformer xtrans = TransformerFactory.newInstance().newTransformer();
         xtrans.setOutputProperty(OutputKeys.INDENT, "yes");
         xtrans.setOutputProperty(OutputKeys.METHOD, "xml");
         xtrans.setOutputProperty(OutputKeys.STANDALONE, "yes");
         Result result = new StreamResult(dest);
         xtrans.transform(source, result);
      }
      catch (TransformerException e)
      {
         e.printStackTrace();
      }
   }

   public static StringBuffer appendURI(StringBuffer buffer, String name, Map<String, String> parameters)
   {
      buffer.append(name);
      Iterator<String> i = parameters.keySet().iterator();
      if (i.hasNext())
      {
         buffer.append("?");
         while (i.hasNext())
         {
            String parameterName = i.next();
            String value = parameters.get(parameterName);
            buffer.append(parameterName).append("=").append(value);
            if(i.hasNext())
            {
               buffer.append("&");
            }
         }
      }
      return buffer;
   }

   protected void endTestCase(TestSuite testSuite, String testName, TestResult result)
   {
      Element test = testsuitesDocument.createElement(XMLConstants.TESTCASE);
      test.setAttribute(XMLConstants.ATTR_CLASSNAME, testSuite.getFQN().toString());
      test.setAttribute(XMLConstants.ATTR_NAME, appendURI(new StringBuffer(), testName, result.getParametrization()).toString());
      test.setAttribute(XMLConstants.ATTR_TIME, DURATION_FORMAT.format(result.getDurationMillis() / 1000.0));

      /** If testcase failed */
      if (result instanceof TestFailure)
      {
         TestFailure testFailure = (TestFailure) result;
         Failure failure = testFailure.getFailure();

         String failureType;
         // Check failure type (failed, error)
         if(FailureType.ASSERTION == failure.getType())
         {
            testSuite.failures++;
            failureType = XMLConstants.FAILURE;
         }
         else
         {
            testSuite.errors++;
            failureType = XMLConstants.ERROR;
         }

         test.appendChild(addGeneralError(failureType, failure.getStackTrace()));
      }

      //
      get(testSuite).elements.add(test);


      //
//      if (info != null)
//      {
//
//
//      }
//      else
//      {
//         // If test is not started - add error message
//         Element errorNotStarted = document.createElement(XMLConstants.ERROR);
//         errorNotStarted.setAttribute(XMLConstants.ATTR_MESSAGE, "TestCase: " + testId + " was not started properly. (No StartTestEvent)");
//         test.appendChild(errorNotStarted);
//      }
   }


   private Element addGeneralError(String type, Throwable t)
   {
      Element error = testsuitesDocument.createElement(type);

      // Testcase error message
      error.setAttribute(XMLConstants.ATTR_MESSAGE, t.getMessage());

      // Testcase error type
      error.setAttribute(XMLConstants.ATTR_TYPE, t.getClass().getName());

      // Testcase stacktrace
      error.appendChild(testsuitesDocument.createCDATASection(stackToString(t)));

      //
      return error;
   }

   private static String stackToString(Throwable t)
   {
      Writer writer = new StringWriter();
      PrintWriter printWriter = new PrintWriter(writer);
      t.printStackTrace(printWriter);
      printWriter.close();
      return writer.toString();
   }

   /**
    * Generates a new timestamp string.
    *
    * @return a newly create timestamp string
    */
   private static String createTimestamp()
   {
      return TIMESTAMP_FORMAT.format(new Date());
   }

   private static String getHostName()
   {
      try
      {
         return InetAddress.getLocalHost().getHostName();
      }
      catch(UnknownHostException e)
      {
         return "localhost";
      }
   }

   private static Document createDocument()
   {
      try
      {
         return DocumentBuilderFactory.newInstance().newDocumentBuilder().newDocument();
      }
      catch(ParserConfigurationException e)
      {
         throw new Error(e);
      }
   }

   private static class TestSuiteKey
   {

      /** . */
      private final String packageName;

      /** . */
      private final String className;

      /** . */
      private final Map<String, String> properties;

      private TestSuiteKey(TestSuite testSuite)
      {
         StringBuffer fqn = testSuite.getFQN();
         int pos = fqn.lastIndexOf(".");
         packageName = (pos == -1) ? "" : fqn.substring(0, pos);
         className = (pos == -1) ? fqn.toString() : fqn.substring(pos + 1);
         properties = testSuite.properties;
      }

      public boolean equals(Object o)
      {
         if (o == this)
         {
            return true;
         }
         if (o instanceof TestSuiteKey)
         {
            TestSuiteKey that = (TestSuiteKey)o;
            return that.packageName.equals(packageName) && that.className.equals(className) && that.properties.equals(properties);
         }
         return false;
      }

      public int hashCode()
      {
         return packageName.hashCode() * 41 + className.hashCode();
      }
   }

   private static class TestSuiteReport
   {

      /** . */
      private final List<Element> elements = new ArrayList<Element>();

      /** . */
      private final StringWriter out = new StringWriter();

      /** . */
      private final StringWriter err = new StringWriter();

      /** . */
      private int tests;

      /** . */
      private int errors;

      /** . */
      private int failures;

      /** . */
      private long durationMillis;

   }
}