/******************************************************************************
 * JBoss, a division of Red Hat                                               *
 * Copyright 2006, Red Hat Middleware, LLC, and individual                    *
 * contributors as indicated by the @authors tag. See the                     *
 * copyright.txt in the distribution for a full listing of                    *
 * individual contributors.                                                   *
 *                                                                            *
 * This is free software; you can redistribute it and/or modify it            *
 * under the terms of the GNU Lesser General Public License as                *
 * published by the Free Software Foundation; either version 2.1 of           *
 * the License, or (at your option) any later version.                        *
 *                                                                            *
 * This software is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU           *
 * Lesser General Public License for more details.                            *
 *                                                                            *
 * You should have received a copy of the GNU Lesser General Public           *
 * License along with this software; if not, write to the Free                *
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA         *
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.                   *
 ******************************************************************************/
package org.jboss.unit.report.impl.writer;

import org.jboss.unit.Failure;
import org.jboss.unit.runner.TestResult;
import org.jboss.unit.runner.TestRunnerEvent;
import org.jboss.unit.runner.TestRunnerEventListener;
import org.jboss.unit.runner.event.EndRunnerEvent;
import org.jboss.unit.runner.event.EndTestCaseEvent;
import org.jboss.unit.runner.event.RunnerFailureEvent;
import org.jboss.unit.runner.event.StartRunnerEvent;
import org.jboss.unit.runner.event.StartTestCaseEvent;
import org.jboss.unit.runner.event.StartTestSuiteEvent;
import org.jboss.unit.runner.event.EndTestSuiteEvent;
import org.jboss.unit.runner.results.TestFailure;
import org.jboss.unit.runner.results.TestSuccess;

import java.io.Writer;
import java.io.PrintWriter;
import java.io.OutputStream;
import java.text.NumberFormat;
import java.util.LinkedList;
import java.util.Iterator;
import java.util.Map;

/**
 * A listener which output messages on a specified stream.
 *
 * @author <a href="mailto:julien@jboss.org">Julien Viet</a>
 * @version $Revision: 1.1 $
 */
public class PrintListener implements TestRunnerEventListener
{

   /** . */
   private final static String NEW_LINE = System.getProperty("line.separator");

   /** . */
   private NumberFormat FORMAT = NumberFormat.getInstance();

   /** . */
   private final PrintWriter out;

   /** The test suite stats. */
   private LinkedList<SuiteStat> stack = new LinkedList<SuiteStat>();

   /** The current test case name. */
   private String currentTestCaseName;

   /**
    * Builds with the <code>System.out</code> print stream.
    */
   public PrintListener()
   {
      this(System.out);
   }

   /**
    * Use a writer, if the writer is an instance of <code>PrintWriter</code> then this object
    * will be reused as is.
    *
    * @param writer the writer 
    */
   public PrintListener(Writer writer) throws IllegalArgumentException
   {
      if (writer == null)
      {
         throw new IllegalArgumentException();
      }
      if (writer instanceof PrintWriter)
      {
         this.out = (PrintWriter)writer;
      }
      else
      {
         this.out = new PrintWriter(writer, true);
      }
   }

   /**
    * Uses a provided output stream.
    *
    * @param stream the stream
    */
   public PrintListener(OutputStream stream) throws IllegalArgumentException
   {
      if (stream == null)
      {
         throw new IllegalArgumentException();
      }
      this.out = new PrintWriter(stream, true);
   }

   public void onEvent(TestRunnerEvent event)
   {
      if (event instanceof StartRunnerEvent)
      {
         out.println("--- Start ---");
      }
      else if (event instanceof EndRunnerEvent)
      {
         out.println("--- End ---");
      }
      else if (event instanceof RunnerFailureEvent)
      {
         RunnerFailureEvent failureEvent = (RunnerFailureEvent)event;
         Failure failure = failureEvent.getFailure(); 
         out.println("Runner failure: " + failure.getType() + " " + failure.getMessage());
         failure.getCause().printStackTrace(out);
      }
      else if (event instanceof StartTestSuiteEvent)
      {
         StartTestSuiteEvent stse = (StartTestSuiteEvent)event;
         stack.add(new SuiteStat(stse.getTestInfo().getName()));
      }
      else if (event instanceof StartTestCaseEvent)
      {
         StartTestCaseEvent ste = (StartTestCaseEvent)event;
         currentTestCaseName = ste.getTestInfo().getName();
      }
      else if (event instanceof EndTestSuiteEvent)
      {
         StringBuffer tmp = new StringBuffer();
         for (Iterator<SuiteStat> i = stack.iterator();i.hasNext();)
         {
            SuiteStat stat = i.next();
            tmp.append(stat.name);
            if (i.hasNext())
            {
               tmp.append('.');
            }
         }

         //
         SuiteStat stat = stack.removeLast();

         StringBuffer sb = new StringBuffer("Testsuite: ");

         sb.append(tmp);
         sb.append(NEW_LINE);

         if (stat.count > 0)
         {
            sb.append("Tests run: ");
            sb.append(stat.count);
            sb.append(", Failures: ");
            sb.append(stat.failures);
            sb.append(", Errors: ");
            sb.append(stat.errors);
            sb.append(", Time elapsed: ");
            sb.append(FORMAT.format((System.currentTimeMillis() - stat.timestamp) / 1000.0));
            sb.append(" sec");
         }
         else
         {
            sb.append("Testcase skipped...");
         }
         sb.append(NEW_LINE);
         sb.append(NEW_LINE);
         out.print(sb);

      }
      else if (event instanceof EndTestCaseEvent)
      {
         stack.getLast().count++;

         //
         EndTestCaseEvent endTestEvent = (EndTestCaseEvent)event;
         TestResult testResult = endTestEvent.getTestResult();

         StringBuffer name = new StringBuffer(currentTestCaseName);
         Map<String, String> parametrization = testResult.getParametrization();
         if (!parametrization.isEmpty())
         {
            name.append('?');
            for (Iterator<Map.Entry<String, String>> i = parametrization.entrySet().iterator();i.hasNext();)
            {
               Map.Entry entry = i.next();
               name.append(entry.getKey()).append('=').append(entry.getValue());
               if (i.hasNext())
               {
                  name.append('&');
               }
            }
         }

         //
         out.println("Testcase:  " + name + " took " + FORMAT.format(testResult.getDurationMillis() / 1000.0) + " sec");

         if (testResult instanceof TestSuccess)
         {
         }
         else if (testResult instanceof TestFailure)
         {
            TestFailure testFailure = (TestFailure)testResult;
            Failure failure = testFailure.getFailure();

            //
            switch(failure.getType())
            {
               case ASSERTION:
                  out.println("\tFAILED");
                  stack.getLast().failures++;
                  break;
               case ERROR:
                  out.println("\tCaused an ERROR");
                  stack.getLast().errors++;
                  break;
            }

            //
            out.print(failure.getMessage());
            if (failure.getCause() != null)
            {
               failure.getCause().printStackTrace(out);
            }
            out.println();
         }
      }
   }

   private static class SuiteStat
   {
      /** . */
      private final String name;

      /** . */
      private int failures = 0;

      /** . */
      private int errors = 0;

      /** . */
      private int count = 0;

      /** . */
      private long timestamp = System.currentTimeMillis();

      private SuiteStat(String name)
      {
         this.name = name;
      }
   }
}
