/******************************************************************************
 * JBoss, a division of Red Hat                                               *
 * Copyright 2006, Red Hat Middleware, LLC, and individual                    *
 * contributors as indicated by the @authors tag. See the                     *
 * copyright.txt in the distribution for a full listing of                    *
 * individual contributors.                                                   *
 *                                                                            *
 * This is free software; you can redistribute it and/or modify it            *
 * under the terms of the GNU Lesser General Public License as                *
 * published by the Free Software Foundation; either version 2.1 of           *
 * the License, or (at your option) any later version.                        *
 *                                                                            *
 * This software is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU           *
 * Lesser General Public License for more details.                            *
 *                                                                            *
 * You should have received a copy of the GNU Lesser General Public           *
 * License along with this software; if not, write to the Free                *
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA         *
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.                   *
 ******************************************************************************/
package org.jboss.unit.runner.impl;

import org.jboss.unit.runner.TestRunnerEventListener;
import org.jboss.unit.runner.TestRunnerEvent;

import java.util.List;
import java.util.ArrayList;

/**
 * Maintains a set of listeners that will receive events when the <code>fireEvent(TestRunnerEvent event)</code>  method
 * is invoked. When a listener method is invoked, any exception is caught and logged, this include checked and unchecked
 * exception. Any other kind of unchecked throwable will not be handled.
 *
 * @author <a href="mailto:julien@jboss.org">Julien Viet</a>
 * @version $Revision: 1.1 $
 */
public class TestRunnerEventBroadcaster implements TestRunnerEventListener
{

   /** . */
   private List<TestRunnerEventListener> listeners = new ArrayList<TestRunnerEventListener>();

   /**
    * Add a listener to the set of listeners.
    *
    * @param listener the listener to add
    * @throws IllegalArgumentException if the listener is null
    */
   public void addListener(TestRunnerEventListener listener) throws IllegalArgumentException
   {
      if (listener == null)
      {
         throw new IllegalArgumentException("No null listener accepted");
      }
      if (!listeners.contains(listener))
      {
         listeners.add(listener);
      }
   }

   /**
    * Remove a listener from the set of listeners.
    *
    * @param listener the listener to remove
    * @throws IllegalArgumentException if the listener is null
    */
   public void removeListener(TestRunnerEventListener listener)
   {
      if (listener == null)
      {
         throw new IllegalArgumentException("No null listener accepted");
      }
      if (listeners.contains(listener))
      {
         listeners.remove(listener);
      }
   }

   public void onEvent(TestRunnerEvent event)
   {
      if (event == null)
      {
         throw new IllegalArgumentException();
      }
      for (TestRunnerEventListener listener : listeners)
      {
         try
         {
            listener.onEvent(event);
         }
         catch (Exception e)
         {
//            e.printStackTrace();
         }
      }
   }
}
