/******************************************************************************
 * JBoss, a division of Red Hat                                               *
 * Copyright 2006, Red Hat Middleware, LLC, and individual                    *
 * contributors as indicated by the @authors tag. See the                     *
 * copyright.txt in the distribution for a full listing of                    *
 * individual contributors.                                                   *
 *                                                                            *
 * This is free software; you can redistribute it and/or modify it            *
 * under the terms of the GNU Lesser General Public License as                *
 * published by the Free Software Foundation; either version 2.1 of           *
 * the License, or (at your option) any later version.                        *
 *                                                                            *
 * This software is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU           *
 * Lesser General Public License for more details.                            *
 *                                                                            *
 * You should have received a copy of the GNU Lesser General Public           *
 * License along with this software; if not, write to the Free                *
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA         *
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.                   *
 ******************************************************************************/
package org.jboss.unit.runner.impl.driver;

import org.jboss.unit.driver.TestDriver;
import org.jboss.unit.driver.DriverResponse;
import org.jboss.unit.driver.impl.pojo.SimpleDriverContext;
import org.jboss.unit.driver.response.EndTestResponse;
import org.jboss.unit.driver.response.FailureResponse;
import org.jboss.unit.driver.command.StartTestCommand;
import org.jboss.unit.info.TestSuiteInfo;
import org.jboss.unit.info.TestCaseInfo;
import org.jboss.unit.info.TestInfo;
import org.jboss.unit.TestId;
import org.jboss.unit.Failure;
import org.jboss.unit.FailureType;
import org.jboss.unit.runner.results.TestSuccess;
import org.jboss.unit.runner.results.TestFailure;
import org.jboss.unit.runner.event.StartTestCaseEvent;
import org.jboss.unit.runner.event.EndTestCaseEvent;
import org.jboss.unit.runner.event.RunnerFailureEvent;
import org.jboss.unit.runner.event.StartTestSuiteEvent;
import org.jboss.unit.runner.event.EndTestSuiteEvent;
import org.jboss.unit.runner.AbstractTestRunner;
import org.jboss.unit.runner.TestResult;
import org.jboss.unit.runner.TestRunnerContext;

import java.util.Map;

/**
 * An implementation that uses a test driver and transform test driver
 * interactions into runner interactions.
 *
 * During the invocation of a test case through the <code>invoke</code> method of the test driver, it
 * expects a non null response of type <code>EndTestResponse</code> or <code>FailureResponse</code>. For a normal
 * test case invocation the runner will fire TODO
 * If the test case invocation fails because an unexpected runtime exception is thrown or because the returned
 * value is not of the expected type then the runner will broadcast an <code>EndTestEvent</code> with a <code>TestResult<code>
 * object of type <code>TestFailure</code>.
 *
 * @author <a href="mailto:julien@jboss.org">Julien Viet</a>
 * @version $Revision: 1.1 $
 */
public class TestDriverRunner extends AbstractTestRunner
{

   /** . */
   private final TestDriver driver;

   /** . */
   private TestId testId;

   public TestDriverRunner(TestDriver driver, TestId testId)
   {
      this.driver = driver;
      this.testId = testId;
   }

   public TestDriverRunner(TestDriver driver)
   {
      this(driver, null);
   }

   public TestId getTestId()
   {
      return testId;
   }

   public void setTestId(TestId testId)
   {
      this.testId = testId;
   }

   protected void internalRun(TestRunnerContext context)
   {
      if (testId == null)
      {
         context.getEventListener().onEvent(new RunnerFailureEvent(Failure.createErrorFailure("No test id specified")));
      }
      else
      {
         // Init the test driver
         try
         {
            driver.initDriver(new SimpleDriverContext(context.getProperties()));
         }
         catch (Exception e)
         {
            context.getEventListener().onEvent(new RunnerFailureEvent(Failure.createFailure("Was not able to start the test driver properly", e)));
            return;
         }

         //
         try
         {
            TestInfo info = driver.getInfo();

            //
            internalRun(context, info, testId, new TestId());
         }
         finally
         {
            // Perform cleanup of life cycle
            try
            {
               driver.destroyDriver();
            }
            catch (RuntimeException ignore)
            {
            }
         }
      }
   }

   private void internalRun(final TestRunnerContext context, final TestInfo currentInfo, final TestId targetId, final TestId currentId)
   {
      if (currentId.equals(targetId))
      {
         if (context.getFilter().include(currentId, currentInfo))
         {
            if (currentInfo instanceof TestSuiteInfo)
            {
               TestSuiteInfo testSuiteInfo = (TestSuiteInfo)currentInfo;

               //
               context.getEventListener().onEvent(new StartTestSuiteEvent(currentId, testSuiteInfo, context.getProperties()));

               //
               for (String name : testSuiteInfo.getNames())
               {
                  TestInfo childInfo = testSuiteInfo.getTest(name);

                  //
                  TestId childId = new TestId(currentId, name);

                  //
                  internalRun(context, childInfo, childId, childId);
               }

               //
               context.getEventListener().onEvent(new EndTestSuiteEvent(testId));
            }
            else
            {
               TestCaseInfo testCaseInfo = (TestCaseInfo)currentInfo;

               //
               for (Map<String,String> parametrization : context.getParametrizations().enumerate(testCaseInfo.getParameters().keySet()))
               {
                  context.getEventListener().onEvent(new StartTestCaseEvent(currentId, testCaseInfo));

                  //
                  StartTestCommand command = new StartTestCommand(parametrization);

                  //
                  TestResult result = null;

                  //
                  long duration = -System.currentTimeMillis();

                  //
                  DriverResponse response = null;
                  try
                  {
                     try
                     {
                        response = driver.invoke(currentId, command);
                     }
                     finally
                     {
                        duration += System.currentTimeMillis();
                     }
                  }
                  catch (Exception e)
                  {
                     Failure failure = new Failure("The test driver threw an unexpected exception ", e, FailureType.ERROR);
                     result = new TestFailure(failure, parametrization, duration);
                  }

                  //
                  if (result == null)
                  {
                     result = unwrap(response, parametrization, duration);
                  }

                  //
                  context.getEventListener().onEvent(new EndTestCaseEvent(currentId, result));
               }
            }
         }
      }
      else
      {
         String name = targetId.getName(currentId.getLength());

         //
         if (currentInfo instanceof TestSuiteInfo)
         {
            //
            TestSuiteInfo currentSuiteInfo = (TestSuiteInfo)currentInfo;

            //
            TestInfo nextInfo = currentSuiteInfo.getTest(name);

            //
            if (nextInfo != null)
            {
               TestId nextId = new TestId(currentId, name);

               //
               context.getEventListener().onEvent(new StartTestSuiteEvent(currentId, currentSuiteInfo, context.getProperties()));

               //
               internalRun(context, nextInfo, targetId, nextId);

               //
               context.getEventListener().onEvent(new EndTestSuiteEvent(currentId));
            }
            else
            {

            }
         }
         else
         {

         }
      }
   }

   private TestResult unwrap(DriverResponse response, Map<String,String> parametrization, long duration)
   {
      if (response instanceof EndTestResponse)
      {
         return new TestSuccess(parametrization, duration);
      }
      else if (response instanceof FailureResponse)
      {
         FailureResponse failureResponse = (FailureResponse)response;
         Failure failure = failureResponse.getFailure();
         return new TestFailure(failure, parametrization, duration);
      }
      else
      {
         Failure failure = new Failure("The test driver returned an unexpected response " + response, FailureType.ERROR);
         return new TestFailure(failure, parametrization, duration);
      }
   }
}
