/******************************************************************************
 * JBoss, a division of Red Hat                                               *
 * Copyright 2006, Red Hat Middleware, LLC, and individual                    *
 * contributors as indicated by the @authors tag. See the                     *
 * copyright.txt in the distribution for a full listing of                    *
 * individual contributors.                                                   *
 *                                                                            *
 * This is free software; you can redistribute it and/or modify it            *
 * under the terms of the GNU Lesser General Public License as                *
 * published by the Free Software Foundation; either version 2.1 of           *
 * the License, or (at your option) any later version.                        *
 *                                                                            *
 * This software is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU           *
 * Lesser General Public License for more details.                            *
 *                                                                            *
 * You should have received a copy of the GNU Lesser General Public           *
 * License along with this software; if not, write to the Free                *
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA         *
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.                   *
 ******************************************************************************/
package org.jboss.unit.runner.model;

import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.w3c.dom.Node;
import org.w3c.dom.Attr;
import org.w3c.dom.Document;

import static org.jboss.unit.util.XMLTools.*;
import org.jboss.unit.runner.model.pojo.POJOTestSuiteDef;
import org.jboss.unit.runner.model.pojo.TestClassDef;
import org.jboss.unit.runner.model.pojo.TestCaseDef;
import org.jboss.unit.runner.model.composite.CompositeTestSuiteDef;
import org.jboss.unit.runner.model.generic.GenericTestSuiteDef;
import org.jboss.unit.runner.model.generic.TestDef;
import org.jboss.unit.TestIdFormat;
import org.jboss.unit.TestId;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;

import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;
import javax.xml.XMLConstants;
import javax.xml.transform.stream.StreamSource;
import java.util.List;
import java.util.ArrayList;
import java.util.Map;
import java.util.HashMap;
import java.io.IOException;
import java.io.InputStream;

/**
 * Builds a test suite from an XML stream.
 *
 * @author <a href="mailto:julien@jboss.org">Julien Viet</a>
 * @version $Revision: 1.1 $
 */
public class ModelBuilder
{

   /** . */
   private final Element rootElt;

   /** . */
   private static final Schema schema;

   static
   {
      try
      {
         SchemaFactory schemaFactory = SchemaFactory.newInstance(XMLConstants.W3C_XML_SCHEMA_NS_URI);
         InputStream in = ModelBuilder.class.getResourceAsStream(Constants.RELATIVE_RESOURCE_NAME);
         StreamSource ss = new StreamSource(in);
         schema = schemaFactory.newSchema(ss);
      }
      catch (SAXException e)
      {
         throw new Error("Could not load jboss unit schema for validation", e);
      }
   }

   public ModelBuilder(InputSource source) throws ParserConfigurationException, IOException, SAXException
   {
      if (source == null)
      {
         throw new IllegalArgumentException("No null source allowed");
      }

      //
      try
      {
         DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
         factory.setNamespaceAware(true);
         factory.setSchema(schema);
         DocumentBuilder builder = factory.newDocumentBuilder();
         Document doc = builder.parse(source);

         //
         this.rootElt = doc.getDocumentElement();
      }
      catch (SAXParseException e)
      {
         System.out.println("Parse exception at line " + e.getLineNumber() + " and column " + e.getColumnNumber() + ": " + e.getMessage());
         throw e;
      }
   }

   public ModelBuilder(Element rootElt)
   {
      if (rootElt == null)
      {
         throw new IllegalArgumentException("No null suite element allowed");
      }

      //
      this.rootElt = rootElt;
   }

   public TestSuiteDef build()
   {
      return buildTestSuite(rootElt);
   }

   private TestSuiteDef buildTestSuite(Element suiteElt)
   {
      String suiteName = suiteElt.getNodeName();

      //
      if (suiteName.equals("jboss-unit") || suiteName.equals("suite"))
      {
         CompositeTestSuiteDef composite = new CompositeTestSuiteDef();
         for (Element childSuiteElt : children(suiteElt))
         {
            TestSuiteDef childSuite = buildTestSuite(childSuiteElt);
            composite.addSuite(childSuite);
         }
         return composite;
      }
      else if (suiteName.equals("pojo"))
      {
         return buildPOJOTestSuite(suiteElt);
      }
      else if (suiteName.equals("generic"))
      {
         return buildGenericTestSuite(suiteElt);
      }

      //
      throw new AssertionError();
   }

   private TestSuiteDef buildGenericTestSuite(Element suiteElt)
   {
      Element classElt = child(suiteElt, "class");
      if (classElt == null)
      {
         throw new IllegalArgumentException();
      }

      //
      Attr nameAttrOfClassElt = classElt.getAttributeNode("name");
      if (nameAttrOfClassElt == null)
      {
         throw new IllegalArgumentException();
      }

      //
      GenericTestSuiteDef suite = new GenericTestSuiteDef(nameAttrOfClassElt.getValue());

      //
      for (Element testElt : children(suiteElt, "test"))
      {
         Attr refidAttr = testElt.getAttributeNode("refid");
         if (refidAttr == null)
         {
            throw new IllegalArgumentException();
         }

         //
         String refIdLitteral = refidAttr.getValue();

         //
         TestId refId = TestIdFormat.CLASSIC.parse(refIdLitteral);

         //
         TestDef test = new TestDef(refId);

         //
         test.setParameters(buildParameters(testElt));

         //
         suite.addTest(test);
      }

      //
      suite.setProperties(buildProperties(suiteElt));

      //
      suite.setParameters(buildParameters(suiteElt));

      //
      return suite;
   }

   private TestSuiteDef buildPOJOTestSuite(Element suiteElt)
   {
      POJOTestSuiteDef suite = new POJOTestSuiteDef();
      
      //
      for (Element testElt : children(suiteElt, "test"))
      {
         Element classElt = child(testElt, "class");

         //
         Attr nameAttr = classElt.getAttributeNode("name");
         if (nameAttr == null)
         {
            throw new IllegalArgumentException();
         }

         //
         String classname = nameAttr.getValue();
         TestClassDef testClass = new TestClassDef(classname);

         //
         testClass.setParameters(buildParameters(testElt));
         testClass.setProperties(buildProperties(testElt));

         //
         for (Element caseElt : children(classElt, "case"))
         {
            Attr caseNameAttr = caseElt.getAttributeNode("name");
            if (caseNameAttr == null)
            {
               throw new IllegalArgumentException();
            }

            //
            String testCaseName = caseNameAttr.getValue();
            TestCaseDef testCase = new TestCaseDef(testCaseName);
            testClass.addTestCase(testCase);
         }

         //
         suite.addClass(testClass);
      }

      //
      suite.setParameters(buildParameters(suiteElt));

      //
      return suite;
   }

   private static final int NO_SOURCE = 0;
   private static final int ELEMENT_SOURCE = 1;
   private static final int ATTRIBUTE_SOURCE = 2;

   private Map<String, String> buildProperties(Element propertyEltContainer)
   {
      Map<String, String> properties = new HashMap<String, String>();
      for (Element propertyElt : children(propertyEltContainer, "property"))
      {
         String propertyName = propertyElt.getAttribute("name");
         String propertyValue = propertyElt.getAttribute("value");
         properties.put(propertyName, propertyValue);
      }
      return properties;
   }

   private ParametersDef buildParameters(Element parameterEltContainer)
   {
      ParametersDef parameters = new ParametersDef();
      for (Element parameterElt : children(parameterEltContainer, "parameter"))
      {
         String parameterName = parameterElt.getAttribute("name");

         List<String> values = new ArrayList<String>();

         Attr attr = parameterElt.getAttributeNode("value");

         int src = NO_SOURCE;

         if (attr != null)
         {
            src = ATTRIBUTE_SOURCE;
            values.add(attr.getValue());
         }

         NodeList children = parameterElt.getChildNodes();

         for (int i = 0;i < children.getLength();i++)
         {
            Node child = children.item(i);
            switch(child.getNodeType())
            {
               case Node.ELEMENT_NODE:
                  Element elt = (Element)child;
                  if (src == ATTRIBUTE_SOURCE)
                  {
                     throw new IllegalArgumentException();
                  }
                  src = ELEMENT_SOURCE;
                  String eltName = elt.getNodeName();
                  if ("value".equals(eltName))
                  {
                     values.add(text(elt));
                  }
                  else if ("null".equals(eltName))
                  {
                     values.add(null);
                  }
                  else
                  {
                     throw new IllegalArgumentException();
                  }
                  break;
            }
         }


         if (values.size() == 0)
         {
            throw new IllegalArgumentException();
         }

         ParameterValueDef parameterValue = new ParameterValueDef(values);

         parameters.setParameter(parameterName, parameterValue);

      }

      return parameters;
   }


}
