/******************************************************************************
 * JBoss, a division of Red Hat                                               *
 * Copyright 2006, Red Hat Middleware, LLC, and individual                    *
 * contributors as indicated by the @authors tag. See the                     *
 * copyright.txt in the distribution for a full listing of                    *
 * individual contributors.                                                   *
 *                                                                            *
 * This is free software; you can redistribute it and/or modify it            *
 * under the terms of the GNU Lesser General Public License as                *
 * published by the Free Software Foundation; either version 2.1 of           *
 * the License, or (at your option) any later version.                        *
 *                                                                            *
 * This software is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU           *
 * Lesser General Public License for more details.                            *
 *                                                                            *
 * You should have received a copy of the GNU Lesser General Public           *
 * License along with this software; if not, write to the Free                *
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA         *
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.                   *
 ******************************************************************************/
package org.jboss.unit.spi.pojo;

import java.util.Map;

/**
 * Breaks down the Start/End protocol of a <code>TestDriver</code> into a protocol adapted to POJOs which have several phases:
 * <ul>
 *    <li>instantiation : instantiate the test case</li>
 *    <li>parametrization : parametrize the test case with the start command parametrization</li>
 *    <li>create : call a create life cycle</li>
 *    <li>invoke : executes the test</li>
 *    <li>destroy : call a destroy life cycle, it is guaranteed to be always called</li>
 * </ul>
 *
 * @author <a href="mailto:julien@jboss.org">Julien Viet</a>
 * @version $Revision: 1.1 $
 */
public interface TestLifeCycle
{

   /**
    * Instantiate life cycle operation of a test case.
    *
    * @param name the test case name
    * @return the test case object
    * @throws TestCaseLifeCycleException
    */
   TestCase newTestCase(String name) throws TestCaseLifeCycleException;

   /**
    * Parametrize life cycle operation of a test case.
    *
    * @param testCase the test case
    * @param parametrization the parametrization
    * @throws TestCaseLifeCycleException
    */
   void testCaseParametrize(TestCase testCase, Map<String,String> parametrization) throws TestCaseLifeCycleException;

   /**
    * Create life cycle operation of a test case.
    *
    * @param testCase the test case
    * @throws TestCaseLifeCycleException
    */
   void testCaseCreate(TestCase testCase) throws TestCaseLifeCycleException;

   /**
    * Invocation life cycle operation of a test case.
    *
    * @param testCase the test case
    * @throws TestCaseLifeCycleException
    */
   void testCaseInvoke(TestCase testCase) throws TestCaseLifeCycleException;

   /**
    * Destroy life cycle operation of a test case.
    *
    * @param testCase the test case
    */
   void testCaseDestroy(TestCase testCase);

}
