/******************************************************************************
 * JBoss, a division of Red Hat                                               *
 * Copyright 2006, Red Hat Middleware, LLC, and individual                    *
 * contributors as indicated by the @authors tag. See the                     *
 * copyright.txt in the distribution for a full listing of                    *
 * individual contributors.                                                   *
 *                                                                            *
 * This is free software; you can redistribute it and/or modify it            *
 * under the terms of the GNU Lesser General Public License as                *
 * published by the Free Software Foundation; either version 2.1 of           *
 * the License, or (at your option) any later version.                        *
 *                                                                            *
 * This software is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU           *
 * Lesser General Public License for more details.                            *
 *                                                                            *
 * You should have received a copy of the GNU Lesser General Public           *
 * License along with this software; if not, write to the Free                *
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA         *
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.                   *
 ******************************************************************************/
package org.jboss.unit.spi.pojo;

import org.jboss.unit.api.pojo.annotations.Test;
import org.jboss.unit.api.pojo.annotations.Parameter;
import org.jboss.unit.api.pojo.annotations.Create;
import org.jboss.unit.api.pojo.annotations.Destroy;
import org.jboss.unit.api.pojo.annotations.Description;
import org.jboss.unit.api.pojo.annotations.Tag;
import org.jboss.unit.util.CollectionTools;

import java.util.SortedMap;
import java.util.TreeMap;
import java.util.Map;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.HashSet;
import java.util.Collections;
import java.lang.reflect.Modifier;
import java.lang.reflect.Constructor;
import java.lang.reflect.AnnotatedElement;
import java.lang.reflect.Method;
import java.lang.reflect.InvocationTargetException;
import java.lang.annotation.Annotation;
import java.beans.Introspector;

/**
 * Provide a default implementation that relies on the annotation defined in the <code>annotations</code> package.
 *
 * @author <a href="mailto:julien@jboss.org">Julien Viet</a>
 * @version $Revision: 1.1 $
 */
public class TestProviderSupport implements TestProvider, TestLifeCycle, TestSuiteDescriptor
{

   /** . */
   private final String suiteName;

   /** . */
   private final String suiteDescription;

   /** . */
   private final Map<String, TestCaseDef> testCases;

   /** . */
   private final Method create;

   /** . */
   private final Method destroy;

   /** . */
   final Map<String, PropertyTestParameter> propertyParameters;

   /** . */
   private final Constructor ctor;

   /** . */
   private final Set<String> suiteKeywords;

   /** . */
   final Map<String, TestParameter> suiteParameters;

   public TestProviderSupport(Class testClass)
   {
      if (testClass == null)
      {
         throw new IllegalArgumentException();
      }

      //
      if (Modifier.isAbstract(testClass.getModifiers()))
      {
         throw new IllegalArgumentException("Test class is abstract");
      }

      // Check constructor
      Constructor ctor;
      try
      {
         ctor = testClass.getConstructor();
      }
      catch (NoSuchMethodException e)
      {
         throw new IllegalArgumentException(e);
      }

      //
      Test testAnnotation = ((AnnotatedElement)testClass).getAnnotation(Test.class);
      String suiteName = testAnnotation != null ? testAnnotation.name() : "";
      if (suiteName.length() == 0)
      {
         suiteName = testClass.getName();
      }

      //
      Description descriptionAnnotation = ((AnnotatedElement)testClass).getAnnotation(Description.class);
      String suiteDescription = descriptionAnnotation != null ? descriptionAnnotation.value() : "";
      if (suiteDescription.length() == 0)
      {
         suiteDescription = "Test of class " + testClass.getName();
      }

      Tag tag = ((AnnotatedElement)testClass).getAnnotation(Tag.class);
      Set<String> suiteKeywords = new HashSet<String>();
      if (tag != null)
      {
         suiteKeywords.addAll(CollectionTools.set(tag.value()));
      }

      //
      SortedMap<MethodKey, Method> methods = new TreeMap<MethodKey, Method>();
      for (Method method : testClass.getMethods())
      {
         MethodKey key = new MethodKey(method);
         methods.put(key, method);
      }

      //
      Map<String, PropertyTestParameter> propertyParameters = new HashMap<String, PropertyTestParameter>();
      Map<String, TestCaseDef> testCases = new HashMap<String, TestCaseDef>();
      Method create = null;
      Method destroy = null;

      // First pass for parameters and life cycle annotations
      for (Method method : methods.values())
      {
         String methodName = method.getName();
         int modifiers = method.getModifiers();
         Parameter parameterMethodAnnotation = method.getAnnotation(Parameter.class);
         Test testMethodAnnotation = method.getAnnotation(Test.class);
         Create createMethodAnnotation = method.getAnnotation(Create.class);
         Destroy destroyMethodAnnotation = method.getAnnotation(Destroy.class);

         //
         if (testMethodAnnotation != null && parameterMethodAnnotation != null)
         {
            throw new IllegalArgumentException();
         }

         //
         if (createMethodAnnotation != null && parameterMethodAnnotation != null)
         {
            throw new IllegalArgumentException();
         }
         if (createMethodAnnotation != null && testMethodAnnotation != null)
         {
            throw new IllegalArgumentException();
         }

         //
         if (destroyMethodAnnotation != null && parameterMethodAnnotation != null)
         {
            throw new IllegalArgumentException();
         }
         if (destroyMethodAnnotation != null && testMethodAnnotation != null)
         {
            throw new IllegalArgumentException();
         }
         if (destroyMethodAnnotation != null && createMethodAnnotation != null)
         {
            throw new IllegalArgumentException();
         }

         //
         if (parameterMethodAnnotation != null)
         {
            if (!methodName.startsWith("set"))
            {
               throw new IllegalArgumentException();
            }
            if (methodName.length() < 4)
            {
               throw new IllegalArgumentException();
            }
            if (method.getParameterTypes().length != 1)
            {
               throw new IllegalArgumentException();
            }
            if (method.getReturnType() != void.class)
            {
               throw new IllegalArgumentException();
            }
            if (Modifier.isAbstract(modifiers))
            {
               throw new IllegalArgumentException();
            }
            if (!Modifier.isPublic(modifiers))
            {
               throw new IllegalArgumentException();
            }
            if (Modifier.isStatic(modifiers))
            {
               throw new IllegalArgumentException();
            }

            //
            String name = parameterMethodAnnotation.name();
            if (name.length() == 0)
            {
               name = Introspector.decapitalize(method.getName().substring(3));
            }

            //
            Description descriptionMethodAnnotation = method.getAnnotation(Description.class);
            String description = descriptionMethodAnnotation != null ? descriptionMethodAnnotation.value() : "";
            if (description.length() == 0)
            {
               description = "Parameter " + name;
            }

            //
            PropertyTestParameter parameter = propertyParameters.get(name);
            if (parameter == null)
            {
               parameter = new PropertyTestParameter(name, description);
               propertyParameters.put(name, parameter);
            }

            //
            parameter.addSetter(method);
         }
         else if (createMethodAnnotation != null)
         {
            if (create != null)
            {
               throw new IllegalArgumentException();
            }
            if (method.getReturnType() != void.class)
            {
               throw new IllegalArgumentException();
            }
            if (Modifier.isAbstract(modifiers))
            {
               throw new IllegalArgumentException();
            }
            if (!Modifier.isPublic(modifiers))
            {
               throw new IllegalArgumentException();
            }
            if (Modifier.isStatic(modifiers))
            {
               throw new IllegalArgumentException();
            }
            if (method.getParameterTypes().length > 0)
            {
               throw new IllegalArgumentException();
            }

            //
            create = method;
         }
         else if (destroyMethodAnnotation != null)
         {
            if (destroy != null)
            {
               throw new IllegalArgumentException();
            }
            if (method.getReturnType() != void.class)
            {
               throw new IllegalArgumentException();
            }
            if (Modifier.isAbstract(modifiers))
            {
               throw new IllegalArgumentException();
            }
            if (!Modifier.isPublic(modifiers))
            {
               throw new IllegalArgumentException();
            }
            if (Modifier.isStatic(modifiers))
            {
               throw new IllegalArgumentException();
            }
            if (method.getParameterTypes().length > 0)
            {
               throw new IllegalArgumentException();
            }

            //
            destroy = method;
         }
      }

      // Second pass for test annotations
      for (Method method : methods.values())
      {
         int modifiers = method.getModifiers();
         Test testMethodAnnotation = method.getAnnotation(Test.class);

         //
         if (testMethodAnnotation != null)
         {
            if (method.getReturnType() != void.class)
            {
               throw new IllegalArgumentException();
            }
            if (Modifier.isAbstract(modifiers))
            {
               throw new IllegalArgumentException();
            }
            if (!Modifier.isPublic(modifiers))
            {
               throw new IllegalArgumentException();
            }
            if (Modifier.isStatic(modifiers))
            {
               throw new IllegalArgumentException();
            }

            //
            LinkedHashMap<String, ArgumentTestParameter> methodParameters = new LinkedHashMap<String, ArgumentTestParameter>();
            for (Annotation[] parameterAnnotations : method.getParameterAnnotations())
            {
               Map<Class<? extends Annotation>,Annotation> parameterAnnotationMap = new HashMap<Class<? extends Annotation>, Annotation>();
               for (Annotation parameterAnnotation : parameterAnnotations)
               {
                  Class<? extends Annotation> blah = parameterAnnotation.annotationType();
                  parameterAnnotationMap.put(blah, parameterAnnotation);
               }

               //
               Parameter argumentParameterAnnotation = (Parameter)parameterAnnotationMap.get(Parameter.class);

               //
               if (argumentParameterAnnotation != null)
               {
                  //
                  String name = argumentParameterAnnotation.name();
                  if (name.length() == 0)
                  {
                     throw new IllegalArgumentException();
                  }
                  if (methodParameters.containsKey(name))
                  {
                     throw new IllegalArgumentException();
                  }

                  //
                  Description descriptionArgumentParameterAnnotation = (Description)parameterAnnotationMap.get(Description.class);
                  String description = descriptionArgumentParameterAnnotation != null ? descriptionArgumentParameterAnnotation.value() : "";
                  if (description.length() == 0)
                  {
                     description = "Method parameter " + name;
                  }

                  //
                  methodParameters.put(name, new ArgumentTestParameter(name, description));
               }
               else
               {
                  throw new IllegalArgumentException();
               }
            }

            //
            String name = testMethodAnnotation.name();
            if (testMethodAnnotation.name().length() == 0)
            {
               name = method.getName();
            }

            //
            Description descriptionMethodAnnotation = method.getAnnotation(Description.class);
            String description = descriptionMethodAnnotation != null ? descriptionMethodAnnotation.value() : "";
            if (description.length() == 0)
            {
               description = "Test of method " + method.getName();
            }

            //
            Set<String> keywords = new HashSet<String>();
            Tag tagMethodAnnotation = method.getAnnotation(Tag.class);
            if (tagMethodAnnotation != null)
            {
               keywords.addAll(CollectionTools.set(tagMethodAnnotation.value()));
            }

            //
            Map<String,TestParameter> parameters = new HashMap<String, TestParameter>(propertyParameters);
            parameters.putAll(methodParameters);

            //
            TestCaseDef testCaseDef = new TestCaseDef(method, name, description, parameters, methodParameters, keywords);

            //
            if (testCases.put(testCaseDef.getName(), testCaseDef) != null)
            {
               throw new IllegalArgumentException();
            }
         }
      }

      // Set the state computed from the annotations
      this.suiteName = suiteName;
      this.suiteDescription = suiteDescription;
      this.suiteKeywords = suiteKeywords;
      this.propertyParameters = propertyParameters;
      this.testCases = testCases;
      this.ctor = ctor;
      this.create = create;
      this.destroy = destroy;

      // Compute the suite parameters
      Map<String, TestParameter> suiteParameters = new HashMap<String, TestParameter>();
      for (TestCaseDef testCase : testCases.values())
      {
         suiteParameters.putAll(testCase.arguments);
      }
      suiteParameters.putAll(propertyParameters);

      // Set the state computed from the existing state
      this.suiteParameters = suiteParameters;
   }

   public TestSuiteDescriptor getDescriptor()
   {
      return this;
   }

   public TestLifeCycle getLifeCycle()
   {
      return this;
   }

   public String getName()
   {
      return suiteName;
   }

   public String getDescription()
   {
      return suiteDescription;
   }

   public Set<String> getKeywords()
   {
      return suiteKeywords;
   }

   public Map<String, ? extends ParameterDescriptor> getParameters()
   {
      return Collections.unmodifiableMap(suiteParameters);
   }

   public Map<String, ? extends TestCaseDescriptor> getTestCases()
   {
      return Collections.unmodifiableMap(testCases);
   }

   public TestCase newTestCase(String name) throws TestCaseLifeCycleException
   {
      TestCaseDef testCaseDef = testCases.get(name);

      //
      if (testCaseDef == null)
      {
         throw new TestCaseLifeCycleException("No such test case <" + name + ">");
      }

      //
      Object instance;
      try
      {
         instance = ctor.newInstance();
      }
      catch (InvocationTargetException e)
      {
         throw new TestCaseLifeCycleException("Cannot create test case " + name, e.getCause());
      }
      catch (Exception e)
      {
         throw new TestCaseLifeCycleException("Cannot configure parameter " + name, e);
      }

      //
      return new TestCaseImpl(testCaseDef, instance);
   }

   public void testCaseParametrize(TestCase _testCase, Map<String,String> parametrization) throws TestCaseLifeCycleException
   {
      TestCaseImpl testCase = (TestCaseImpl)_testCase;

      // Save parametrization
      testCase.parametrization = parametrization;

      //
      for (PropertyTestParameter parameter : propertyParameters.values())
      {
         if (!parametrization.containsKey(parameter.getName()))
         {
            throw new TestCaseLifeCycleException("Missing parameter " + parameter.getName());
         }
         String parameterValue = parametrization.get(parameter.getName());
         for (Method setter : parameter.setters)
         {
            try
            {
               setter.invoke(testCase.pojo, parameterValue);
            }
            catch (InvocationTargetException e)
            {
               throw new TestCaseLifeCycleException("Cannot configure parameter " + parameter.getName(), e.getCause());
            }
            catch (Exception e)
            {
               throw new TestCaseLifeCycleException("Cannot configure parameter " + parameter.getName(), e);
            }
         }
      }
   }

   public void testCaseCreate(TestCase _testCase) throws TestCaseLifeCycleException
   {
      TestCaseImpl testCase = (TestCaseImpl)_testCase;

      //
      if (create != null)
      {
         try
         {
            create.invoke(testCase.getPOJO());
         }
         catch (InvocationTargetException e)
         {
            throw new TestCaseLifeCycleException("Cannot create test case", e.getCause());
         }
         catch (Exception e)
         {
            throw new TestCaseLifeCycleException("Cannot create test case", e);
         }
      }
   }

   public void testCaseInvoke(TestCase _testCase) throws TestCaseLifeCycleException
   {
      TestCaseImpl testCase = (TestCaseImpl)_testCase;

      List<Object> argList = new ArrayList<Object>();
      for (String parameterName : testCase.def.arguments.keySet())
      {
         if (!testCase.parametrization.containsKey(parameterName))
         {
            throw new TestCaseLifeCycleException("Missing parameter " + parameterName);
         }
         String parameterValue = testCase.parametrization.get(parameterName);
         if (parameterValue == null)
         {
            throw new IllegalArgumentException();
         }

         //
         argList.add(parameterValue);
      }
      Object[] args = argList.toArray();

      try
      {
         testCase.def.method.invoke(testCase.pojo, args);
      }
      catch (InvocationTargetException e)
      {
         throw new TestCaseLifeCycleException("Cannot invoke test case", e.getCause());
      }
      catch (Exception e)
      {
         throw new TestCaseLifeCycleException("Cannot invoke test case", e);
      }
   }

   public void testCaseDestroy(TestCase _testCase)
   {
      TestCaseImpl testCase = (TestCaseImpl)_testCase;

      //
      if (destroy != null)
      {
         try
         {
            destroy.invoke(testCase.getPOJO());
         }
         catch (Throwable t)
         {
            // Log this as a warn
            // t.printStackTrace();
         }
      }
   }
}
