/*
* JBoss, a division of Red Hat
* Copyright 2006, Red Hat Middleware, LLC, and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/

package org.jboss.unit.tooling;

import org.jboss.unit.runner.TestRunner;
import org.jboss.unit.runner.ParametrizationSet;
import org.jboss.unit.runner.impl.TestRunnerContextSupport;
import org.jboss.unit.runner.impl.TestRunnerEventBroadcaster;
import org.jboss.unit.runner.model.TestSuiteDef;
import org.jboss.unit.tooling.filter.MainTestFilter;
import org.jboss.unit.report.impl.writer.PrintListener;

import java.util.Set;
import java.util.HashSet;
import java.util.Map;
import java.util.HashMap;
import java.util.Collections;
import java.io.StringWriter;

/**
 * @author <a href="mailto:boleslaw dot dawidowicz at redhat anotherdot com">Boleslaw Dawidowicz</a>
 * @version $Revision: 0.1 $
 */
public class Main
{

   private final MainTestFilter mainFilter;
   
   private final TestRunner runner;

   private final TestSuiteDef suiteDef;

   private final Map<String,String> properties;

   private final Map<String, String[]> parameters;

   private final TestRunnerEventBroadcaster broadcaster;

   private final boolean failonerror;

   public static final String JBOSS_UNIT_PROPERTIES = "jboss.unit.properties";

   public static final String JBOSS_UNIT_PROPERTY = "jboss.unit.property";

   public static final String JBOSS_UNIT_PARAMETER = "jboss.unit.parameter";

   public static final String JBOSS_UNIT_PARAMETERS = "jboss.unit.parameters";

   public static final String OPT_FAIL_ON_ERROR = "--failonerror";

   public static final String OPT_ID = "--id";

   public static final String OPT_IDS = "--ids";

   public static final String OPT_KEYWORDS = "--keywords";

   public static final String OPT_EX_ID = "--ex_id";

   public static final String OPT_EX_IDS = "--ex_ids";

   public static final String OPT_EX_KEYWORDS = "--ex_keywords";

   public static final String OPT_CONFIG = "--config";

   public static final String OPT_SUITE_NAME = "--suite_name";

   public static final String OPT_NO_CONSOLE = "--no_console";

   public static final String OPT_XML_REPORT_DIR = "--xml_report_dir";

   public static final String OPT_HTML_REPORT_DIR = "--html_report_dir";

   /**
    * For now usage should be like
    *
    * --property name=value .
    */
   public static final String OPT_PROPERTY = "--property";

   public static final String OPT_PROPERTIES = "--properties";

   public static final String OPT_PARAMETER = "--parameter";

   public static final String OPT_PARAMETERS = "--parameters";

   public static final Set options;

   public static final Set options_no_arg;

   static {
      Set<String> opts = new HashSet<String>();
      opts.add(OPT_ID);
      opts.add(OPT_IDS);
      opts.add(OPT_KEYWORDS);
      opts.add(OPT_EX_ID);
      opts.add(OPT_EX_IDS);
      opts.add(OPT_EX_KEYWORDS);
      opts.add(OPT_CONFIG);
      opts.add(OPT_SUITE_NAME);
      opts.add(OPT_NO_CONSOLE);
      opts.add(OPT_XML_REPORT_DIR);
      opts.add(OPT_HTML_REPORT_DIR);
      opts.add(OPT_PROPERTY);
      opts.add(OPT_FAIL_ON_ERROR);

      options = Collections.unmodifiableSet(opts);

      opts = new HashSet<String>();
      opts.add(OPT_NO_CONSOLE);
      opts.add(OPT_FAIL_ON_ERROR);
      options_no_arg = Collections.unmodifiableSet(opts);
   }

   public Main(TestRunner runner, MainTestFilter mainFilter, TestRunnerEventBroadcaster broadcaster, TestSuiteDef suiteDef, Map<String,String> properties, Map<String, String[]> parameters, boolean failonerror)
   {
      if (runner == null)
      {
         throw new IllegalArgumentException("runner cannot be null");
      }

      if (mainFilter == null)
      {
         throw new IllegalArgumentException("mainFilter cannot be null");
      }

      if (broadcaster == null)
      {
         throw new IllegalArgumentException("broadcaster cannot be null");
      }

      if (suiteDef == null)
      {
         throw new IllegalArgumentException("suiteDef cannot be null");
      }

      if (properties == null)
      {
         throw new IllegalArgumentException("properties cannot be null");
      }

      if (parameters == null)
      {
         throw new IllegalArgumentException("parameters cannot be null");
      }

      this.mainFilter = mainFilter;
      this.runner = runner;
      this.suiteDef = suiteDef;
      this.properties = properties;
      this.broadcaster = broadcaster;
      this.parameters = parameters;
      this.failonerror = failonerror;
   }

   public void execute() throws Exception
   {
      SystemOutputManager systemManager = new SystemOutputManager();
      systemManager.start();

      FailureManagerListener failureListener = new FailureManagerListener();

      //
      try
      {
         // For now lets just be verbose. Later we need some real logging + 'debug' switch
         System.out.println("TestRunnerContext contains: ");
         System.out.println("Properties: " + properties.toString());

//      System.out.println("System properties:");
//      for (Object key : System.getProperties().keySet())
//      {
//
//         System.out.println(key + "=" + System.getProperty((String)key));
//      }

         StringWriter sw = new StringWriter();
         broadcaster.addListener(new PrintListener(sw));



         // Track runner and test case failures;
         broadcaster.addListener(failureListener);

         TestRunnerContextSupport runnerContext = new TestRunnerContextSupport(properties, new ParametrizationSet(parameters), mainFilter, broadcaster);
         runner.run(runnerContext);

         sw.close();

         System.out.println(sw.toString());




      }
      finally
      {
         systemManager.stop();

         if (failonerror && (failureListener.isRunnerFailure() || failureListener.isTestCaseFailure()))
         {
            System.exit(-1);
         }
      }

   }

   public static void main(String[] args) throws Exception
   {
      Map<String, String> arguments = new HashMap();

      for (String arg : args)
      {
         String[] params = arg.split("=",2);

         if (!options_no_arg.contains(params[0]))
         {
            if (params.length < 2)
            {
               throw new IllegalArgumentException("Argument '" + arg + "' doesn't follow the name=value pattern");
            }
            
            arguments.put(params[0], params[1]);
         }
         else
         {
            arguments.put(params[0], null);
         }
      }

      if (!arguments.keySet().contains(OPT_CONFIG))
      {
         throw new IllegalArgumentException("Argument '" + OPT_CONFIG + "' is missing");
      }



      MainBuilder builder = null;

      if (arguments.keySet().contains(OPT_SUITE_NAME))
      {
         builder = new MainBuilder(arguments.get(OPT_CONFIG), arguments.get(OPT_SUITE_NAME));

         builder.setSuiteName(arguments.get(OPT_SUITE_NAME));
      }
      else
      {
         builder = new MainBuilder(arguments.get(OPT_CONFIG));

      }

      for (String argument : arguments.keySet())
      {

         if (argument.equals(OPT_IDS))
         {
            builder.addUnparsedNames(arguments.get(OPT_IDS));
         }

         if (argument.equals(OPT_ID))
         {
            builder.addName(arguments.get(OPT_ID));
         }

         if (argument.equals(OPT_KEYWORDS))
         {
            builder.addUnparsedKeywords(arguments.get(OPT_KEYWORDS));
         }

         if (argument.equals(OPT_EX_IDS))
         {
            builder.addUnparsedExcludeNames(arguments.get(OPT_EX_IDS));
         }

         if (argument.equals(OPT_EX_ID))
         {
            builder.addExcludeName(arguments.get(OPT_EX_ID));
         }

         if (argument.equals(OPT_EX_KEYWORDS))
         {
            builder.addUnparsedExcludeKeywords(arguments.get(OPT_EX_KEYWORDS));
         }

         if (argument.equals(OPT_NO_CONSOLE))
         {
            builder.setNoConsole(true);
         }

         if (argument.equals(OPT_FAIL_ON_ERROR))
         {
            builder.setFailonerror(true);
         }

         if (argument.equals(OPT_XML_REPORT_DIR))
         {
            builder.setXMLReportDir(arguments.get(OPT_XML_REPORT_DIR));
         }

         if (argument.equals(OPT_HTML_REPORT_DIR))
         {
            builder.setHTMLReportDir(arguments.get(OPT_HTML_REPORT_DIR));
         }

         if (argument.startsWith(OPT_PROPERTY) && !argument.equals(OPT_PROPERTIES))
         {

            // Pass whole --properties.name=value to be parsed
            builder.addUnparsedProperty(argument + arguments.get(OPT_PROPERTY));
         }

         if (argument.equals(OPT_PROPERTIES))
         {
            builder.addUnparsedProperties(arguments.get(OPT_PROPERTIES));
         }

         if (argument.startsWith(OPT_PARAMETER) && !argument.equals(OPT_PARAMETERS))
         {

            // Pass whole --parameter.name=value/values to be parsed
            builder.addUnparsedParameter(argument + arguments.get(OPT_PARAMETER));
         }

         if (argument.equals(OPT_PARAMETERS))
         {
            builder.addUnparsedParameters(arguments.get(OPT_PARAMETERS));
         }
         
      }





      Main main = builder.build();

      main.execute();
   }
}
