/*
* JBoss, a division of Red Hat
* Copyright 2006, Red Hat Middleware, LLC, and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/

package org.jboss.unit.tooling;

import org.jboss.unit.report.impl.junit.JUnitReporter;
import org.jboss.unit.report.impl.junit.JUnitTestsuiteReport;
import org.jboss.unit.runner.TestRunner;
import org.jboss.unit.runner.TestRunnerEventListener;
import org.jboss.unit.runner.impl.NullFilter;
import org.jboss.unit.runner.impl.TestRunnerEventBroadcaster;
import org.jboss.unit.runner.model.ModelBuilder;
import org.jboss.unit.runner.model.TestSuiteDef;
import org.jboss.unit.tooling.filter.ExcludeTestFilter;
import org.jboss.unit.tooling.filter.IncludeTestFilter;
import org.jboss.unit.tooling.filter.MainTestFilter;
import org.xml.sax.InputSource;

import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.io.FileNotFoundException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * @author <a href="mailto:boleslaw dot dawidowicz at redhat anotherdot com">Boleslaw Dawidowicz</a>
 * @version $Revision: 0.1 $
 */
public class MainBuilder
{

   private Set<String> keywords = new HashSet<String>();

   private Set<String> names = new HashSet<String>();

   private Set<String> namePatterns = new HashSet<String>();

   private Set<String> excludeKeywords = new HashSet<String>();

   private Set<String> excludeNames = new HashSet<String>();

   private Set<String> excludeNamePatterns = new HashSet<String>();

   private Set<TestRunnerEventListener> listeners = new HashSet<TestRunnerEventListener>();

   private String suiteName;

   private String configFile;

   private boolean noConsole = false;

   private boolean failonerror = false;

   private String xmlReportDir;

   private String htmlReportDir;

   private Map<String,String> properties = new HashMap<String, String>();

   private Map<String, List<String>> parameters = new HashMap<String, List<String>>();


   private MainBuilder()
   {

   }

   public MainBuilder(String configFile)
   {

      if (configFile == null)
      {
         throw new IllegalArgumentException("configFile cannot be null");
      }

      this.configFile = configFile;
   }

   public MainBuilder(String configFile, String name)
   {
      this(configFile);

      if (name == null)
      {
         throw new IllegalArgumentException("name cannot be null");
      }

      this.suiteName = name;

   }

   public MainBuilder addKeyword(String keyword)
   {
      if (keyword == null)
      {
         throw new IllegalArgumentException("keyword cannot be null");
      }

      keywords.add(keyword);
      return this;
   }

   public MainBuilder addExcludeKeyword(String keyword)
   {
      if (keyword == null)
      {
         throw new IllegalArgumentException("keyword cannot be null");
      }

      excludeKeywords.add(keyword);
      return this;
   }

   public MainBuilder addUnparsedKeywords(String keywordsString)
   {
      if (keywordsString == null)
      {
         throw new IllegalArgumentException("keywordsString cannot be null");
      }
      String[] keywordValues = keywordsString.split(",");
      for (String keyword : keywordValues)
      {
         this.keywords.add(keyword.trim());
      }
      return this;
   }

   public MainBuilder addUnparsedExcludeKeywords(String keywordsString)
   {
      if (keywordsString == null)
      {
         throw new IllegalArgumentException("keywordsString cannot be null");
      }
      String[] keywordValues = keywordsString.split(",");
      for (String keyword : keywordValues)
      {
         this.excludeKeywords.add(keyword.trim());
      }
      return this;
   }

   public MainBuilder addName(String name)
   {                                                       
      if (name == null)
      {
         throw new IllegalArgumentException("name cannot be null");
      }

      names.add(name);
      return this;
   }

    public MainBuilder addExcludeName(String name)
   {
      if (name == null)
      {
         throw new IllegalArgumentException("name cannot be null");
      }

      excludeNames.add(name);
      return this;
   }

   public MainBuilder addUnparsedNames(String namesString)
   {
      if (namesString == null)
      {
         throw new IllegalArgumentException("namesString cannot be null");
      }
      String[] namesValues = namesString.split(",");
      for (String name : namesValues)
      {
         this.names.add(name.trim());
      }
      return this;
   }

   public MainBuilder addUnparsedExcludeNames(String namesString)
   {
      if (namesString == null)
      {
         throw new IllegalArgumentException("namesString cannot be null");
      }
      String[] namesValues = namesString.split(",");
      for (String name : namesValues)
      {
         this.excludeNames.add(name.trim());
      }
      return this;
   }

   public MainBuilder addNamePattern(String pattern)
   {
      if (pattern == null)
      {
         throw new IllegalArgumentException("pattern cannot be null");
      }

      namePatterns.add(pattern);
      return this;
   }

   public MainBuilder addExcludeNamePattern(String pattern)
   {
      if (pattern == null)
      {
         throw new IllegalArgumentException("pattern cannot be null");
      }

      excludeNamePatterns.add(pattern);
      return this;
   }

   public MainBuilder addUnparsedNamePatterns(String patternString)
   {
      if (patternString == null )
      {
         throw new IllegalArgumentException("patternString cannot be null");
      }
      String[] patternValues = patternString.split(",");
      for (String pattern : patternValues)
      {
         this.namePatterns.add(pattern.trim());
      }
      return this;
   }

   public MainBuilder addUnparsedExcludeNamePatterns(String patternString)
   {
      if (patternString == null)
      {
         throw new IllegalArgumentException("patternString cannot be null");
      }
      String[] patternValues = patternString.split(",");
      for (String pattern : patternValues)
      {
         this.excludeNamePatterns.add(pattern.trim());
      }
      return this;
   }

   public MainBuilder addJUnitReporting(String directory)
   {
      if (directory == null)
      {
         throw new IllegalArgumentException("directory cannot be null");
      }

      listeners.add(new JUnitReporter(directory, suiteName));

      return this;
   }

   public MainBuilder addUnparsedProperty(String propertyString)
   {
      if (propertyString == null)
      {
         throw new IllegalArgumentException("No null property name accepted");
      }

      propertyString = propertyString.replaceFirst(Main.OPT_PROPERTY + "\\.", "");

      //
      String[] patternValues = propertyString.split("=");

      //
      if (patternValues.length != 2)
      {
         throw new IllegalArgumentException("Property not accepted, it should match a=b and it is " + propertyString);
      }

      //
      properties.put(patternValues[0], patternValues[1]);

      //
      return this;
   }

   public MainBuilder addProperty(String propertyName, String propertyValue)
   {
      if (propertyName == null)
      {
         throw new IllegalArgumentException("No null property name accepted");
      }

      //
      properties.put(propertyName, propertyValue);

      //
      return this;
   }

   public void addUnparsedProperties(String propertiesString)
   {
      if (propertiesString == null)
      {
         throw new IllegalArgumentException("Properties cannot be null");
      }

      Map<String, String> props =  parseProperties(propertiesString);

      for (String key : props.keySet())
      {
         addProperty(key, props.get(key));
      }


   }

   public static Map<String,String> parseProperties(String propertiesString)
   {
      if (!propertiesString.matches("\\[.+=.+(,.+=.+)*\\]"))
      {
         throw new IllegalArgumentException("Parameters don't follow proper format: [a=b,c=d,e=f]");
      }

      Map<String,String> parsedProps = new HashMap<String,String>();

      propertiesString = propertiesString.replaceAll("\\[","").replaceAll("\\]","");

      String propertyGroups[] = propertiesString.split(",");

      for (String propertyGroup : propertyGroups)
      {
         String property[] = propertyGroup.split("=");

         //
         if (property.length != 2)
         {
            throw new IllegalArgumentException("Property not accepted, it should match a=b and it is " + property);
         }

         parsedProps.put(property[0],property[1]);
      }
      return parsedProps;
   }

   public void addUnparsedParameter(String paramString)
   {
      paramString = paramString.replaceFirst(Main.OPT_PARAMETER + "\\.", "");

      //
      String[] parameter = paramString.split("=",2);

      //
      if (parameter.length != 2)
      {
         throw new IllegalArgumentException("Parameter not accepted, it should match a=b and it is " + paramString);
      }


      List<String> parameterValues = new LinkedList<String>();

      if (parameter[1].matches("\\[.+(,.+)*\\]"))
      {

         String valuesString = parameter[1];

         valuesString = valuesString.replaceAll("\\[","").replaceAll("\\]","");

         String[] values = valuesString.split(",");

         for (String value : values)
         {
            parameterValues.add(value);
         }

      }
      else
      {
         parameterValues.add(parameter[1]);
      }

      if (parameters.containsKey(parameter[0]))
      {
         parameters.get(parameter[0]).addAll(parameterValues);
      }
      else
      {
         parameters.put(parameter[0], parameterValues);
      }

   }

   public void addUnparsedParameters(String parametersString)
   {

      parametersString = parametersString.replaceAll("\\[","").replaceAll("\\]","");

      String parameterGroups[] = parametersString.split(",");

      for (String parameterGroup : parameterGroups)
      {

         String parameter[] = parameterGroup.split("=");

         //
         if (parameter.length != 2)
         {
            throw new IllegalArgumentException("Parameter not accepted, it should match a=b and it is " + parameter);
         }

         List<String> values;

         if (parameters.containsKey(parameter[0]))
         {
            values = parameters.get(parameter[0]);
         }
         else
         {
            values = new LinkedList<String>();
         }

         values.add(parameter[1]);

         parameters.put(parameter[0], values);
      }
   }

   public Map<String, String[]> getParameters()
   {
      Map<String, String[]> params = new HashMap<String,String[]>();

      for (String paramKey : parameters.keySet())
      {
         List<String> values = parameters.get(paramKey);
         params.put(paramKey, values.toArray(new String[values.size()]));
      }

      return params;

   }


   public String getSuiteName()
   {
      return suiteName;
   }

   public void setSuiteName(String suiteName)
   {
      this.suiteName = suiteName;
   }

   public void setNoConsole(boolean noConsole)
   {
      this.noConsole = noConsole;
   }

   public boolean isFailonerror()
   {
      return failonerror;
   }

   public void setFailonerror(boolean failonerror)
   {
      this.failonerror = failonerror;
   }

   public void setXMLReportDir(String xmlReportDir)
   {
      this.xmlReportDir = xmlReportDir;
   }

   public void setHTMLReportDir(String htmlReportDir)
   {
      this.htmlReportDir = htmlReportDir;
   }

   public Main build() throws Exception
   {


      InputStream in;

      try
      {
         in = new FileInputStream(configFile);
      }
      catch (FileNotFoundException e)
      {
         
         // If file was not found try to obtain it from context class loader
         
         in = Thread.currentThread().getContextClassLoader().getResourceAsStream(configFile);
      }

      if (in == null)
      {
         throw new IllegalArgumentException("Could not find specified file in classpath: " + configFile);  
      }
      
      InputSource source = new InputSource(in);
      ModelBuilder builder = new ModelBuilder(source);
      TestSuiteDef suite = builder.build();
      TestRunner runner = suite.createRunner();
      TestRunnerEventBroadcaster broadcaster = new TestRunnerEventBroadcaster();

      // Add listeners

      for (TestRunnerEventListener listener : listeners)
      {
         broadcaster.addListener(listener);
      }

      //add PrintListener by default

      //
      File htmlReportFile = null;
      if (htmlReportDir != null)
      {
         htmlReportFile = checkCreateOutputDirectory(htmlReportDir);
      }
      File xmlReportFile = null;
      if (xmlReportDir != null)
      {
         xmlReportFile = checkCreateOutputDirectory(xmlReportDir);
      }
      JUnitTestsuiteReport fileReporter = new JUnitTestsuiteReport(xmlReportFile, htmlReportFile);
      broadcaster.addListener(fileReporter);

      MainTestFilter filter = new MainTestFilter();

      if (keywords.size() == 0 && names.size() == 0 && namePatterns.size() == 0)
      {
         // If no conditions - add filter that match all

         filter.addIncludeFilter(new NullFilter());
      }
      else
      {
         filter.addIncludeFilter(new IncludeTestFilter(keywords, names, namePatterns));
      }

      // Add exclude filter only if there is a need for one...
         
      if (excludeKeywords.size() != 0 || excludeNames.size() != 0 || excludeNamePatterns.size() != 0)
      {
         filter.addExcludeFilter(new ExcludeTestFilter(excludeKeywords, excludeNames, excludeNamePatterns));
      }

      return new Main(runner, filter, broadcaster, suite, properties, getParameters(), isFailonerror());
   }

   public static File checkCreateOutputDirectory(String dir) throws Exception
   {
      try
      {
         File output = new File(dir);

         // heck if the dir exits and if not create it
         if (!output.exists())
         {
            output.mkdirs();
         }
         else
         {
            if (!output.isDirectory())
            {
               throw new IllegalStateException("Provided path points to a file instead of directory");
            }
         }

         //
         return output;
      }
      catch (Exception e)
      {
         throw new IllegalStateException("Failed to use provided output directory: '" + dir + "'", e);
      }
   }
}

