/*
* JBoss, a division of Red Hat
* Copyright 2006, Red Hat Middleware, LLC, and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/

package org.jboss.unit.tooling.filter;

import org.jboss.unit.runner.TestFilter;
import org.jboss.unit.TestId;
import org.jboss.unit.info.TestInfo;

import java.util.Set;
import java.util.HashSet;
import java.util.Map;

/**
 * @author <a href="mailto:boleslaw dot dawidowicz at redhat anotherdot com">Boleslaw Dawidowicz</a>
 * @version $Revision: 0.1 $
 */
public abstract class InfoTestFilter implements TestFilter
{
   
   private Set<String> keywords = new HashSet<String>();

   private Set<String> names = new HashSet<String>();

   private Set<String> namePatterns = new HashSet<String>();

   public InfoTestFilter()
   {
   }

   public InfoTestFilter(Set<String> keywords, Set<String> names, Set<String> namePatterns)
   {
      if (keywords == null)
      {
         throw new IllegalArgumentException("keywords cannot be null");
      }
      if (names == null)
      {
         throw new IllegalArgumentException("names cannot be null");
      }
      if (namePatterns == null)
      {
         throw new IllegalArgumentException("namePatterns cannot be null");
      }

      this.keywords = keywords;
      this.names = names;
      this.namePatterns = namePatterns;
   }

   public InfoTestFilter addKeyword(String keyword)
   {
      if (keyword == null)
      {
         throw new IllegalArgumentException("keyword cannot be null");
      }

      keywords.add(keyword);

      return this;
   }


   public InfoTestFilter addUnparsedKeywords(String keywordsString)
   {
      if (keywordsString == null)
      {
         throw new IllegalArgumentException("keywordsString cannot be null");
      }
      String[] keywordValues = keywordsString.split(",");
      for (String keyword : keywordValues)
      {
         this.keywords.add(keyword.trim());
      }
      return this;
   }

   public InfoTestFilter addName(String name)
   {
      if (name == null)
      {
         throw new IllegalArgumentException("name cannot be null");
      }
      names.add(name);

      return this;
   }

   public InfoTestFilter addUnparsedNames(String namesString)
   {
      if (namesString == null)
      {
         throw new IllegalArgumentException("namesString cannot be null");
      }
      String[] namesValues = namesString.split(",");
      for (String name : namesValues)
      {
         this.names.add(name.trim());
      }
      return this;
   }

   public InfoTestFilter addNamePattern(String pattern)
   {
      if (pattern == null)
      {
         throw new IllegalArgumentException("pattern cannot be null");
      }
      namePatterns.add(pattern);

      return this;
   }

   public InfoTestFilter addUnparsedNamePatterns(String patternString)
   {
      if (patternString == null)
      {
         throw new IllegalArgumentException("patternString cannot be null");
      }
      String[] patternValues = patternString.split(",");
      for (String pattern : patternValues)
      {
         this.namePatterns.add(pattern.trim());
      }
      return this;
   }

   public abstract boolean include(TestId testId, TestInfo testInfo);

   public boolean include(TestId testId, Map<String, String> parametrization, TestInfo testInfo)
   {
      // At the moment no parametrization match

      return include (testId, testInfo);
   }

   boolean checkName(String name)
   {
      for (String s : names)
      {
         if (s.equals(name))
         {
            return true;
         }
      }
      return false;
   }

   boolean checkKeywords(Set<String> keywords)
   {
      for (String keyword : this.keywords)
      {
         for (String toMatch : keywords)
         {
            if (keyword.equals(toMatch))
            {
               return true;
            }
         }
      }
      return false;
   }

   boolean checkNamePatterns(String name)
   {
      for (String regex : this.namePatterns)
      {
         if (name.matches(regex))
         {
            return true;
         }
      }
      return false;
   }


   public String toString()
   {
      StringBuffer sb = new StringBuffer();
      sb.append("Filter: ")
         .append(getClass().getName())
         .append("; names: ")
         .append(names.toString())
         .append("; keywords: ")
         .append(keywords.toString())
         .append("; namePatterns: ")
         .append(namePatterns.toString())
         .append(";");
      return sb.toString();
   }


   public Set<String> getKeywords()
   {
      return keywords;
   }

   public Set<String> getNames()
   {
      return names;
   }

   public Set<String> getNamePatterns()
   {
      return namePatterns;
   }
}
