/*
 * (C) Copyright 2006-2010 Nuxeo SAS (http://nuxeo.com/) and contributors.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser General Public License
 * (LGPL) version 2.1 which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl.html
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * Contributors:
 *     bstefanescu
 */
package org.nuxeo.connect.update.task;

import java.util.Map;

import org.nuxeo.connect.update.LocalPackage;
import org.nuxeo.connect.update.PackageException;
import org.nuxeo.connect.update.ValidationStatus;

/**
 * A task is implementing the logic of an install or uninstall. if this get
 * wrongs (the run method is throwing an exception) then the rollback method
 * should be invoked to revert all modifications that was done by the run
 * method.
 * 
 * @author <a href="mailto:bs@nuxeo.com">Bogdan Stefanescu</a>
 * 
 */
public interface Task {

    /**
     * Initialize the task. Must be called before executing or validating the
     * task.
     * 
     * The given parameters are the ones filled by the user in the installation
     * wizard. If no parameters are specified an empty or a null map is
     * specified.
     * 
     * @param pkg
     * @param params
     * @param restart
     * @throws PackageException
     */
    void initialize(LocalPackage pkg, boolean restart) throws PackageException;

    /**
     * Validate that the task can be run. Returns a validation state. If the
     * state contain errors the task cannot be run on the running platform. If
     * the state contains warnings the user should be asked if it really want to
     * run the task.
     * 
     * @return
     */
    ValidationStatus validate() throws PackageException;

    /**
     * Run the task. Throws an exception if something goes wrong. At the end of
     * the run the commands log must be written on disk.
     * 
     * @param params the user parameters or null if none.
     * 
     * @throws PackageException
     */
    void run(Map<String, String> params) throws PackageException;

    /**
     * Rollback the work done so far. Should be called if the run method failed.
     * 
     * @throws PackageException
     */
    void rollback() throws PackageException;

    /**
     * The target package
     * 
     * @return
     */
    LocalPackage getPackage();

    /**
     * Whether or not the platform must be restarted after the task is executed.
     * 
     * @return
     */
    boolean isRestartRequired();

    /**
     * Set if restart is required
     * 
     * @param isRestartRequired
     */
    void setRestartRequired(boolean isRestartRequired);

}
