/*
 * Decompiled with CFR 0.152.
 */
package org.nuxeo.common.file;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.Map;
import java.util.regex.Pattern;
import org.apache.commons.collections.map.ReferenceMap;
import org.apache.commons.io.FileCleaningTracker;
import org.apache.commons.io.IOUtils;
import org.nuxeo.common.file.FileCache;

public class LRUFileCache
implements FileCache {
    public static final Pattern SIMPLE_ASCII = Pattern.compile("[-_a-zA-Z0-9]+");
    protected final File dir;
    protected final long maxSize;
    protected final Map<String, LRUFileCacheEntry> cache;
    protected final Map<String, File> files;
    protected long cacheSize;
    protected final LinkedList<String> lru;
    private static final FileCleaningTracker fileCleaningTracker = new FileCleaningTracker();

    public LRUFileCache(File dir, long maxSize) {
        this.dir = dir;
        this.maxSize = maxSize;
        this.cache = new HashMap<String, LRUFileCacheEntry>();
        this.files = new ReferenceMap(0, 2);
        this.lru = new LinkedList();
    }

    @Override
    public long getSize() {
        return this.cacheSize;
    }

    @Override
    public int getNumberOfItems() {
        return this.lru.size();
    }

    @Override
    public File getTempFile() throws IOException {
        File tmp = File.createTempFile("nxbin_", null, this.dir);
        tmp.deleteOnExit();
        return tmp;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public synchronized File putFile(String key, InputStream in) throws IOException {
        try {
            LRUFileCacheEntry entry = this.cache.get(key);
            if (entry != null) {
                File file = entry.file;
                return file;
            }
            File file = this.files.get(key);
            if (file != null) {
                File file2 = this.putFileInCache(key, file);
                return file2;
            }
            file = this.getTempFile();
            try (FileOutputStream out = new FileOutputStream(file);){
                IOUtils.copy((InputStream)in, (OutputStream)out);
            }
            File file3 = this.putFile(key, file);
            return file3;
        }
        finally {
            in.close();
        }
    }

    @Override
    public synchronized File putFile(String key, File file) throws IllegalArgumentException, IOException {
        LRUFileCacheEntry entry = this.cache.get(key);
        if (entry != null) {
            file.delete();
            return entry.file;
        }
        File dest = this.files.get(key);
        if (dest != null) {
            return this.putFileInCache(key, dest);
        }
        this.checkKey(key);
        dest = new File(this.dir, key);
        if (!file.renameTo(dest)) {
            file.delete();
        }
        return this.putFileInCache(key, dest);
    }

    protected File putFileInCache(String key, File file) {
        long size = file.length();
        this.ensureCapacity(size);
        LRUFileCacheEntry entry = new LRUFileCacheEntry();
        entry.size = size;
        entry.file = file;
        this.add(key, entry);
        return file;
    }

    protected void checkKey(String key) throws IllegalArgumentException {
        if (!SIMPLE_ASCII.matcher(key).matches() || ".".equals(key) || "..".equals(key)) {
            throw new IllegalArgumentException("Invalid key: " + key);
        }
    }

    @Override
    public synchronized File getFile(String key) {
        LRUFileCacheEntry entry = this.cache.get(key);
        if (entry != null) {
            this.recordAccess(key);
            return entry.file;
        }
        return this.files.get(key);
    }

    @Override
    public synchronized void clear() {
        for (String key : this.lru) {
            this.remove(key);
        }
        this.lru.clear();
        this.cache.clear();
        this.files.clear();
    }

    protected void recordAccess(String key) {
        this.lru.remove(key);
        this.lru.addFirst(key);
    }

    protected void add(String key, LRUFileCacheEntry entry) {
        this.cache.put(key, entry);
        this.files.put(key, entry.file);
        this.lru.addFirst(key);
        this.cacheSize += entry.size;
    }

    protected void remove(String key) {
        LRUFileCacheEntry entry = this.cache.remove(key);
        this.cacheSize -= entry.size;
        fileCleaningTracker.track(entry.file, (Object)entry.file);
    }

    protected void ensureCapacity(long size) {
        while (this.cacheSize + size > this.maxSize && !this.lru.isEmpty()) {
            this.remove(this.lru.removeLast());
        }
    }

    protected static class LRUFileCacheEntry {
        public File file;
        public long size;

        protected LRUFileCacheEntry() {
        }
    }
}

