/*
 * Decompiled with CFR 0.152.
 */
package org.nuxeo.lib.stream.tools.command;

import java.time.DateTimeException;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.nuxeo.lib.stream.computation.Record;
import org.nuxeo.lib.stream.computation.Watermark;
import org.nuxeo.lib.stream.log.LogLag;
import org.nuxeo.lib.stream.log.LogManager;
import org.nuxeo.lib.stream.log.LogOffset;
import org.nuxeo.lib.stream.log.LogPartition;
import org.nuxeo.lib.stream.log.LogRecord;
import org.nuxeo.lib.stream.log.LogTailer;
import org.nuxeo.lib.stream.tools.command.Command;

public class PositionCommand
extends Command {
    public static final Duration FIRST_READ_TIMEOUT = Duration.ofMillis(1000L);
    public static final Duration READ_TIMEOUT = Duration.ofMillis(100L);
    protected static final String NAME = "position";

    protected static long getTimestampFromDate(String dateIso8601) {
        if (dateIso8601 == null || dateIso8601.isEmpty()) {
            return -1L;
        }
        try {
            Instant instant = Instant.parse(dateIso8601);
            return instant.toEpochMilli();
        }
        catch (DateTimeException e) {
            System.err.println("Failed to read the timeout: " + e.getMessage());
            System.err.println("The timestamp should be in ISO-8601 format, eg. " + Instant.now());
            return -1L;
        }
    }

    @Override
    public String name() {
        return NAME;
    }

    @Override
    public void updateOptions(Options options) {
        options.addOption(Option.builder((String)"l").longOpt("log-name").desc("Log name").required().hasArg().argName("LOG_NAME").build());
        options.addOption(Option.builder((String)"g").longOpt("group").desc("Consumer group").hasArg().argName("GROUP").build());
        options.addOption(Option.builder().longOpt("reset").desc("Resets all committed positions for the group").build());
        options.addOption(Option.builder().longOpt("to-end").desc("Sets the committed positions to the end of partitions for the group").build());
        options.addOption(Option.builder().longOpt("after-date").desc("Sets the committed positions for the group to a specific date. The date used to find the offset depends on the implementation, for Kafka this is the LogAppendTime. The position is set to the earliest offset whose timestamp is greater than or equal to the given date. The date is specified in ISO-8601 format, eg. " + Instant.now() + ". If no record offset is found with an appropriate timestamp then the command fails.").hasArg().argName("DATE").build());
        options.addOption(Option.builder().longOpt("to-watermark").desc("Sets the committed positions for the group to a specific date. The date used to find the offset is contained in a record watermark.  This means that the LOG_NAME is expected to be a computation stream with records with populated watermark. The position is set to the biggest record offset with a watermark date inferior or equals to the given date.\" The date is specified in ISO-8601 format, eg. " + Instant.now() + ". If no record offset is found with an appropriate timestamp then the command fails.").hasArg().argName("DATE").build());
    }

    @Override
    public boolean run(LogManager manager, CommandLine cmd) throws InterruptedException {
        String name = cmd.getOptionValue("log-name");
        String group = cmd.getOptionValue("group", "tools");
        if (cmd.hasOption("after-date")) {
            long timestamp = PositionCommand.getTimestampFromDate(cmd.getOptionValue("after-date"));
            if (timestamp >= 0L) {
                return this.positionAfterDate(manager, group, name, timestamp);
            }
        } else if (cmd.hasOption("to-watermark")) {
            long timestamp = PositionCommand.getTimestampFromDate(cmd.getOptionValue("to-watermark"));
            if (timestamp >= 0L) {
                return this.positionToWatermark(manager, group, name, timestamp);
            }
        } else {
            if (cmd.hasOption("to-end")) {
                return this.toEnd(manager, group, name);
            }
            if (cmd.hasOption("reset")) {
                return this.reset(manager, group, name);
            }
            System.err.println("Invalid option, try 'help position'");
        }
        return false;
    }

    protected boolean toEnd(LogManager manager, String group, String name) {
        LogLag lag = manager.getLag(name, group);
        try (LogTailer tailer = manager.createTailer(group, name);){
            tailer.toEnd();
            tailer.commit();
        }
        System.out.println(String.format("# Moved log %s, group: %s, from: %s to %s", name, group, lag.lower(), lag.upper()));
        return true;
    }

    protected boolean reset(LogManager manager, String group, String name) {
        LogLag lag = manager.getLag(name, group);
        long pos = lag.lower();
        try (LogTailer tailer = manager.createTailer(group, name);){
            tailer.reset();
        }
        System.out.println(String.format("# Reset log %s, group: %s, from: %s to 0", name, group, pos));
        return true;
    }

    protected boolean positionAfterDate(LogManager manager, String group, String name, long timestamp) {
        try (LogTailer tailer = manager.createTailer(group, name);){
            boolean movedOffset = false;
            for (int partition = 0; partition < manager.getAppender(name).size(); ++partition) {
                LogPartition logPartition = new LogPartition(name, partition);
                LogOffset logOffset = tailer.offsetForTimestamp(logPartition, timestamp);
                if (logOffset == null) {
                    System.err.println(String.format("# Could not find an offset for group: %s, partition: %s", group, logPartition));
                    continue;
                }
                tailer.seek(logOffset);
                movedOffset = true;
                System.out.println(String.format("# Set log %s, group: %s, to offset %s", name, group, logOffset.offset()));
            }
            if (movedOffset) {
                tailer.commit();
                boolean bl = true;
                return bl;
            }
        }
        System.err.println("No offset found for the specified date");
        return false;
    }

    protected boolean positionToWatermark(LogManager manager, String group, String name, long timestamp) throws InterruptedException {
        String newGroup = "tools";
        int size = manager.getAppender(name).size();
        ArrayList<LogOffset> offsets = new ArrayList<LogOffset>(size);
        List<LogLag> lags = manager.getLagPerPartition(name, newGroup);
        int partition = 0;
        for (LogLag lag : lags) {
            if (lag.lag() == 0L) {
                offsets.add(null);
            } else {
                try (LogTailer<Record> tailer = manager.createTailer(newGroup, new LogPartition(name, partition));){
                    offsets.add(this.searchWatermarkOffset(tailer, timestamp));
                }
            }
            ++partition;
        }
        if (offsets.stream().noneMatch(Objects::nonNull)) {
            if (LogLag.of(lags).upper() == 0L) {
                System.err.println("No offsets found because log is empty");
                return false;
            }
            System.err.println("Timestamp: " + timestamp + " is earlier as any records, resetting positions");
            return this.reset(manager, group, name);
        }
        try (LogTailer tailer = manager.createTailer(group, name);){
            offsets.stream().filter(Objects::nonNull).forEach(tailer::seek);
            tailer.commit();
            offsets.stream().filter(Objects::nonNull).forEach(offset -> System.out.println("# Moving consumer to: " + offset));
        }
        return true;
    }

    protected LogOffset searchWatermarkOffset(LogTailer<Record> tailer, long timestamp) throws InterruptedException {
        LogOffset lastOffset = null;
        LogRecord<Record> rec = tailer.read(FIRST_READ_TIMEOUT);
        while (rec != null) {
            long recTimestamp = Watermark.ofValue(rec.message().watermark).getTimestamp();
            if (recTimestamp == timestamp) {
                return rec.offset();
            }
            if (recTimestamp > timestamp) {
                return lastOffset;
            }
            if (recTimestamp == 0L) {
                throw new IllegalArgumentException("Cannot find position because Record has empty watermark: " + rec);
            }
            lastOffset = rec.offset();
            rec = tailer.read(READ_TIMEOUT);
        }
        return lastOffset;
    }
}

