/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.indices.recovery;

import java.io.IOException;
import java.util.Arrays;
import java.util.Collections;
import java.util.Map;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.lucene.index.CorruptIndexException;
import org.apache.lucene.store.AlreadyClosedException;
import org.apache.lucene.store.IndexOutput;
import org.apache.lucene.store.RateLimiter;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.ElasticsearchIllegalStateException;
import org.elasticsearch.ExceptionsHelper;
import org.elasticsearch.Version;
import org.elasticsearch.cluster.ClusterService;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.common.Nullable;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.component.AbstractComponent;
import org.elasticsearch.common.inject.Inject;
import org.elasticsearch.common.lucene.Lucene;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.common.util.CancellableThreads;
import org.elasticsearch.common.util.concurrent.AbstractRunnable;
import org.elasticsearch.index.IndexShardMissingException;
import org.elasticsearch.index.engine.RecoveryEngineException;
import org.elasticsearch.index.settings.IndexSettings;
import org.elasticsearch.index.shard.IllegalIndexShardStateException;
import org.elasticsearch.index.shard.IndexShard;
import org.elasticsearch.index.shard.IndexShardClosedException;
import org.elasticsearch.index.shard.IndexShardNotStartedException;
import org.elasticsearch.index.shard.ShardId;
import org.elasticsearch.index.store.Store;
import org.elasticsearch.index.store.StoreFileMetaData;
import org.elasticsearch.index.translog.Translog;
import org.elasticsearch.indices.IndexMissingException;
import org.elasticsearch.indices.IndicesLifecycle;
import org.elasticsearch.indices.recovery.DelayRecoveryException;
import org.elasticsearch.indices.recovery.RecoveriesCollection;
import org.elasticsearch.indices.recovery.RecoveryCleanFilesRequest;
import org.elasticsearch.indices.recovery.RecoveryFailedException;
import org.elasticsearch.indices.recovery.RecoveryFileChunkRequest;
import org.elasticsearch.indices.recovery.RecoveryFilesInfoRequest;
import org.elasticsearch.indices.recovery.RecoveryFinalizeRecoveryRequest;
import org.elasticsearch.indices.recovery.RecoveryPrepareForTranslogOperationsRequest;
import org.elasticsearch.indices.recovery.RecoveryResponse;
import org.elasticsearch.indices.recovery.RecoverySettings;
import org.elasticsearch.indices.recovery.RecoveryState;
import org.elasticsearch.indices.recovery.RecoveryStatus;
import org.elasticsearch.indices.recovery.RecoveryTranslogOperationsRequest;
import org.elasticsearch.indices.recovery.StartRecoveryRequest;
import org.elasticsearch.threadpool.ThreadPool;
import org.elasticsearch.transport.BaseTransportRequestHandler;
import org.elasticsearch.transport.ConnectTransportException;
import org.elasticsearch.transport.FutureTransportResponseHandler;
import org.elasticsearch.transport.TransportChannel;
import org.elasticsearch.transport.TransportResponse;
import org.elasticsearch.transport.TransportService;

public class RecoveryTarget
extends AbstractComponent {
    private final ThreadPool threadPool;
    private final TransportService transportService;
    private final RecoverySettings recoverySettings;
    private final ClusterService clusterService;
    private final RecoveriesCollection onGoingRecoveries;

    @Inject
    public RecoveryTarget(Settings settings, ThreadPool threadPool, TransportService transportService, IndicesLifecycle indicesLifecycle, RecoverySettings recoverySettings, ClusterService clusterService) {
        super(settings);
        this.threadPool = threadPool;
        this.transportService = transportService;
        this.recoverySettings = recoverySettings;
        this.clusterService = clusterService;
        this.onGoingRecoveries = new RecoveriesCollection(this.logger, threadPool);
        transportService.registerHandler("internal:index/shard/recovery/filesInfo", new FilesInfoRequestHandler());
        transportService.registerHandler("internal:index/shard/recovery/file_chunk", new FileChunkTransportRequestHandler());
        transportService.registerHandler("internal:index/shard/recovery/clean_files", new CleanFilesRequestHandler());
        transportService.registerHandler("internal:index/shard/recovery/prepare_translog", new PrepareForTranslogOperationsRequestHandler());
        transportService.registerHandler("internal:index/shard/recovery/translog_ops", new TranslogOperationsRequestHandler());
        transportService.registerHandler("internal:index/shard/recovery/finalize", new FinalizeRecoveryRequestHandler());
        indicesLifecycle.addListener(new IndicesLifecycle.Listener(){

            @Override
            public void beforeIndexShardClosed(ShardId shardId, @Nullable IndexShard indexShard, @IndexSettings Settings indexSettings) {
                if (indexShard != null) {
                    RecoveryTarget.this.onGoingRecoveries.cancelRecoveriesForShard(shardId, "shard closed");
                }
            }
        });
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public RecoveryState recoveryState(IndexShard indexShard) {
        try (RecoveriesCollection.StatusRef statusRef = this.onGoingRecoveries.findRecoveryByShard(indexShard);){
            if (statusRef == null) {
                RecoveryState recoveryState = null;
                return recoveryState;
            }
            RecoveryStatus recoveryStatus = statusRef.status();
            RecoveryState recoveryState = recoveryStatus.state();
            return recoveryState;
        }
        catch (Exception e) {
            throw new ElasticsearchException("error while getting recovery state", e);
        }
    }

    public void startRecovery(IndexShard indexShard, RecoveryState.Type recoveryType, DiscoveryNode sourceNode, RecoveryListener listener) {
        try {
            indexShard.recovering("from " + sourceNode, recoveryType, sourceNode);
        }
        catch (IllegalIndexShardStateException e) {
            this.logger.debug("{} ignore recovery. already in recovering process, {}", indexShard.shardId(), e.getMessage());
            return;
        }
        long recoveryId = this.onGoingRecoveries.startRecovery(indexShard, sourceNode, listener, this.recoverySettings.activityTimeout());
        this.threadPool.generic().execute(new RecoveryRunner(recoveryId));
    }

    protected void retryRecovery(RecoveryStatus recoveryStatus, String reason, TimeValue retryAfter, StartRecoveryRequest currentRequest) {
        this.logger.trace("will retrying recovery with id [{}] in [{}] (reason [{}])", recoveryStatus.recoveryId(), retryAfter, reason);
        try {
            recoveryStatus.resetRecovery();
        }
        catch (IOException e) {
            this.onGoingRecoveries.failRecovery(recoveryStatus.recoveryId(), new RecoveryFailedException(currentRequest, (Throwable)e), true);
        }
        this.threadPool.schedule(retryAfter, "generic", new RecoveryRunner(recoveryStatus.recoveryId()));
    }

    Map<String, StoreFileMetaData> existingFiles(DiscoveryNode sourceNode, Store store) throws IOException {
        Version sourceNodeVersion = sourceNode.version();
        if (sourceNodeVersion.onOrAfter(Version.V_1_4_0)) {
            return store.getMetadataOrEmpty().asMap();
        }
        this.logger.debug("Force full recovery source node version {}", sourceNodeVersion);
        return Collections.EMPTY_MAP;
    }

    private void doRecovery(RecoveryStatus recoveryStatus) {
        Map<String, StoreFileMetaData> existingFiles;
        assert (recoveryStatus.sourceNode() != null) : "can't do a recovery without a source node";
        this.logger.trace("collecting local files for {}", recoveryStatus);
        try {
            existingFiles = this.existingFiles(recoveryStatus.sourceNode(), recoveryStatus.store());
        }
        catch (Exception e) {
            this.logger.debug("error while listing local files", e, new Object[0]);
            this.onGoingRecoveries.failRecovery(recoveryStatus.recoveryId(), new RecoveryFailedException(recoveryStatus.state(), "failed to list local files", (Throwable)e), true);
            return;
        }
        Version sourceNodeVersion = recoveryStatus.sourceNode().version();
        if (sourceNodeVersion.before(Version.V_1_3_2) && this.recoverySettings.compress()) {
            throw new ElasticsearchIllegalStateException("Can't recovery from node " + recoveryStatus.sourceNode() + " with [" + "indices.recovery.compress" + " : true] due to compression bugs -  see issue #7210 for details");
        }
        final StartRecoveryRequest request = new StartRecoveryRequest(recoveryStatus.shardId(), recoveryStatus.sourceNode(), this.clusterService.localNode(), false, existingFiles, recoveryStatus.state().getType(), recoveryStatus.recoveryId());
        final AtomicReference responseHolder = new AtomicReference();
        try {
            this.logger.trace("[{}][{}] starting recovery from {}", request.shardId().index().name(), request.shardId().id(), request.sourceNode());
            recoveryStatus.indexShard().prepareForIndexRecovery();
            recoveryStatus.CancellableThreads().execute(new CancellableThreads.Interruptable(){

                @Override
                public void run() throws InterruptedException {
                    responseHolder.set(RecoveryTarget.this.transportService.submitRequest(request.sourceNode(), "internal:index/shard/recovery/start_recovery", request, new FutureTransportResponseHandler<RecoveryResponse>(){

                        @Override
                        public RecoveryResponse newInstance() {
                            return new RecoveryResponse();
                        }
                    }).txGet());
                }
            });
            RecoveryResponse recoveryResponse = (RecoveryResponse)responseHolder.get();
            assert (responseHolder != null);
            TimeValue recoveryTime = new TimeValue(recoveryStatus.state().getTimer().time());
            this.onGoingRecoveries.markRecoveryAsDone(recoveryStatus.recoveryId());
            if (this.logger.isTraceEnabled()) {
                StringBuilder sb = new StringBuilder();
                sb.append('[').append(request.shardId().index().name()).append(']').append('[').append(request.shardId().id()).append("] ");
                sb.append("recovery completed from ").append(request.sourceNode()).append(", took[").append(recoveryTime).append("]\n");
                sb.append("   phase1: recovered_files [").append(recoveryResponse.phase1FileNames.size()).append("]").append(" with total_size of [").append(new ByteSizeValue(recoveryResponse.phase1TotalSize)).append("]").append(", took [").append(TimeValue.timeValueMillis(recoveryResponse.phase1Time)).append("], throttling_wait [").append(TimeValue.timeValueMillis(recoveryResponse.phase1ThrottlingWaitTime)).append(']').append("\n");
                sb.append("         : reusing_files   [").append(recoveryResponse.phase1ExistingFileNames.size()).append("] with total_size of [").append(new ByteSizeValue(recoveryResponse.phase1ExistingTotalSize)).append("]\n");
                sb.append("   phase2: start took [").append(TimeValue.timeValueMillis(recoveryResponse.startTime)).append("]\n");
                sb.append("         : recovered [").append(recoveryResponse.phase2Operations).append("]").append(" transaction log operations").append(", took [").append(TimeValue.timeValueMillis(recoveryResponse.phase2Time)).append("]").append("\n");
                sb.append("   phase3: recovered [").append(recoveryResponse.phase3Operations).append("]").append(" transaction log operations").append(", took [").append(TimeValue.timeValueMillis(recoveryResponse.phase3Time)).append("]");
                this.logger.trace(sb.toString(), new Object[0]);
            } else {
                this.logger.debug("{} recovery done from [{}], took [{}]", request.shardId(), recoveryStatus.sourceNode(), recoveryTime);
            }
        }
        catch (CancellableThreads.ExecutionCancelledException e) {
            this.logger.trace("recovery cancelled", e, new Object[0]);
        }
        catch (Throwable e) {
            Throwable cause;
            if (this.logger.isTraceEnabled()) {
                this.logger.trace("[{}][{}] Got exception on recovery", e, request.shardId().index().name(), request.shardId().id());
            }
            if ((cause = ExceptionsHelper.unwrapCause(e)) instanceof RecoveryEngineException) {
                cause = cause.getCause();
            }
            if ((cause = ExceptionsHelper.unwrapCause(cause)) instanceof RecoveryEngineException) {
                cause = cause.getCause();
            }
            if (cause instanceof IndexShardNotStartedException || cause instanceof IndexMissingException || cause instanceof IndexShardMissingException) {
                this.retryRecovery(recoveryStatus, "remote shard not ready", this.recoverySettings.retryDelayStateSync(), request);
                return;
            }
            if (cause instanceof DelayRecoveryException) {
                this.retryRecovery(recoveryStatus, cause.getMessage(), this.recoverySettings.retryDelayStateSync(), request);
                return;
            }
            if (cause instanceof ConnectTransportException) {
                this.logger.debug("delaying recovery of {} for [{}] due to networking error [{}]", recoveryStatus.shardId(), this.recoverySettings.retryDelayNetwork(), cause.getMessage());
                this.retryRecovery(recoveryStatus, cause.getMessage(), this.recoverySettings.retryDelayNetwork(), request);
                return;
            }
            if (cause instanceof IndexShardClosedException) {
                this.onGoingRecoveries.failRecovery(recoveryStatus.recoveryId(), new RecoveryFailedException(request, "source shard is closed", cause), false);
                return;
            }
            if (cause instanceof AlreadyClosedException) {
                this.onGoingRecoveries.failRecovery(recoveryStatus.recoveryId(), new RecoveryFailedException(request, "source shard is closed", cause), false);
                return;
            }
            this.onGoingRecoveries.failRecovery(recoveryStatus.recoveryId(), new RecoveryFailedException(request, e), true);
        }
    }

    class RecoveryRunner
    extends AbstractRunnable {
        final long recoveryId;

        RecoveryRunner(long recoveryId) {
            this.recoveryId = recoveryId;
        }

        @Override
        public void onFailure(Throwable t) {
            try (RecoveriesCollection.StatusRef statusRef = RecoveryTarget.this.onGoingRecoveries.getStatus(this.recoveryId);){
                if (statusRef != null) {
                    RecoveryTarget.this.logger.error("unexpected error during recovery [{}], failing shard", t, this.recoveryId);
                    RecoveryTarget.this.onGoingRecoveries.failRecovery(this.recoveryId, new RecoveryFailedException(statusRef.status().state(), "unexpected error", t), true);
                } else {
                    RecoveryTarget.this.logger.debug("unexpected error during recovery, but recovery id [{}] is finished", t, this.recoveryId);
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void doRun() {
            RecoveriesCollection.StatusRef statusRef = RecoveryTarget.this.onGoingRecoveries.getStatus(this.recoveryId);
            if (statusRef == null) {
                RecoveryTarget.this.logger.trace("not running recovery with id [{}] - can't find it (probably finished)", this.recoveryId);
                return;
            }
            try {
                RecoveryTarget.this.doRecovery(statusRef.status());
            }
            finally {
                statusRef.close();
            }
        }
    }

    class FileChunkTransportRequestHandler
    extends BaseTransportRequestHandler<RecoveryFileChunkRequest> {
        final AtomicLong bytesSinceLastPause = new AtomicLong();

        FileChunkTransportRequestHandler() {
        }

        @Override
        public RecoveryFileChunkRequest newInstance() {
            return new RecoveryFileChunkRequest();
        }

        @Override
        public String executor() {
            return "generic";
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void messageReceived(RecoveryFileChunkRequest request, TransportChannel channel) throws Exception {
            block20: {
                try (RecoveriesCollection.StatusRef statusRef = RecoveryTarget.this.onGoingRecoveries.getStatusSafe(request.recoveryId(), request.shardId());){
                    long bytes;
                    RateLimiter rl;
                    RecoveryStatus recoveryStatus = statusRef.status();
                    Store store = recoveryStatus.store();
                    recoveryStatus.state().getTranslog().totalOperations(request.totalTranslogOps());
                    RecoveryState.Index indexState = recoveryStatus.state().getIndex();
                    if (request.sourceThrottleTimeInNanos() != -1L) {
                        indexState.addSourceThrottling(request.sourceThrottleTimeInNanos());
                    }
                    IndexOutput indexOutput = request.position() == 0L ? recoveryStatus.openAndPutIndexOutput(request.name(), request.metadata(), store) : recoveryStatus.getOpenIndexOutput(request.name());
                    BytesReference content = request.content();
                    if (!content.hasArray()) {
                        content = content.toBytesArray();
                    }
                    if ((rl = RecoveryTarget.this.recoverySettings.rateLimiter()) != null && (bytes = this.bytesSinceLastPause.addAndGet(content.length())) > rl.getMinPauseCheckBytes()) {
                        this.bytesSinceLastPause.addAndGet(-bytes);
                        long throttleTimeInNanos = rl.pause(bytes);
                        indexState.addTargetThrottling(throttleTimeInNanos);
                        recoveryStatus.indexShard().recoveryStats().addThrottleTime(throttleTimeInNanos);
                    }
                    indexOutput.writeBytes(content.array(), content.arrayOffset(), content.length());
                    indexState.addRecoveredBytesToFile(request.name(), content.length());
                    if (indexOutput.getFilePointer() < request.length() && !request.lastChunk()) break block20;
                    try {
                        Store.verify(indexOutput);
                    }
                    finally {
                        indexOutput.close();
                    }
                    recoveryStatus.legacyChecksums().add(request.metadata());
                    String temporaryFileName = recoveryStatus.getTempNameForFile(request.name());
                    assert (Arrays.asList(store.directory().listAll()).contains(temporaryFileName));
                    store.directory().sync(Collections.singleton(temporaryFileName));
                    IndexOutput remove = recoveryStatus.removeOpenIndexOutputs(request.name());
                    assert (remove == null || remove == indexOutput);
                }
            }
            channel.sendResponse(TransportResponse.Empty.INSTANCE);
        }
    }

    class CleanFilesRequestHandler
    extends BaseTransportRequestHandler<RecoveryCleanFilesRequest> {
        CleanFilesRequestHandler() {
        }

        @Override
        public RecoveryCleanFilesRequest newInstance() {
            return new RecoveryCleanFilesRequest();
        }

        @Override
        public String executor() {
            return "generic";
        }

        @Override
        public void messageReceived(RecoveryCleanFilesRequest request, TransportChannel channel) throws Exception {
            try (RecoveriesCollection.StatusRef statusRef = RecoveryTarget.this.onGoingRecoveries.getStatusSafe(request.recoveryId(), request.shardId());){
                RecoveryStatus recoveryStatus = statusRef.status();
                recoveryStatus.state().getTranslog().totalOperations(request.totalTranslogOps());
                recoveryStatus.renameAllTempFiles();
                Store store = recoveryStatus.store();
                recoveryStatus.legacyChecksums().write(store);
                Store.MetadataSnapshot sourceMetaData = request.sourceMetaSnapshot();
                if (sourceMetaData == null) {
                    if (recoveryStatus.state().getSourceNode().getVersion().onOrAfter(Version.V_1_5_0)) {
                        throw new ElasticsearchIllegalStateException("Source node version is on or after 1.5 but has not metadata snapshot: " + recoveryStatus.state().getSourceNode());
                    }
                    if (request.legacySnapshotFiles() == null) {
                        throw new ElasticsearchIllegalStateException("Legacy snapshot files are null");
                    }
                    try {
                        store.cleanup("recovery CleanFilesRequestHandler", request.legacySnapshotFiles());
                    }
                    catch (Exception ex) {
                        throw new RecoveryFailedException(recoveryStatus.state(), "failed to clean after recovery", (Throwable)ex);
                    }
                }
                try {
                    store.cleanupAndVerify("recovery CleanFilesRequestHandler", sourceMetaData);
                }
                catch (CorruptIndexException ex) {
                    try {
                        Lucene.cleanLuceneIndex(store.directory());
                    }
                    catch (Throwable e) {
                        RecoveryTarget.this.logger.debug("Failed to clean lucene index", e, new Object[0]);
                        ex.addSuppressed(e);
                    }
                    throw new RecoveryFailedException(recoveryStatus.state(), "failed to clean after recovery", (Throwable)ex);
                }
                catch (Exception ex) {
                    throw new RecoveryFailedException(recoveryStatus.state(), "failed to clean after recovery", (Throwable)ex);
                }
                channel.sendResponse(TransportResponse.Empty.INSTANCE);
            }
        }
    }

    class FilesInfoRequestHandler
    extends BaseTransportRequestHandler<RecoveryFilesInfoRequest> {
        FilesInfoRequestHandler() {
        }

        @Override
        public RecoveryFilesInfoRequest newInstance() {
            return new RecoveryFilesInfoRequest();
        }

        @Override
        public String executor() {
            return "generic";
        }

        @Override
        public void messageReceived(RecoveryFilesInfoRequest request, TransportChannel channel) throws Exception {
            try (RecoveriesCollection.StatusRef statusRef = RecoveryTarget.this.onGoingRecoveries.getStatusSafe(request.recoveryId(), request.shardId());){
                int i;
                RecoveryStatus recoveryStatus = statusRef.status();
                RecoveryState.Index index = recoveryStatus.state().getIndex();
                for (i = 0; i < request.phase1ExistingFileNames.size(); ++i) {
                    index.addFileDetail(request.phase1ExistingFileNames.get(i), request.phase1ExistingFileSizes.get(i), true);
                }
                for (i = 0; i < request.phase1FileNames.size(); ++i) {
                    index.addFileDetail(request.phase1FileNames.get(i), request.phase1FileSizes.get(i), false);
                }
                recoveryStatus.state().getTranslog().totalOperations(request.totalTranslogOps);
                recoveryStatus.state().getTranslog().totalOperationsOnStart(request.totalTranslogOps);
                channel.sendResponse(TransportResponse.Empty.INSTANCE);
            }
        }
    }

    class TranslogOperationsRequestHandler
    extends BaseTransportRequestHandler<RecoveryTranslogOperationsRequest> {
        TranslogOperationsRequestHandler() {
        }

        @Override
        public RecoveryTranslogOperationsRequest newInstance() {
            return new RecoveryTranslogOperationsRequest();
        }

        @Override
        public String executor() {
            return "generic";
        }

        @Override
        public void messageReceived(RecoveryTranslogOperationsRequest request, TransportChannel channel) throws Exception {
            try (RecoveriesCollection.StatusRef statusRef = RecoveryTarget.this.onGoingRecoveries.getStatusSafe(request.recoveryId(), request.shardId());){
                RecoveryStatus recoveryStatus = statusRef.status();
                RecoveryState.Translog translog = recoveryStatus.state().getTranslog();
                translog.totalOperations(request.totalTranslogOps());
                for (Translog.Operation operation : request.operations()) {
                    recoveryStatus.indexShard().performRecoveryOperation(operation);
                    translog.incrementRecoveredOperations();
                }
            }
            channel.sendResponse(TransportResponse.Empty.INSTANCE);
        }
    }

    class FinalizeRecoveryRequestHandler
    extends BaseTransportRequestHandler<RecoveryFinalizeRecoveryRequest> {
        FinalizeRecoveryRequestHandler() {
        }

        @Override
        public RecoveryFinalizeRecoveryRequest newInstance() {
            return new RecoveryFinalizeRecoveryRequest();
        }

        @Override
        public String executor() {
            return "generic";
        }

        @Override
        public void messageReceived(RecoveryFinalizeRecoveryRequest request, TransportChannel channel) throws Exception {
            try (RecoveriesCollection.StatusRef statusRef = RecoveryTarget.this.onGoingRecoveries.getStatusSafe(request.recoveryId(), request.shardId());){
                RecoveryStatus recoveryStatus = statusRef.status();
                recoveryStatus.indexShard().finalizeRecovery();
            }
            channel.sendResponse(TransportResponse.Empty.INSTANCE);
        }
    }

    class PrepareForTranslogOperationsRequestHandler
    extends BaseTransportRequestHandler<RecoveryPrepareForTranslogOperationsRequest> {
        PrepareForTranslogOperationsRequestHandler() {
        }

        @Override
        public RecoveryPrepareForTranslogOperationsRequest newInstance() {
            return new RecoveryPrepareForTranslogOperationsRequest();
        }

        @Override
        public String executor() {
            return "generic";
        }

        @Override
        public void messageReceived(RecoveryPrepareForTranslogOperationsRequest request, TransportChannel channel) throws Exception {
            try (RecoveriesCollection.StatusRef statusRef = RecoveryTarget.this.onGoingRecoveries.getStatusSafe(request.recoveryId(), request.shardId());){
                RecoveryStatus recoveryStatus = statusRef.status();
                recoveryStatus.state().getTranslog().totalOperations(request.totalTranslogOps());
                recoveryStatus.indexShard().prepareForTranslogRecovery();
            }
            channel.sendResponse(TransportResponse.Empty.INSTANCE);
        }
    }

    public static interface RecoveryListener {
        public void onRecoveryDone(RecoveryState var1);

        public void onRecoveryFailure(RecoveryState var1, RecoveryFailedException var2, boolean var3);
    }

    public static class Actions {
        public static final String FILES_INFO = "internal:index/shard/recovery/filesInfo";
        public static final String FILE_CHUNK = "internal:index/shard/recovery/file_chunk";
        public static final String CLEAN_FILES = "internal:index/shard/recovery/clean_files";
        public static final String TRANSLOG_OPS = "internal:index/shard/recovery/translog_ops";
        public static final String PREPARE_TRANSLOG = "internal:index/shard/recovery/prepare_translog";
        public static final String FINALIZE = "internal:index/shard/recovery/finalize";
    }
}

