/*
 * Decompiled with CFR 0.152.
 */
package org.nuxeo.common.codec;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.charset.Charset;
import java.security.GeneralSecurityException;
import java.security.InvalidKeyException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.Security;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.SecretKey;
import javax.crypto.spec.SecretKeySpec;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public class Crypto {
    protected static final Pattern CRYPTO_PATTERN = Pattern.compile("\\{\\$(?<algo>.*)\\$(?<value>.+)\\}");
    private static final Log log = LogFactory.getLog(Crypto.class);
    public static final String AES = "AES";
    public static final String AES_ECB_PKCS5PADDING = "AES/ECB/PKCS5Padding";
    public static final String DES = "DES";
    public static final String DES_ECB_PKCS5PADDING = "DES/ECB/PKCS5Padding";
    public static final String[] IMPLEMENTED_ALGOS = new String[]{"AES", "DES", "AES/ECB/PKCS5Padding", "DES/ECB/PKCS5Padding"};
    public static final String DEFAULT_ALGO = "AES/ECB/PKCS5Padding";
    private static final String SHA1 = "SHA-1";
    private final byte[] secretKey;
    private final Map<String, SecretKey> secretKeys = new HashMap<String, SecretKey>();
    private boolean initialized = true;
    private final byte[] digest;

    public Crypto(byte[] secretKey) {
        this.secretKey = secretKey;
        this.digest = this.getSHA1DigestOrEmpty(secretKey);
        if (this.digest.length == 0) {
            this.clear();
        }
    }

    public Crypto(Map<String, SecretKey> secretKeys) {
        this(secretKeys, Crypto.class.getName().toCharArray());
    }

    public Crypto(Map<String, SecretKey> secretKeys, char[] digest) {
        this.secretKey = new byte[0];
        this.digest = this.getSHA1DigestOrEmpty(Crypto.getBytes(digest));
        this.secretKeys.putAll(secretKeys);
        if (this.digest.length == 0) {
            this.clear();
        }
    }

    public Crypto(String keystorePath, char[] keystorePass, String keyAlias, char[] keyPass) throws GeneralSecurityException, IOException {
        this(Crypto.getKeysFromKeyStore(keystorePath, keystorePass, keyAlias, keyPass), keystorePass);
    }

    protected SecretKey getSecretKey(String algorithm, byte[] key) throws NoSuchAlgorithmException {
        if (!this.initialized) {
            throw new RuntimeException("The Crypto object has been cleared.");
        }
        if ("AES/ECB/PKCS5Padding".equals(algorithm)) {
            algorithm = AES;
        } else if (DES_ECB_PKCS5PADDING.equals(algorithm)) {
            algorithm = DES;
        }
        if (!this.secretKeys.containsKey(algorithm)) {
            if (this.secretKey.length == 0) {
                throw new NoSuchAlgorithmException("Unsupported algorithm: " + algorithm);
            }
            if (AES.equals(algorithm)) {
                key = Arrays.copyOf(this.getSHA1Digest(key), 16);
                this.secretKeys.put(AES, new SecretKeySpec(key, AES));
            } else if (DES.equals(algorithm)) {
                key = Arrays.copyOf(this.getSHA1Digest(key), 8);
                this.secretKeys.put(DES, new SecretKeySpec(key, DES));
            } else {
                throw new NoSuchAlgorithmException("Unsupported algorithm: " + algorithm);
            }
        }
        return this.secretKeys.get(algorithm);
    }

    public byte[] getSHA1Digest(byte[] key) throws NoSuchAlgorithmException {
        MessageDigest sha = MessageDigest.getInstance(SHA1);
        return sha.digest(key);
    }

    public byte[] getSHA1DigestOrEmpty(byte[] bytes) {
        byte[] aDigest = new byte[]{};
        try {
            aDigest = this.getSHA1Digest(bytes);
        }
        catch (NoSuchAlgorithmException e) {
            log.error(e);
        }
        return aDigest;
    }

    public String encrypt(byte[] bytesToEncrypt) throws GeneralSecurityException {
        return this.encrypt(null, bytesToEncrypt);
    }

    public String encrypt(String algorithm, byte[] bytesToEncrypt) throws GeneralSecurityException {
        String encryptedAlgo;
        if (StringUtils.isBlank(algorithm)) {
            algorithm = "AES/ECB/PKCS5Padding";
            encryptedAlgo = "";
        } else {
            encryptedAlgo = Base64.encodeBase64String(algorithm.getBytes());
        }
        Cipher cipher = Cipher.getInstance(algorithm);
        cipher.init(1, this.getSecretKey(algorithm, this.secretKey));
        String encryptedString = Base64.encodeBase64String(cipher.doFinal(bytesToEncrypt));
        return String.format("{$%s$%s}", encryptedAlgo, encryptedString);
    }

    public byte[] decrypt(String strToDecrypt) {
        Matcher matcher = CRYPTO_PATTERN.matcher(strToDecrypt);
        if (!matcher.matches()) {
            return strToDecrypt.getBytes();
        }
        try {
            String algorithm = new String(Base64.decodeBase64(matcher.group("algo")));
            if (StringUtils.isBlank(algorithm)) {
                algorithm = "AES/ECB/PKCS5Padding";
            }
            Cipher decipher = Cipher.getInstance(algorithm);
            decipher.init(2, this.getSecretKey(algorithm, this.secretKey));
            return decipher.doFinal(Base64.decodeBase64(matcher.group("value")));
        }
        catch (NoSuchAlgorithmException | NoSuchPaddingException e) {
            log.trace("Available algorithms: " + Security.getAlgorithms("Cipher"));
            log.trace("Available security providers: " + Arrays.asList(Security.getProviders()));
            log.debug(e, e);
        }
        catch (IllegalArgumentException | InvalidKeyException | BadPaddingException | IllegalBlockSizeException e) {
            log.debug(e, e);
        }
        return strToDecrypt.getBytes();
    }

    public void clear() {
        Arrays.fill(this.secretKey, (byte)0);
        Arrays.fill(this.digest, (byte)0);
        this.secretKeys.clear();
        this.initialized = false;
    }

    public boolean verifyKey(byte[] candidateDigest) {
        boolean success = Arrays.equals(this.getSHA1DigestOrEmpty(candidateDigest), this.digest);
        if (!success) {
            this.clear();
        }
        return success;
    }

    public boolean verifyKey(char[] candidateDigest) {
        return this.verifyKey(Crypto.getBytes(candidateDigest));
    }

    public static byte[] getBytes(char[] chars) {
        CharBuffer charBuffer = CharBuffer.wrap(chars);
        ByteBuffer byteBuffer = Charset.defaultCharset().encode(charBuffer);
        return Arrays.copyOfRange(byteBuffer.array(), 0, byteBuffer.limit());
    }

    public static char[] getChars(byte[] bytes) {
        ByteBuffer byteBuffer = ByteBuffer.wrap(bytes);
        CharBuffer charBuffer = Charset.defaultCharset().decode(byteBuffer);
        return Arrays.copyOfRange(charBuffer.array(), 0, charBuffer.limit());
    }

    public static boolean isEncrypted(String value) {
        return value != null && CRYPTO_PATTERN.matcher(value).matches();
    }

    public static Map<String, SecretKey> getKeysFromKeyStore(String keystorePath, char[] keystorePass, String keyAlias, char[] keyPass) throws GeneralSecurityException, IOException {
        KeyStore keystore = KeyStore.getInstance("JCEKS");
        try (FileInputStream keystoreStream = new FileInputStream(keystorePath);){
            keystore.load(keystoreStream, keystorePass);
        }
        HashMap<String, SecretKey> secretKeys = new HashMap<String, SecretKey>();
        for (String algo : IMPLEMENTED_ALGOS) {
            if (!keystore.containsAlias(keyAlias + algo)) continue;
            SecretKey key = (SecretKey)keystore.getKey(keyAlias + algo, keyPass);
            secretKeys.put(algo, key);
        }
        if (secretKeys.isEmpty()) {
            throw new KeyStoreException(String.format("No alias \"%s<algo>\" found in %s", keyAlias, keystorePath));
        }
        return secretKeys;
    }

    public static void setKeyInKeyStore(String keystorePath, char[] keystorePass, String keyAlias, char[] keyPass, SecretKey key) throws GeneralSecurityException, IOException {
        KeyStore keystore = KeyStore.getInstance("JCEKS");
        if (!new File(keystorePath).exists()) {
            log.info("Creating a new JCEKS keystore at " + keystorePath);
            keystore.load(null);
        } else {
            try (FileInputStream keystoreStream = new FileInputStream(keystorePath);){
                keystore.load(keystoreStream, keystorePass);
            }
        }
        KeyStore.SecretKeyEntry keyStoreEntry = new KeyStore.SecretKeyEntry(key);
        KeyStore.PasswordProtection keyPassword = new KeyStore.PasswordProtection(keyPass);
        keystore.setEntry(keyAlias, keyStoreEntry, keyPassword);
        try (FileOutputStream keystoreStream = new FileOutputStream(keystorePath);){
            keystore.store(keystoreStream, keystorePass);
        }
    }

    public static final class NoOp
    extends Crypto {
        public static final Crypto NO_OP = new NoOp();

        private NoOp() {
            super(new byte[0]);
        }

        @Override
        public String encrypt(String algorithm, byte[] bytesToEncrypt) {
            return null;
        }

        @Override
        public byte[] decrypt(String strToDecrypt) {
            return strToDecrypt.getBytes();
        }

        @Override
        public void clear() {
        }
    }
}

