/*
 * (C) Copyright 2020 Nuxeo (http://nuxeo.com/) and others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Contributors:
 *     Kevin Leturc <kleturc@nuxeo.com>
 */

package org.nuxeo.launcher.config;

import static java.nio.charset.StandardCharsets.UTF_8;
import static java.nio.file.StandardCopyOption.COPY_ATTRIBUTES;
import static java.nio.file.StandardCopyOption.REPLACE_EXISTING;
import static java.nio.file.StandardOpenOption.APPEND;
import static java.nio.file.StandardOpenOption.CREATE;
import static java.nio.file.StandardOpenOption.WRITE;
import static org.apache.commons.lang3.StringUtils.defaultIfBlank;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.nuxeo.common.function.ThrowableConsumer.asConsumer;
import static org.nuxeo.launcher.config.ConfigurationConstants.FILE_NUXEO_DEFAULTS;
import static org.nuxeo.launcher.config.ConfigurationConstants.PARAM_FORCE_GENERATION;
import static org.nuxeo.launcher.config.ConfigurationConstants.PARAM_TEMPLATES_FREEMARKER_EXTENSIONS;
import static org.nuxeo.launcher.config.ConfigurationConstants.PARAM_TEMPLATES_NAME;
import static org.nuxeo.launcher.config.ConfigurationConstants.PARAM_TEMPLATES_PARSING_EXTENSIONS;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.nio.file.Files;
import java.nio.file.Path;
import java.security.DigestInputStream;
import java.security.DigestOutputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.Properties;
import java.util.TreeSet;
import java.util.function.Function;
import java.util.stream.Collectors;

import org.apache.commons.codec.binary.Hex;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.nuxeo.common.utils.TextTemplate;

import freemarker.core.ParseException;
import freemarker.template.TemplateException;

/**
 * Class used to read/write the {@code nuxeo.conf} file.
 *
 * @since 11.5
 */
public class ConfigurationMarshaller {

    private static final Logger log = LogManager.getLogger(ConfigurationMarshaller.class);

    protected static final String BOUNDARY_BEGIN = "### BEGIN - DO NOT EDIT BETWEEN BEGIN AND END ###";

    protected static final String BOUNDARY_END = "### END - DO NOT EDIT BETWEEN BEGIN AND END ###";

    protected static final String PROTECTION_COMMENT = "## DO NOT EDIT THIS FILE";

    protected static final String NEW_FILES = "files.list";

    protected static final List<String> NUXEO_SYSTEM_PROPERTIES = List.of("nuxeo.conf", "nuxeo.home", "log.id");

    protected final Properties systemProperties;

    public ConfigurationMarshaller(Properties systemProperties) {
        this.systemProperties = systemProperties;
    }

    public void persistNuxeoConf(ConfigurationHolder configHolder) throws ConfigurationException {
        NuxeoConfContent content = readConfiguration(configHolder.getNuxeoConfPath());
        editUserConf(content, configHolder);
        editConfiguratorConf(content, configHolder);
        writeConfiguration(content, configHolder);
    }

    public Map<String, String> persistNuxeoDefaults(Path nuxeoDefaultsPath, Map<String, String> newParametersToSave)
            throws ConfigurationException {
        NuxeoConfContent content = readConfiguration(nuxeoDefaultsPath);
        if (content.lines.stream().anyMatch(l -> l.startsWith(PROTECTION_COMMENT))) {
            throw new ConfigurationException("The template states in its header that it must not be modified.");
        }
        // compute old values - HashMap on purpose to allow null values
        var oldValues = new HashMap<String, String>();
        for (var key : newParametersToSave.keySet()) {
            oldValues.put(key, content.properties.get(key));
        }
        // comment parameters to unset & edit parameters
        ListIterator<String> it = content.lines.listIterator();
        while (it.hasNext()) {
            String line = it.next();
            int equalIdx = line.indexOf("=");
            if (equalIdx > 0 && !line.startsWith("#")) {
                String key = line.substring(0, equalIdx);
                if (newParametersToSave.containsKey(key)) {
                    if (newParametersToSave.get(key) == null) {
                        it.set('#' + line);
                    } else {
                        it.set(key + '=' + newParametersToSave.get(key));
                    }
                }
            }
        }
        // append new parameters
        for (var key : newParametersToSave.keySet()) {
            if (!oldValues.containsKey(key)) {
                content.lines.add(key + '=' + newParametersToSave.get(key));
            }
        }
        // write the file
        try (var writer = Files.newBufferedWriter(nuxeoDefaultsPath)) {
            content.lines.forEach(asConsumer(line -> writer.append(line).append(System.lineSeparator())));
        } catch (IOException e) {
            throw new ConfigurationException("Unable to write to " + nuxeoDefaultsPath, e);
        }
        return oldValues;
    }

    protected NuxeoConfContent readConfiguration(Path nuxeoConfPath) throws ConfigurationException {
        NuxeoConfContent content = new NuxeoConfContent();
        try (var dis = new DigestInputStream(Files.newInputStream(nuxeoConfPath), DigestUtils.getMd5Digest());
                var reader = new BufferedReader(new InputStreamReader(dis))) {
            boolean onConfiguratorContent = false;
            String line;
            while ((line = reader.readLine()) != null) {
                line = line.trim();
                if (line.startsWith(BOUNDARY_BEGIN)) {
                    onConfiguratorContent = true;
                } else if (line.startsWith(BOUNDARY_END)) {
                    onConfiguratorContent = false;
                } else if (onConfiguratorContent) {
                    content.addConfiguratorLine(line);
                } else {
                    content.addLine(line);
                }
            }
            content.digest = Hex.encodeHexString(dis.getMessageDigest().digest());
        } catch (IOException e) {
            throw new ConfigurationException("Error reading:" + nuxeoConfPath, e);
        }
        return content;
    }

    protected void editUserConf(NuxeoConfContent content, ConfigurationHolder configHolder) {
        Integer generationIndex = null;
        Integer templatesIndex = null;
        String generationParam = configHolder.getProperty(PARAM_FORCE_GENERATION);
        String templatesParam = configHolder.getProperty(PARAM_TEMPLATES_NAME);
        int i;
        ListIterator<String> it = content.lines.listIterator();
        while ((i = it.nextIndex()) != content.lines.size()) {
            String line = it.next();
            if (line.startsWith(PARAM_FORCE_GENERATION)) {
                // keep only one force generation parameter
                if (generationParam != null) {
                    line = PARAM_FORCE_GENERATION + "=" + generationParam;
                }
                if (generationIndex == null) {
                    generationIndex = i;
                } else {
                    it.remove();
                }
                content.lines.set(generationIndex, line);
            } else if (line.startsWith(PARAM_TEMPLATES_NAME)) {
                // keep only one templates parameter
                if (templatesParam != null) {
                    line = PARAM_TEMPLATES_NAME + "=" + templatesParam;
                }
                if (templatesIndex == null) {
                    templatesIndex = i;
                } else {
                    it.remove();
                }
                content.lines.set(templatesIndex, line);
            } else {
                // keep only the user parameters
                int equalIdx = line.indexOf('=');
                if (equalIdx > 0 && !line.startsWith("#")) {
                    String key = line.substring(0, equalIdx).trim();
                    // property is unset by generator
                    if (configHolder.getProperty(key) == null) {
                        content.lines.set(i, '#' + line);
                        content.properties.remove(key);
                    }
                }
            }
        }
        if (templatesIndex == null) {
            content.lines.add(PARAM_TEMPLATES_NAME + '=' + defaultIfBlank(templatesParam, "default"));
        }
    }

    protected void editConfiguratorConf(NuxeoConfContent content, ConfigurationHolder configHolder) {
        content.configuratorProperties.entrySet()
                                      .removeIf(e -> e.getValue().equals(configHolder.getRawProperty(e.getKey())));
    }

    protected void writeConfiguration(NuxeoConfContent content, ConfigurationHolder configHolder)
            throws ConfigurationException {
        try (var baos = new ByteArrayOutputStream();
                var dos = new DigestOutputStream(baos, DigestUtils.getMd5Digest());
                var writer = new OutputStreamWriter(dos)) {
            // copy back user conf
            content.lines.forEach(asConsumer(line -> writer.append(line).append(System.lineSeparator())));
            // add some space between user and configurator configuration
            if (isNotBlank(content.lines.get(content.lines.size() - 1))) {
                writer.append(System.lineSeparator());
            }
            // write changed/configurator parameters
            writer.append(BOUNDARY_BEGIN).append(System.lineSeparator());
            for (String key : new TreeSet<>(configHolder.keySet())) {
                // ignore parameters already stored in user conf
                if (PARAM_FORCE_GENERATION.equals(key) || PARAM_TEMPLATES_NAME.equals(key)) {
                    continue;
                }
                // ignore user parameters not modified by generator
                String userValue = content.properties.get(key);
                String generatorValue = content.configuratorProperties.get(key);
                String valueToSet = StringUtils.trimToEmpty(configHolder.getRawProperty(key));
                if (content.properties.containsKey(key) && valueToSet.equals(userValue)) {
                    continue;
                }
                if (generatorValue != null) {
                    writer.append("#").append(key).append("=").append(generatorValue).append(System.lineSeparator());
                }
                writer.append(key).append("=").append(valueToSet).append(System.lineSeparator());
            }
            writer.append(BOUNDARY_END).append(System.lineSeparator());
            writer.flush();
            // write file only if content has changed
            if (!Hex.encodeHexString(dos.getMessageDigest().digest()).equals(content.digest)) {
                try (var os = Files.newOutputStream(configHolder.getNuxeoConfPath())) {
                    baos.writeTo(os);
                }
            }
        } catch (IOException e) {
            throw new ConfigurationException("Error writing: " + configHolder.getNuxeoConfPath(), e);
        }
    }

    /**
     * @param configHolder configuration to dump into {@code config} directory
     */
    public void dumpConfiguration(ConfigurationHolder configHolder) throws ConfigurationException {
        try {
            parseAndCopy(configHolder);
            dumpProperties(configHolder);
        } catch (FileNotFoundException e) {
            throw new ConfigurationException("Missing file: " + e.getMessage(), e);
        } catch (TemplateException | ParseException e) {
            throw new ConfigurationException("Could not process FreeMarker template: " + e.getMessage(), e);
        } catch (IOException e) {
            throw new ConfigurationException("Configuration failure: " + e.getMessage(), e);
        }
    }

    /**
     * Generate configuration files from templates and given configuration parameters
     */
    protected void parseAndCopy(ConfigurationHolder configHolder)
            throws IOException, TemplateException, ConfigurationException {
        // FilenameFilter for excluding "nuxeo.defaults" files from copy
        FilenameFilter filter = (dir, name) -> !FILE_NUXEO_DEFAULTS.equals(name)
                // exclude nuxeo.ENVIRONMENT files
                && !(name.startsWith("nuxeo.") && Files.exists(dir.toPath().resolve(FILE_NUXEO_DEFAULTS)));
        TextTemplate templateParser = instantiateTemplateParser(configHolder);

        deleteTemplateFiles(configHolder);
        // add included templates directories
        List<String> newFilesList = new ArrayList<>();
        for (Path includedTemplate : configHolder.getIncludedTemplates()) {
            File[] listFiles = includedTemplate.toFile().listFiles(filter);
            if (listFiles != null) {
                String templateName = includedTemplate.getFileName().toString();
                log.debug("Parsing template: {}... files: {}", () -> templateName, () -> Arrays.toString(listFiles));
                // Check for deprecation
                if (configHolder.getPropertyAsBoolean(templateName + ".deprecated")) {
                    log.warn("WARNING: Template {} is deprecated.", templateName);
                    configHolder.getOptProperty(templateName + ".deprecation").ifPresent(log::warn);
                }
                // Retrieve optional target directory if defined
                Path out = configHolder.getOptProperty(templateName + ".target")
                                       .map(configHolder.getHomePath()::resolve)
                                       .orElseGet(configHolder::getRuntimeHomePath);
                for (File in : listFiles) {
                    // copy template(s) directories parsing properties
                    newFilesList.addAll(templateParser.processDirectory(in, out.resolve(in.getName()).toFile()));
                }
            }
        }
        storeNewFilesList(configHolder, newFilesList);
    }

    protected TextTemplate instantiateTemplateParser(ConfigurationHolder configHolder) {
        TextTemplate templateParser = new TextTemplate(configHolder.userConfig);
        templateParser.setKeepEncryptedAsVar(true);
        templateParser.setTrim(true);
        templateParser.setTextParsingExtensions(
                configHolder.getProperty(PARAM_TEMPLATES_PARSING_EXTENSIONS, "xml,properties,nx"));
        templateParser.setFreemarkerParsingExtensions(
                configHolder.getProperty(PARAM_TEMPLATES_FREEMARKER_EXTENSIONS, "nxftl"));
        return templateParser;
    }

    /**
     * Delete files previously deployed by templates. If a file had been overwritten by a template, it will be restored.
     * Helps the server returning to the state before any template was applied.
     */
    protected void deleteTemplateFiles(ConfigurationHolder configHolder) throws IOException, ConfigurationException {
        Path newFiles = configHolder.getTemplatesPath().resolve(NEW_FILES);
        if (Files.notExists(newFiles)) {
            return;
        }
        try (BufferedReader reader = Files.newBufferedReader(newFiles)) {
            String line;
            while ((line = reader.readLine()) != null) {
                if (line.endsWith(".bak")) {
                    log.debug("Restore {}", line);
                    String originalName = line.substring(0, line.length() - 4);
                    try {
                        Path backup = configHolder.getHomePath().resolve(line);
                        Path original = configHolder.getHomePath().resolve(originalName);
                        Files.copy(backup, original, REPLACE_EXISTING, COPY_ATTRIBUTES);
                        Files.delete(backup);
                    } catch (IOException e) {
                        throw new ConfigurationException(
                                String.format("Failed to restore %s from %s\nEdit or delete %s to bypass that error.",
                                        originalName, line, newFiles),
                                e);
                    }
                } else {
                    log.debug("Remove {}", line);
                    Files.deleteIfExists(configHolder.getHomePath().resolve(line));
                }
            }
        }
        Files.delete(newFiles);
    }

    /**
     * Store into {@link #NEW_FILES} the list of new files deployed by the templates. For later use by
     * {@link #deleteTemplateFiles(ConfigurationHolder)}.
     */
    protected void storeNewFilesList(ConfigurationHolder configHolder, List<String> newFilesList) throws IOException {
        Path newFiles = configHolder.getTemplatesPath().resolve(NEW_FILES);
        try (BufferedWriter writer = Files.newBufferedWriter(newFiles, UTF_8, APPEND, CREATE, WRITE)) {
            // Store new files listing
            for (String filepath : newFilesList) {
                writer.write(configHolder.getHomePath().relativize(Path.of(filepath)).normalize().toString());
                writer.newLine();
            }
        }
    }

    /**
     * @param configHolder configuration to dump into {@code config} directory
     */
    protected void dumpProperties(ConfigurationHolder configHolder) {
        var properties = configHolder.stringPropertyNames()
                                     .stream()
                                     .sorted()
                                     // Exclude System properties except Nuxeo's System properties
                                     .filter(k -> systemProperties.getProperty(k) == null
                                             || NUXEO_SYSTEM_PROPERTIES.contains(k))
                                     .collect(Collectors.toMap(Function.identity(), configHolder::getRawProperty,
                                             (v1, v2) -> v2, Properties::new));
        Path dumpedFile = configHolder.getDumpedConfigurationPath();
        try {
            Files.createDirectories(dumpedFile.getParent());
            try (var os = Files.newBufferedWriter(dumpedFile, UTF_8)) {
                properties.store(os, "Generated by " + getClass());
            }
        } catch (IOException e) {
            log.error("Could not dump properties to {}", dumpedFile, e);
        }
    }

    protected static class NuxeoConfContent {

        protected List<String> lines = new ArrayList<>();

        /** Used as reference to detect configurator properties. */
        protected Map<String, String> properties = new HashMap<>();

        /** Used to preserve the old values in nuxeo.conf. */
        protected Map<String, String> configuratorProperties = new HashMap<>();

        protected String digest;

        public void addLine(String line) {
            lines.add(line);
            int equalIdx = line.indexOf('=');
            if (equalIdx > 0 && !line.startsWith("#")) {
                String key = line.substring(0, equalIdx).trim();
                String value = line.substring(equalIdx + 1).trim();
                properties.put(key, value);
            }
        }

        public void addConfiguratorLine(String line) {
            // Backward compliance, it must be ignored
            boolean wanted = !line.startsWith("#" + PARAM_TEMPLATES_NAME) && !line.startsWith(PARAM_TEMPLATES_NAME);
            // Ignore non-readable lines
            int equalIdx = line.indexOf('=');
            wanted &= equalIdx > 0;
            if (wanted) {
                String key;
                if (line.startsWith("#")) {
                    key = line.substring(1, equalIdx).trim();
                } else {
                    key = line.substring(0, equalIdx).trim();
                }
                String value = line.substring(equalIdx + 1).trim();
                configuratorProperties.put(key, value);
            }
        }
    }

}
