/*
 * Decompiled with CFR 0.152.
 */
package org.nuxeo.launcher.config;

import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.net.InetAddress;
import java.net.MalformedURLException;
import java.net.ServerSocket;
import java.net.URL;
import java.net.URLClassLoader;
import java.net.UnknownHostException;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.PathMatcher;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.Properties;
import java.util.concurrent.TimeUnit;
import java.util.stream.Stream;
import net.jodah.failsafe.Failsafe;
import net.jodah.failsafe.FailsafeException;
import net.jodah.failsafe.RetryPolicy;
import net.jodah.failsafe.SyncFailsafe;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.util.Supplier;
import org.nuxeo.common.utils.TextTemplate;
import org.nuxeo.launcher.config.ConfigurationException;
import org.nuxeo.launcher.config.ConfigurationHolder;
import org.nuxeo.launcher.config.JVMVersion;
import org.nuxeo.launcher.config.backingservices.BackingChecker;

public class ConfigurationChecker {
    private static final Logger log = LogManager.getLogger(ConfigurationChecker.class);
    protected static final String PARAM_RETRY_POLICY_ENABLED = "nuxeo.backing.check.retry.enabled";
    protected static final String PARAM_RETRY_POLICY_MAX_RETRIES = "nuxeo.backing.check.retry.maxRetries";
    protected static final String PARAM_RETRY_POLICY_DELAY_IN_MS = "nuxeo.backing.check.retry.delayInMs";
    protected static final int PARAM_POLICY_DEFAULT_DELAY_IN_MS = 5000;
    protected static final int PARAM_RETRY_POLICY_DEFAULT_RETRIES = 20;
    protected static final String DEFAULT_CONTEXT_NAME = "/nuxeo";
    protected static final String[] COMPLIANT_JAVA_VERSIONS = new String[]{"11"};
    protected static final String JVMCHECK_PROP = "jvmcheck";
    protected static final String JVMCHECK_FAIL = "fail";
    protected static final String JVMCHECK_NOFAIL = "nofail";
    protected static final int ADDRESS_PING_TIMEOUT_MS = 1000;
    protected static final int MIN_PORT = 1;
    protected static final int MAX_PORT = 65535;
    protected static final Path BAD_INSTANCE_CLID_PATH = Path.of("data", "instance.clid");
    protected final Properties systemProperties;

    public ConfigurationChecker(Properties systemProperties) {
        this.systemProperties = systemProperties;
    }

    public boolean isConfigured(ConfigurationHolder configHolder) {
        Path nuxeoContext = Path.of("conf", "Catalina", "localhost", this.getContextName(configHolder) + ".xml");
        return Files.exists(configHolder.getHomePath().resolve(nuxeoContext), new LinkOption[0]);
    }

    protected String getContextName(ConfigurationHolder configHolder) {
        return configHolder.getProperty("org.nuxeo.ecm.contextPath", DEFAULT_CONTEXT_NAME).substring(1);
    }

    public void verify(ConfigurationHolder configHolder) throws ConfigurationException {
        this.checkJavaVersionIsCompliant();
        this.checkAddressesAndPorts(configHolder);
        this.checkPaths(configHolder);
        this.checkBackingServices(configHolder);
    }

    protected void checkJavaVersionIsCompliant() throws ConfigurationException {
        String version = this.systemProperties.getProperty("java.version");
        this.checkJavaVersion(version, COMPLIANT_JAVA_VERSIONS);
    }

    protected void checkJavaVersion(String version, String[] compliantVersions) throws ConfigurationException {
        String lastCompliantVersion = null;
        for (String compliantVersion : compliantVersions) {
            if (this.checkJavaVersion(version, compliantVersion, false, false)) {
                lastCompliantVersion = compliantVersion;
                continue;
            }
            if (lastCompliantVersion != null) {
                return;
            }
            if (!this.checkJavaVersion(version, compliantVersion, true, true)) continue;
            return;
        }
        if (lastCompliantVersion != null) {
            this.checkJavaVersion(version, lastCompliantVersion, false, true);
            return;
        }
        String message = String.format("Nuxeo requires Java %s (detected %s).", ArrayUtils.toString((Object)compliantVersions), version);
        throw new ConfigurationException(message + " See 'jvmcheck' option to bypass version check.");
    }

    protected boolean checkJavaVersion(String version, String requiredVersion, boolean allowNoFailFlag, boolean warnIfLooseCompliance) {
        allowNoFailFlag = allowNoFailFlag && JVMCHECK_NOFAIL.equalsIgnoreCase(this.systemProperties.getProperty(JVMCHECK_PROP, JVMCHECK_FAIL));
        try {
            boolean compliant;
            JVMVersion required = JVMVersion.parse(requiredVersion);
            JVMVersion actual = JVMVersion.parse(version);
            boolean bl = compliant = actual.compareTo(required) >= 0;
            if (compliant && actual.compareTo(required, JVMVersion.UpTo.MAJOR) == 0) {
                return true;
            }
            if (!compliant && !allowNoFailFlag) {
                return false;
            }
            if (warnIfLooseCompliance) {
                log.warn("Nuxeo requires Java {}+ (detected {}).", (Object)requiredVersion, (Object)version);
            }
            return true;
        }
        catch (ParseException cause) {
            if (allowNoFailFlag) {
                log.warn("Cannot check java version", (Throwable)cause);
                return true;
            }
            throw new IllegalArgumentException("Cannot check java version", cause);
        }
    }

    protected void checkAddressesAndPorts(ConfigurationHolder configHolder) throws ConfigurationException {
        InetAddress bindAddress = this.getBindAddress(configHolder);
        if (bindAddress.isMulticastAddress()) {
            throw new ConfigurationException("Multicast address won't work: " + bindAddress);
        }
        this.checkAddressReachable(bindAddress);
        this.checkPortAvailable(bindAddress, configHolder.getPropertyAsInteger("nuxeo.server.http.port", 8080));
        this.checkPortAvailable(bindAddress, configHolder.getPropertyAsInteger("nuxeo.server.tomcat_admin.port", 8005));
    }

    protected InetAddress getBindAddress(ConfigurationHolder configHolder) throws ConfigurationException {
        InetAddress bindAddress;
        try {
            String hostName = configHolder.getProperty("nuxeo.bind.address");
            bindAddress = InetAddress.getByName(hostName);
            if (bindAddress.isAnyLocalAddress()) {
                boolean preferIPv6 = "false".equals(this.systemProperties.getProperty("java.net.preferIPv4Stack")) && "true".equals(this.systemProperties.getProperty("java.net.preferIPv6Addresses"));
                bindAddress = preferIPv6 ? InetAddress.getByName("::1") : InetAddress.getByName("127.0.0.1");
                log.debug("Bind address is \"ANY\", using local address instead: {}", (Object)bindAddress);
            }
            log.debug("Configured bind address: {}", (Object)bindAddress);
        }
        catch (UnknownHostException e) {
            throw new ConfigurationException(e);
        }
        return bindAddress;
    }

    protected void checkAddressReachable(InetAddress address) throws ConfigurationException {
        try {
            log.debug("Checking availability of address: {}", (Object)address);
            address.isReachable(1000);
        }
        catch (IOException | IllegalArgumentException e) {
            throw new ConfigurationException("Unreachable bind address " + address, e);
        }
    }

    protected void checkPortAvailable(InetAddress address, int port) throws ConfigurationException {
        if (port == 0 || port == -1) {
            log.warn("Port is set to {} - assuming it is disabled - skipping availability check", (Object)port);
            return;
        }
        if (port < 1 || port > 65535) {
            throw new IllegalArgumentException("Invalid port: " + port);
        }
        log.debug("Checking availability of port {} on address {}", (Object)port, (Object)address);
        try (ServerSocket socketTCP = new ServerSocket(port, 0, address);){
            socketTCP.setReuseAddress(true);
        }
        catch (IOException e) {
            throw new ConfigurationException(e.getMessage() + ": " + address + ":" + port, e);
        }
    }

    protected void checkPaths(ConfigurationHolder configHolder) throws ConfigurationException {
        Path badInstanceClid = configHolder.getRuntimeHomePath().resolve(BAD_INSTANCE_CLID_PATH);
        Path dataPath = configHolder.getDataPath();
        if (Files.exists(badInstanceClid, new LinkOption[0]) && !badInstanceClid.startsWith(dataPath)) {
            log.warn("Moving {} to {}.", (Object)badInstanceClid, (Object)dataPath);
            try {
                FileUtils.moveFileToDirectory((File)badInstanceClid.toFile(), (File)dataPath.toFile(), (boolean)true);
            }
            catch (IOException e) {
                throw new ConfigurationException("NXP-6722 move failed: " + e.getMessage(), e);
            }
        }
        Path oldPackagesPath = dataPath.resolve("packages");
        Path packagesPath = configHolder.getPackagesPath();
        if (Files.exists(oldPackagesPath, new LinkOption[0]) && !oldPackagesPath.equals(packagesPath)) {
            log.warn("NXP-8014 Packages cache location changed. You can safely delete {} or move its content to {}", (Object)oldPackagesPath, (Object)packagesPath);
        }
    }

    public void checkBackingServices(ConfigurationHolder configHolder) throws ConfigurationException {
        RetryPolicy retryPolicy = this.buildRetryPolicy(configHolder);
        for (BackingChecker checker : this.instantiateBackingCheckers(configHolder)) {
            if (!checker.accepts(configHolder)) continue;
            ClassLoader classLoader = Thread.currentThread().getContextClassLoader();
            try {
                Thread.currentThread().setContextClassLoader(checker.getClass().getClassLoader());
                ((SyncFailsafe)((SyncFailsafe)Failsafe.with((RetryPolicy)retryPolicy).onFailedAttempt(failure -> log.error(failure.getMessage(), failure))).onRetry((c, f, ctx) -> log.warn("Failure {}. Retrying....", (Object)ctx.getExecutions()))).run(() -> checker.check(configHolder));
            }
            catch (FailsafeException e) {
                if (e.getCause() instanceof ConfigurationException) {
                    throw (ConfigurationException)e.getCause();
                }
                throw new ConfigurationException("Error during backing checks", e);
            }
            finally {
                Thread.currentThread().setContextClassLoader(classLoader);
            }
        }
    }

    protected RetryPolicy buildRetryPolicy(ConfigurationHolder configHolder) {
        RetryPolicy retryPolicy = new RetryPolicy().withMaxRetries(0);
        if (configHolder.getPropertyAsBoolean(PARAM_RETRY_POLICY_ENABLED)) {
            int maxRetries = configHolder.getPropertyAsInteger(PARAM_RETRY_POLICY_MAX_RETRIES, 20);
            int delay = configHolder.getPropertyAsInteger(PARAM_RETRY_POLICY_DELAY_IN_MS, 5000);
            retryPolicy = retryPolicy.retryOn(ConfigurationException.class).withMaxRetries(maxRetries).withDelay((long)delay, TimeUnit.MILLISECONDS);
        }
        return retryPolicy;
    }

    protected List<BackingChecker> instantiateBackingCheckers(ConfigurationHolder configHolder) throws ConfigurationException {
        ArrayList<BackingChecker> checkers = new ArrayList<BackingChecker>();
        ArrayList<String> items = new ArrayList<String>(configHolder.getIncludedTemplateNames());
        items.add("elasticsearch");
        items.add("kafka");
        for (String item : items) {
            try {
                log.debug("Resolving checker: {}", (Object)item);
                URLClassLoader ucl = this.getBackingCheckerClassLoader(configHolder, item);
                if (ucl.getURLs().length <= 0) continue;
                log.debug("Adding checker: {} with class path: {}", new Supplier[]{() -> item, () -> Arrays.toString(ucl.getURLs())});
                String checkClass = configHolder.getProperty(item + ".check.class");
                log.debug("Instantiating checker: {} class: {}", (Object)item, (Object)checkClass);
                Class<?> klass = Class.forName(checkClass, true, ucl);
                checkers.add((BackingChecker)klass.getDeclaredConstructor(new Class[0]).newInstance(new Object[0]));
            }
            catch (ClassCastException | ReflectiveOperationException e) {
                throw new ConfigurationException("Unable to check configuration for backing service: " + item, e);
            }
        }
        return checkers;
    }

    protected URLClassLoader getBackingCheckerClassLoader(ConfigurationHolder configHolder, String checker) {
        Path templatePath = this.getTemplatePath(configHolder, checker);
        String classPath = this.getBackingCheckerClasspath(configHolder, checker);
        URL[] urls = (URL[])Stream.of(classPath.split(":")).flatMap(e -> this.getJarsFromClasspathEntry(configHolder, templatePath, (String)e)).map(this::convertToJarFileURL).filter(Objects::nonNull).peek(u -> log.debug("Adding url: {}", u)).toArray(URL[]::new);
        return new URLClassLoader(urls);
    }

    private Path getTemplatePath(ConfigurationHolder configHolder, String checker) {
        Path templatePath = configHolder.getTemplatesPath().resolve(checker);
        if (Files.notExists(templatePath, new LinkOption[0])) {
            templatePath = configHolder.getTemplatesPath().resolve("default");
        }
        return templatePath;
    }

    protected String getBackingCheckerClasspath(ConfigurationHolder configHolder, String template) {
        String classPath = configHolder.getProperty(template + ".check.classpath");
        TextTemplate templateParser = new TextTemplate((Properties)configHolder.userConfig);
        return StringUtils.trimToEmpty((String)templateParser.processText(classPath));
    }

    protected Stream<Path> getJarsFromClasspathEntry(ConfigurationHolder configHolder, Path templatePath, String entry) {
        Path entryPath = Path.of(entry, new String[0]);
        if (!entryPath.isAbsolute()) {
            entryPath = templatePath.resolve(entryPath);
        }
        String pattern = "glob:" + entryPath.toString().replaceAll("\\\\", "\\\\\\\\");
        PathMatcher matcher = FileSystems.getDefault().getPathMatcher(pattern);
        FileFilter filter = f -> matcher.matches(f.toPath()) && f.toPath().startsWith(configHolder.getHomePath());
        File[] matchingFiles = entryPath.getParent().toFile().listFiles(filter);
        if (matchingFiles != null) {
            Stream.Builder<Path> builder = Stream.builder();
            for (File file : matchingFiles) {
                if (file.isDirectory()) {
                    List.of(file.listFiles(f -> f.getName().endsWith(".jar"))).forEach(f -> builder.add(f.toPath()));
                    continue;
                }
                if (!file.getName().endsWith(".jar")) continue;
                builder.add(file.toPath());
            }
            return builder.build();
        }
        return Stream.empty();
    }

    protected URL convertToJarFileURL(Path path) {
        try {
            return new URL("jar:file:" + path + "!/");
        }
        catch (MalformedURLException e) {
            log.error("Unable to convert path: {} to URL", (Object)path, (Object)e);
            return null;
        }
    }

    public static void checkJavaVersion() throws ConfigurationException {
        new ConfigurationChecker(System.getProperties()).checkJavaVersionIsCompliant();
    }

    public static boolean checkJavaVersion(String version, String requiredVersion) {
        return new ConfigurationChecker(System.getProperties()).checkJavaVersion(version, requiredVersion, false, false);
    }
}

