/*
 * (C) Copyright 2007 Nuxeo SAS (http://nuxeo.com/) and contributors.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser General Public License
 * (LGPL) version 2.1 which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl.html
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * Contributors:
 *     Nuxeo - initial API and implementation
 *
 * $Id$
 */

package org.nuxeo.runtime.api.login;

import java.util.ArrayList;
import java.util.List;

import javax.security.auth.Subject;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.login.AppConfigurationEntry;
import javax.security.auth.login.LoginContext;
import javax.security.auth.login.LoginException;
import javax.security.auth.login.AppConfigurationEntry.LoginModuleControlFlag;

import org.nuxeo.common.xmap.annotation.XNode;
import org.nuxeo.common.xmap.annotation.XNodeList;
import org.nuxeo.common.xmap.annotation.XObject;

/**
 * @author <a href="mailto:bs@nuxeo.com">Bogdan Stefanescu</a>
 *
 */
@XObject("domain")
public class SecurityDomain {

    @XNode("@name")
    private String name;

    private AppConfigurationEntry[] entries;

    public SecurityDomain() {

    }
    
    public SecurityDomain(String name) {
        this.name = name;
    }

    public SecurityDomain(String name, AppConfigurationEntry[] entries) {
        this.name = name;
        this.entries = entries;
    }

    /**
     * @return the name.
     */
    public String getName() {
        return name;
    }

    public AppConfigurationEntry[] getAppConfigurationEntries() {
        return entries;
    }
    
    public void setAppConfigurationEntries(AppConfigurationEntry[] entries) {
        this.entries = entries;
    }

    @XNodeList(value = "login-module", type = ArrayList.class, componentType = LoginModuleDescriptor.class)
    public void setEntries(List<LoginModuleDescriptor> descriptors) {
        entries = new AppConfigurationEntry[descriptors.size()];
        int i = 0;
        for (LoginModuleDescriptor descriptor : descriptors) {
            LoginModuleControlFlag flag = null;
            if (descriptor.flag == null) {
                flag = LoginModuleControlFlag.OPTIONAL;
            } else if ("optional".equals(descriptor.flag)) {
                flag = LoginModuleControlFlag.OPTIONAL;
            } else if ("sufficient".equals(descriptor.flag)) {
                flag = LoginModuleControlFlag.SUFFICIENT;
            } else if ("required".equals(descriptor.flag)) {
                flag = LoginModuleControlFlag.REQUIRED;
            } else if ("requisite".equals(descriptor.flag)) {
                flag = LoginModuleControlFlag.REQUISITE;
            }
            entries[i++] = new AppConfigurationEntry(descriptor.code, flag,
                    descriptor.options);
        }
    }

    public LoginContext login(Subject subject) throws LoginException {
        LoginContext ctx = new LoginContext(name, subject);
        ctx.login();
        return ctx;
    }

    public LoginContext login(CallbackHandler handler) throws LoginException {
        LoginContext ctx = new LoginContext(name, handler);
        ctx.login();
        return ctx;
    }

    public LoginContext login(Subject subject, CallbackHandler handler)
            throws LoginException {
        LoginContext ctx = new LoginContext(name, subject, handler);
        ctx.login();
        return ctx;
    }

    public LoginContext login(String username, Object credentials)
            throws LoginException {
        DefaultCallbackHandler handler = new DefaultCallbackHandler(username,
                credentials);
        LoginContext ctx = new LoginContext(name, handler);
        ctx.login();
        return ctx;
    }
    
    
    public static final String controlFlagToString(LoginModuleControlFlag flag) {
        if (flag == LoginModuleControlFlag.OPTIONAL) {
            return "optional";
        } else if (flag == LoginModuleControlFlag.REQUIRED) {
            return "required";
        } else if (flag == LoginModuleControlFlag.REQUISITE) {            
            return "requisite";
        } else if (flag == LoginModuleControlFlag.SUFFICIENT) {
            return "sufficient";            
        }
        throw new IllegalArgumentException("Not a supported LoginModuleControlFlag: "+flag);
    }

    public static final LoginModuleControlFlag controlFlagFromString(String flag) {
        if (flag == null) {
            return LoginModuleControlFlag.OPTIONAL;
        } else if ("optional".equals(flag)) {
            return LoginModuleControlFlag.OPTIONAL;
        } else if ("sufficient".equals(flag)) {
            return LoginModuleControlFlag.SUFFICIENT;
        } else if ("required".equals(flag)) {
            return LoginModuleControlFlag.REQUIRED;
        } else if ("requisite".equals(flag)) {
            return LoginModuleControlFlag.REQUISITE;
        }
        throw new IllegalArgumentException("Not a supported LoginModuleControlFlag: "+flag);
    }
    
}
