/*
 * (C) Copyright 2006-2007 Nuxeo SAS (http://nuxeo.com/) and contributors.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser General Public License
 * (LGPL) version 2.1 which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl.html
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * Contributors:
 *     Nuxeo - initial API and implementation
 *
 * $Id$
 */

package org.nuxeo.runtime.remoting;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.remoting.InvokerLocator;
import org.nuxeo.runtime.ComponentEvent;
import org.nuxeo.runtime.ComponentListener;
import org.nuxeo.runtime.RuntimeService;
import org.nuxeo.runtime.api.Framework;
import org.nuxeo.runtime.model.ComponentContext;
import org.nuxeo.runtime.model.ComponentName;
import org.nuxeo.runtime.model.DefaultComponent;
import org.nuxeo.runtime.model.Extension;
import org.nuxeo.runtime.model.Property;
import org.nuxeo.runtime.model.RegistrationInfo;
import org.nuxeo.runtime.model.impl.ComponentManagerImpl;
import org.nuxeo.runtime.model.impl.ExtensionImpl;
import org.nuxeo.runtime.remoting.net.EventHandler;
import org.nuxeo.runtime.remoting.net.NetworkNode;
import org.nuxeo.runtime.remoting.net.NetworkNodeFactory;
import org.nuxeo.runtime.remoting.net.NodeInfo;
import org.nuxeo.runtime.remoting.net.impl.NetworkNodeFactoryImpl;
import org.nuxeo.runtime.remoting.transporter.TransporterClient;
import org.nuxeo.runtime.remoting.transporter.TransporterServer;

/**
 * @author <a href="mailto:bs@nuxeo.com">Bogdan Stefanescu</a>
 *
 */
public class RemotingService extends DefaultComponent implements
        ComponentListener, EventHandler {

    public static final ComponentName NAME = new ComponentName(
            "org.nuxeo.runtime.remoting.RemotingService");

    private static final Log log = LogFactory.getLog(RemotingService.class);

    private ServerRegistry servers;

    private TransporterServer transporterServer;

    private Server server;

    private NetworkNode node;

    private RuntimeService runtime;


    /**
     * Helper method to connect to a remote nuxeo runtime server
     * @param host the remote host
     * @param port the remote port
     * @return the server object
     */
    public final static Server connect(String host, int port) throws Exception {
        return (Server) TransporterClient.createTransporterClient(new InvokerLocator(getServerURI(host, port)), Server.class);
    }
    
    /**
     * Helper method to disconnect from a remote server
     * @param server
     */
    public final static void disconnect(Server server) {
        TransporterClient.destroyTransporterClient(server);
    }
    
    public final static String getServerURI(String host, int port) {
        return "socket://"+host+":"+port+"/nxruntime";
    }
    
    /**
     * Test the connection with a remote server
     * 
     * @return the product info if successful, null otherwise 
     */
    public static String ping(String host, int port) {
        try {
            Server server = connect(host, port);            
            try {
                return server.getProductInfo();
            } finally {
                TransporterClient.destroyTransporterClient(server);    
            }
        } catch (Throwable t) {
        }
        return null;
    }

    @Override
    public void activate(ComponentContext context) throws Exception {
        runtime = context.getRuntimeContext().getRuntime();
        Property host = context.getProperty("server-host");
        Property port = context.getProperty("server-port");
        String hdef = System.getProperty("nxruntime-host", "127.0.0.1");
        String h = host != null ? host.getString() : hdef;
        Integer pdef = Integer.getInteger("nxruntime-port");
        int p = port != null ? port.getInteger() : 62474;
        if (pdef != null) {
            p = pdef; // use the system property if defined
        }
        InvokerLocator serverLocator = new InvokerLocator("socket", h, p,
                "nxruntime", null);
        server = new ServerImpl(this, context.getRuntimeContext().getRuntime());
        transporterServer = TransporterServer.createTransporterServer(
                serverLocator, server, Server.class.getName());


        boolean isNetworkingEnabled = (Boolean) context.getPropertyValue("net-enabled", Boolean.FALSE);

        if (isNetworkingEnabled) {
            log.info("Starting networking");

            Framework.getRuntime().getComponentManager().addComponentListener(this);

            String netFactory = null;
            Property netFactoryProp = context.getProperty("net-factory");
            if (netFactoryProp != null) {
                netFactory = netFactoryProp.getString();
            }
            NetworkNodeFactory factory;
            if (netFactory != null) {
                factory = (NetworkNodeFactory) context.getRuntimeContext()
                    .loadClass(netFactory).newInstance();
            } else {
                factory = new NetworkNodeFactoryImpl();
            }

            NodeInfo info = new NodeInfo(serverLocator.getLocatorURI());
            info.name = (String) context.getPropertyValue("server-name");
            info.description  = (String) context.getPropertyValue("server-description", runtime.getDescription());
            node = factory.createNode(this, info);

            Property groupProp = context.getProperty("net-group");
            // join group if any
            String group = null;
            if (groupProp != null) {
                group = groupProp.getString();
            }
            servers = new ServerRegistry(node);
            node.setEventHandler(this);
            node.connect(group);
        } else {
            log.info("Networking is disabled.");
        }
    }

    public TransporterServer getTransporterServer() {
        return transporterServer;
    }
    

    @Override
    public void deactivate(ComponentContext context) throws Exception {
        if (servers != null) {
            servers.clear();
        }
        Framework.getRuntime().getComponentManager().removeComponentListener(
                this);
        if (node != null) {
            node.disconnect();
        }
        if (transporterServer != null) {
            transporterServer.stop();
            transporterServer = null;
        }
    }

    public Server getServer() {
        return server;
    }

    public ServerDescriptor[] getServers() {
        return servers.getServers();
    }

    public ServerDescriptor getServer(String uri) {
        return servers.get(uri);
    }

    public void handleEvent(ComponentEvent event) {
        RemoteEvent revent;
        switch (event.id) {
        case ComponentEvent.COMPONENT_ACTIVATED:
            revent = new RemoteEvent(RemoteEvent.COMPONENTS_ADDED,
                    null, new ComponentName[] {event.registrationInfo.getName()});
            sendRemoteEvent(revent);
            break;
        case ComponentEvent.COMPONENT_DEACTIVATED:
            revent = new RemoteEvent(RemoteEvent.COMPONENTS_REMOVED,
                    null, new ComponentName[] {event.registrationInfo.getName()});
            sendRemoteEvent(revent);
            break;
        case ComponentEvent.EXTENSION_PENDING:
            ExtensionImpl extension = (ExtensionImpl) event.data;
            ComponentName target = extension.getTargetComponent();
            ComponentName contributor = extension.getComponent().getName();
            revent = new RemoteEvent(RemoteEvent.EXTENSION_ADDED,
                    contributor, extension.toXML());
            for (ServerDescriptor sd : getServers()) {
                if (sd.hasComponent(target)) {
                    boolean isNew = sd.addExtension(extension.getId());
                    if (isNew) {
                        try {
                            log.info("Sending extension " + extension.getId() + " to " + sd.getURI());
                            node.sendTo(sd.getURI(), revent);
                        } catch (Exception e) {
                            log.error("Failed to send extension " + extension.getId() + " to " + sd.getURI(), e);
                        }
                    }
                }
            }
            break;
        case ComponentEvent.EXTENSION_UNREGISTERED:
            extension = (ExtensionImpl) event.data;
            target = extension.getTargetComponent();
            contributor = extension.getComponent().getName();
            revent = new RemoteEvent(RemoteEvent.EXTENSION_REMOVED,
                    contributor, extension.toXML());

            for (ServerDescriptor sd : getServers()) {
                if (sd.hasComponent(target)) {
                    if (sd.removeExtension(extension.getId())) {
                        try {
                            log.info("Sending extension " + extension.getId() + " to " + sd.getURI());
                            node.sendTo(sd.getURI(), revent);
                        } catch (Exception e) {
                            log.error("Failed to send extension " + extension.getId() + " to " + sd.getURI(), e);
                        }
                    }
                }
            }

            break;
        }
    }

    private void sendRemoteEvent(RemoteEvent event) {
        try {
            node.send(event);
        } catch (Exception e) {
            log.error("Failed to broadcast event " + event, e);
        }
    }

    public ComponentName[] getComponents() {
        Collection<RegistrationInfo> regs = Framework.getRuntime().getComponentManager().getRegistrations();
        List<ComponentName> comps = new ArrayList<ComponentName>();
        for (RegistrationInfo ri : regs) {
            comps.add(ri.getName());
        }
        return comps.toArray(new ComponentName[comps.size()]);
    }

    // -------------  EventHandler API -----------

    public void peerConnected(NodeInfo info) {
        log.info("Peer connected " + info + ". Sending our configuration");
        try {
            // send all registered components to the remote server
            // Note that in the meantime components may be already sent to that server so
            // that the target server must be prepared toreceive duplicate components - and ignore them if the case
            ComponentName[] components = getComponents();
            RemoteEvent revent = new RemoteEvent(RemoteEvent.COMPONENTS_ADDED, null, components);
            try {
                node.sendTo(info.uri, revent);
            } catch (Exception e) {
                log.error("Failed to send COMPONENTS_ADDED event to " + info, e);
            }
        } catch (Exception e) {
            log.error("Failed to add peer " + info, e);
        }
    }

    public void peerRemoved(String uri) {
        log.info("Removing peer server: " + uri);
        servers.remove(uri);
    }

    public void handleEvent(String senderUri, Serializable object) {
        if (!(object instanceof RemoteEvent)) {
            return; //ignore unkown events
        }
        RemoteEvent event = (RemoteEvent) object;
        switch (event.id) {
        case RemoteEvent.COMPONENTS_ADDED:
            handleComponentsAdded(senderUri, event);
            break;
        case RemoteEvent.COMPONENTS_REMOVED:
            handleComponentsRemoved(senderUri, event);
            break;
        case RemoteEvent.EXTENSION_ADDED:
            try {
                handleExtensionAdded(senderUri, event);
            } catch (Exception e) {
                log.error("Failed to handle EXTENSION_ADDED event", e);
            }
            break;
        case RemoteEvent.EXTENSION_REMOVED:
            try {
                handleExtensionRemoved(senderUri, event);
            } catch (Exception e) {
                log.error("Failed to handle EXTENSION_REMOVED event", e);
            }
            break;
        }
    }

    protected void sendPendingExtensions(ServerDescriptor sd) throws Exception {
        for (ComponentName target : sd.getComponents()) {
            sendPendingExtensions(sd, target);
        }
    }

    protected void sendPendingExtensions(ServerDescriptor sd, ComponentName target)
            throws Exception {
        ComponentManagerImpl mgr = (ComponentManagerImpl) runtime.getComponentManager();
        Collection<Extension> extensions = mgr.getPendingExtensions(target);
        if (extensions != null) {
            // TODO: don't reuse same object over the loop.
            RemoteEvent event = new RemoteEvent(RemoteEvent.EXTENSION_ADDED, null, null);
            for (Extension extension : extensions) {
                boolean isNew = sd.addExtension(extension.getId());
                if (isNew) {// avoid sending twice the same extension
                    event.component = extension.getComponent().getName();
                    event.data = extension.toXML();
                    node.sendTo(sd.getURI(), event);
                }
            }
        }
    }

    protected void handleComponentsAdded(String senderUri, RemoteEvent event) {
        ServerDescriptor sd = servers.get(senderUri);
        if (sd == null) {
            log.error("COMPONENTS_ADDED was received from an unknown server: "
                    + senderUri + ". Ignoring it");
            return;
        }
        ComponentName[] comps = (ComponentName[]) event.data;
        log.info("COMPONENTS_ADDED event received from " + senderUri + " : " + Arrays.toString(comps));
        for (ComponentName name : comps) {
            try {
                if (!sd.hasComponent(name)) {
                    sd.addComponent(name);
                    sendPendingExtensions(sd, name);
                }
            } catch (Exception e) {
                log.error("Failed to send EXTENSIONS_ADDED to r: " + senderUri);
            }
        }
    }

    protected void handleComponentsRemoved(String senderUri, RemoteEvent event) {
        ServerDescriptor sd = servers.get(senderUri);
        if (sd == null) {
            log.error("COMPONENTS_REMOVED was received from an unknown server: "
                    + senderUri + ". Ignoring it");
            return;
        }
        ComponentName[] comps = (ComponentName[]) event.data;
        log.info("COMPONENTS_REMOVED event received from " + senderUri + " : " + Arrays.toString(comps));
        ComponentManagerImpl mgr = (ComponentManagerImpl) runtime.getComponentManager();
        for (ComponentName name : comps) {
            RemoteComponent rco = sd.removeComponent(name);
            if (rco != null) {
                for (Extension xt : rco.getExtensions()) {
                    try {
                        log.info("Unregistering remote extension: " + xt.getId());
                        mgr.unregisterExtension(xt);
                    } catch (Exception e) {
                        log.error("Failed to unregister remote extension", e);
                    }
                }
            }
        }
    }

    protected void handleExtensionAdded(String senderUri, RemoteEvent event) throws Exception {
        ServerDescriptor sd = servers.get(senderUri);
        if (sd == null) {
            log.error("EXTENSION_ADDED was received from an unknown server: "
                    + senderUri + ". Ignoring it");
            return;
        }
        log.info("EXTENSION_ADDED event received from " + senderUri + " : " + event.data);
        ComponentManagerImpl mgr = (ComponentManagerImpl) runtime.getComponentManager();
        RemoteContext ctx = new RemoteContext(sd, event.component, null);
        ExtensionImpl extension = ExtensionImpl.fromXML(ctx, (String) event.data);
        extension.setComponent(new RemoteComponentInstance(event.component, ctx));
        try {
            mgr.registerExtension(extension);
            ComponentName target = extension.getTargetComponent();
            RemoteComponent rco = sd.getComponent(target);
            if (rco == null) {
                rco = sd.addComponent(target);
            }
            rco.addExtension(extension);
        } catch (Throwable t) {
            t.printStackTrace();
        }
    }

    //TODO: this is not working for now
    protected void handleExtensionRemoved(String senderUri, RemoteEvent event) throws Exception {
        ServerDescriptor sd = servers.get(senderUri);
        if (sd == null) {
            log.error("EXTENSION_REMOVED was received from an unknown server: "
                    + senderUri + ". Ignoring it");
            return;
        }
        log.info("EXTENSION_REMOVED event received from " + senderUri + " : " + event.data);
        ComponentManagerImpl mgr = (ComponentManagerImpl) runtime.getComponentManager();
        RemoteContext ctx = new RemoteContext(sd, event.component, null);
        ExtensionImpl extension = ExtensionImpl.fromXML(ctx, (String) event.data);
        extension.setComponent(new RemoteComponentInstance(event.component, ctx));
        try {
            mgr.unregisterExtension(extension);
        } catch (Throwable t) {
            t.printStackTrace();
        }
    }

}
