/*
 * (C) Copyright 2006-2007 Nuxeo SAS (http://nuxeo.com/) and contributors.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser General Public License
 * (LGPL) version 2.1 which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl.html
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * Contributors:
 *     Nuxeo - initial API and implementation
 *
 * $Id$
 */

package org.nuxeo.runtime.remoting;

import java.util.HashMap;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuxeo.runtime.remoting.net.NetworkNode;

/**
 * @author  <a href="mailto:bs@nuxeo.com">Bogdan Stefanescu</a>
 *
 */
public class ServerRegistry {

    private static final Log log = LogFactory.getLog(ServerRegistry.class);

    /**
     * Connected servers.
     */
    private final Map<String, ServerDescriptor> servers = new HashMap<String, ServerDescriptor>();

    private final NetworkNode node;


    public ServerRegistry(NetworkNode node) {
        this.node = node;
    }

    protected ServerDescriptor get(String uri) {
        ServerDescriptor sd = servers.get(uri);
        try {
            if (sd == null) {
                synchronized (servers) {
                    sd = servers.get(uri);
                    if (sd != null) {
                        return sd;
                    }
                    if (!node.isConnected(uri)) {
                        return null;
                    }
                    sd = new ServerDescriptor(uri);
                    servers.put(uri, sd);
                }
            }
        } catch (Exception e) {
            log.error("Failed to create server descriptor for " + uri, e);
        }
        return sd;
    }

    protected void remove(String uri) {
        synchronized (servers) {
            ServerDescriptor sd = servers.remove(uri);
            if (sd != null) {
                sd.destroy();
            }
        }
    }

    protected void clear() {
        synchronized (servers) {
            ServerDescriptor[] tmp =  servers.values().toArray(new ServerDescriptor[servers.size()]);
            for (ServerDescriptor sd : tmp) {
                sd.destroy();
            }
            servers.clear();
        }
    }

    protected ServerDescriptor[] getServers() {
        synchronized (servers) {
            return servers.values().toArray(new ServerDescriptor[servers.size()]);
        }
    }

}
