/*
 * (C) Copyright 2006-2007 Nuxeo SAS (http://nuxeo.com/) and contributors.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser General Public License
 * (LGPL) version 2.1 which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl.html
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * Contributors:
 *     Nuxeo - initial API and implementation
 *
 * $Id$
 */

package org.nuxeo.runtime.remoting.net.impl;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.remoting.InvokerLocator;
import org.nuxeo.runtime.remoting.RemotingService;
import org.nuxeo.runtime.remoting.net.EventHandler;
import org.nuxeo.runtime.remoting.net.NetworkNode;
import org.nuxeo.runtime.remoting.net.NodeInfo;
import org.nuxeo.runtime.remoting.transporter.TransporterClient;
import org.nuxeo.runtime.remoting.transporter.TransporterServer;


/**
 *
 * A connection to a node is established as follow:
 * <p>
 *
 * 1. The node is detected ( nodeOnline() called by the detection routine)
 * The node is put in a pending state until a SYN will be received from the detected node
 * <br>
 * 2. A SYN event is sent to the detected node
 * <br>
 * 3. A SYN is received from the detected node. This will activate the connection.
 * The upper layer is notified about the new peer
 *
 * <p>
 * A connection to a node should be activated only after a SYN is received from the node.
 *
 * @author  <a href="mailto:bs@nuxeo.com">Bogdan Stefanescu</a>
 *
 */
public class NetworkNodeImpl implements NetworkNode, NetworkPeer {

    private static final Log log = LogFactory.getLog(NetworkNodeImpl.class);

    private static final Object lock = new Object();

    private final Map<String, PeerEntry> detectedNodes = new HashMap<String, PeerEntry>();
    private final Map<String, PeerEntry> peers = new HashMap<String, PeerEntry>();
    private final NodeInfo info;
    private EventHandler handler;

    public NetworkNodeImpl(RemotingService remoting, NodeInfo info) throws Exception {
        this.info = info;
        TransporterServer transporterServer = remoting.getTransporterServer();
        transporterServer.addHandler(this, NetworkPeer.class.getName());
    }

    public void connect(String group) throws Exception {
        // if group is null I should be the group master
        if (group == null || group.equals(info.uri)) {
            log.info("Ignore connecting to the network group -> I am the group master");
            return;
        }
        log.info("Connecting to the network group: " + group);
        // group should be the uri of a master peer
        PeerEntry entry = nodeAdded(group);
        if (entry == null) {
            return;
        }
        // notify all visible peers about me
        NodeInfo[] infos = entry.peer.getPeers();
        if (infos == null) {
            return;
        }
        for (NodeInfo nodeInfo : infos) {
            // take care because I may exists in the peer list - ignore myself
            if (!nodeInfo.uri.equals(info.uri)) {
                nodeAdded(nodeInfo.uri);
            }
        }
    }

    public void disconnect() throws Exception {
        log.info("Disconnecting from network group");
        Iterator<PeerEntry> it = peers.values().iterator();
        while (it.hasNext()) {
            PeerEntry entry = it.next();
            try {
                entry.peer.close(info.uri);
            } catch (Exception e) {
                log.error("Failed to send nodeRemoved event to peer: " + entry.info);
            } finally {
                TransporterClient.destroyTransporterClient(entry.peer);
            }
            it.remove();
        }
        peers.clear(); // make sure all is cleared
    }

    public EventHandler getEventHandler() {
        return handler;
    }

    public NodeInfo getNodeInfo() {
        return info;
    }

    public String getNodeURI() {
        return info.uri;
    }

    public  NodeInfo[] getPeers() {
        List<NodeInfo> infos = new ArrayList<NodeInfo>(peers.size());
        for (PeerEntry peer : peers.values()) {
            infos.add(peer.info);
        }
        return infos.toArray(new NodeInfo[infos.size()]);
    }

    public void send(Serializable event) throws Exception {
        PeerEntry[] entries = peers.values().toArray(new PeerEntry[peers.size()]);
        if (entries == null) {
            return;
        }
        for (PeerEntry entry : entries) {
            try {
                entry.peer.handleEvent(info.uri, event);
            } catch (Exception e) {
                log.error("Failed to send event " + event + " to peer: " + entry.info);
            }
        }
    }

    public void sendTo(String uri, Serializable event) throws Exception {
        PeerEntry entry = peers.get(uri);
        if (entry == null) {
            return;
        }
        try {
            //Profiler.checkpoint("sendTo");
            entry.peer.handleEvent(info.uri, event);
            //Profiler.print("sendTo");
        } catch (Exception e) {
            log.error("Failed to send event " + event + " to peer: " + entry.info);
        }
    }

    public void setEventHandler(EventHandler handler) {
        this.handler = handler;
    }

    public void handleEvent(String source, Serializable event) throws Exception {
        //Profiler.checkpoint("handleEvent");
        if (handler != null) {
            handler.handleEvent(source, event);
        }
        //Profiler.print("handleEvent");
    }

    public boolean isConnected(String uri) {
        synchronized (lock) {
            return peers.containsKey(uri);
        }
    }

    public void syn(NodeInfo info) throws Exception {
        //Profiler.checkpoint("syn");
        log.info("Received SYN from " + info);
        boolean isNew = false;
        PeerEntry entry;
        synchronized (lock) {
            // activate the peer connection
            if (peers.containsKey(info.uri)) {
                // already connected
                log.info("Already connected. Ignoring SYN");
                return;
            }
            entry = detectedNodes.remove(info.uri);
            if (entry == null) { // not yet detected -> create it now
                entry = createPeerEntry(info.uri);
                isNew = true;
            } else {
                entry.info = info; // update info
            }
            peers.put(info.uri, entry);
        } // <<  end synchronized
        if (isNew) { // send SYN
            log.info("Sending SYN to " + info);
            //Profiler.checkpoint("syn.syn");
            entry.peer.syn(this.info);
            //Profiler.print("syn.syn");
        }
        // notifying upper layer
        if (handler != null) {
            handler.peerConnected(info);
        }
        //Profiler.print("syn");
    }

    public void close(String uri) {
        nodeRemoved(uri);
    }

    public void nodeRemoved(String uri) {
        synchronized (lock) {
            PeerEntry entry = peers.remove(uri);
            if (entry == null) {
                entry = detectedNodes.remove(uri);
            }
            if (entry != null) {
                log.info("Node removed: " + uri);
                try {
                    if (handler != null) {
                        handler.peerRemoved(uri);
                    }
                } finally {
                    TransporterClient.destroyTransporterClient(entry.peer);
                }
            }
        } // << end of synchronized
    }

    public PeerEntry nodeAdded(String uri) {
        log.info("Detected node: " + uri);
        PeerEntry entry = null;
        try {
            synchronized (lock) {
                entry = detectedNodes.get(uri);
                if (entry != null) {
                        log.info("Node already detected. Ignoring.");
                        return entry;
                }
                entry = peers.get(uri);
                if (entry != null) {
                    log.info("Node already detected. Ignoring.");
                    return entry;
                }
                entry = createPeerEntry(uri);
                detectedNodes.put(uri, entry);
            } // << end of synchronized
            log.info("Sending SYN to " + uri);
            //Profiler.checkpoint("nodeAdded.syn");
            entry.peer.syn(this.info);
            //Profiler.print("nodeAdded.syn");
        } catch (Exception e) {
            log.error("Failed to add peer node", e);
        }
        return entry;
    }


    protected PeerEntry createPeerEntry(String uri) throws Exception {
        return createPeerEntry(new NodeInfo(uri));
    }

    protected PeerEntry createPeerEntry(NodeInfo info) throws Exception {
        InvokerLocator locator = new InvokerLocator(info.uri);
        //Profiler.checkpoint("createTransporterClient");
        NetworkPeer peer = (NetworkPeer) TransporterClient.createTransporterClient(
                locator, NetworkPeer.class);
        //Profiler.print("createTransporterClient");
        return new PeerEntry(info, peer);
    }


    static class PeerEntry {

        public NodeInfo info;
        public final NetworkPeer peer;

        public PeerEntry(NodeInfo info, NetworkPeer peer) {
           this.info = info;
           this.peer = peer;
        }

    }

}
