/*
 * (C) Copyright 2007 Nuxeo SAS (http://nuxeo.com/) and contributors.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser General Public License
 * (LGPL) version 2.1 which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl.html
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * Contributors:
 *     Nuxeo - initial API and implementation
 *
 * $Id$
 */

package org.nuxeo.runtime.services.streaming;

import java.io.File;

import org.nuxeo.runtime.api.Framework;
import org.nuxeo.runtime.model.ComponentContext;
import org.nuxeo.runtime.model.ComponentInstance;
import org.nuxeo.runtime.model.ComponentName;
import org.nuxeo.runtime.model.DefaultComponent;

/**
 * @author <a href="mailto:bs@nuxeo.com">Bogdan Stefanescu</a>
 *
 */
public class StreamingService extends DefaultComponent{

    public static final ComponentName NAME = new ComponentName("org.nuxeo.runtime.streaming");

    protected StreamManager manager;
    protected ComponentContext context;

    protected boolean isServer;
    protected String serverLocator;
    
    
    @Override
    public void activate(ComponentContext context) throws Exception {
        // start the manager
        isServer = (Boolean) context.getPropertyValue("isServer", false);
        serverLocator = (String) context.getPropertyValue("serverLocator", "socket://127.0.0.1:3233");
        this.context = context;
        startManager();
    }

    @Override
    public void deactivate(ComponentContext context) throws Exception {
        // stop the manager
        stopManager();
        this.context = null;
        super.deactivate(context);
    }
    
    /**
     * @param isServer the isServer to set.
     */
    public void setServer(boolean isServer) {
        this.isServer = isServer;
    }
    
    /**
     * @param serverLocator the serverLocator to set.
     */
    public void setServerLocator(String serverLocator) {
        this.serverLocator = serverLocator;
    }
    
    /**
     * @return the isServer.
     */
    public boolean isServer() {
        return isServer;
    }
    
    /**
     * @return the serverLocator.
     */
    public String getServerLocator() {
        return serverLocator;
    }
    
    /**
     * @return the context.
     */
    public ComponentContext getContext() {
        return context;
    }

    @Override
    public void registerContribution(Object contribution,
            String extensionPoint, ComponentInstance contributor) {
    }

    @Override
    public void unregisterContribution(Object contribution,
            String extensionPoint, ComponentInstance contributor) {
    }

    public StreamManager getStreamManager() {
        return manager;
    }
    
    public synchronized void startManager() throws Exception {
        if (manager != null) {
            throw new IllegalStateException("StreamingManager is already started");
        }
        if (serverLocator == null) {
            throw new IllegalArgumentException(
                    "serverLocator must not be null when defining a StreamingService");
        }

        if (isServer) {
            File tmpDir = new File(Framework.getRuntime().getHome(), "tmp/uploads");
            manager = new StreamManagerServer(serverLocator, tmpDir);
        } else {
            int minBufSize = (Integer) context.getPropertyValue("minBufferSize", 1024*8);
            int maxBufSize = (Integer) context.getPropertyValue("maxBufferSize", 1024*1024*8);
            manager = new StreamManagerClient(serverLocator, minBufSize, maxBufSize);
        }
        manager.start();
    }
    
    public synchronized void stopManager() throws Exception {
        if (manager != null) {
            manager.stop();
            manager = null;
        }
    }
    
    
    @Override
    @SuppressWarnings("unchecked")
    public <T> T getAdapter(Class<T> adapter) {
        if (StreamManager.class.isAssignableFrom(adapter)) {
            return (T) manager;
        }
        return null;
    }

}
