/*
 * Decompiled with CFR 0.152.
 */
package org.nuxeo.elasticsearch.core;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonGenerator;
import io.dropwizard.metrics5.MetricName;
import io.dropwizard.metrics5.MetricRegistry;
import io.dropwizard.metrics5.SharedMetricRegistries;
import io.dropwizard.metrics5.Timer;
import java.io.IOException;
import java.io.OutputStream;
import java.util.HashSet;
import java.util.List;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.elasticsearch.action.bulk.BulkItemResponse;
import org.elasticsearch.action.bulk.BulkRequest;
import org.elasticsearch.action.bulk.BulkResponse;
import org.elasticsearch.action.delete.DeleteRequest;
import org.elasticsearch.action.get.GetRequest;
import org.elasticsearch.action.get.GetResponse;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.action.search.SearchScrollRequest;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.io.stream.BytesStreamOutput;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.common.xcontent.XContentType;
import org.elasticsearch.index.VersionType;
import org.elasticsearch.index.query.ConstantScoreQueryBuilder;
import org.elasticsearch.index.query.QueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.rest.RestStatus;
import org.elasticsearch.search.SearchHit;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.elasticsearch.search.fetch.subphase.FetchSourceContext;
import org.nuxeo.common.logging.SequenceTracer;
import org.nuxeo.ecm.core.api.ConcurrentUpdateException;
import org.nuxeo.ecm.core.api.DocumentModel;
import org.nuxeo.ecm.core.api.DocumentNotFoundException;
import org.nuxeo.ecm.core.api.NuxeoException;
import org.nuxeo.ecm.core.api.model.BlobNotFoundException;
import org.nuxeo.elasticsearch.api.ElasticSearchIndexing;
import org.nuxeo.elasticsearch.commands.IndexingCommand;
import org.nuxeo.elasticsearch.core.ElasticSearchAdminImpl;
import org.nuxeo.elasticsearch.io.JsonESDocumentWriter;
import org.nuxeo.runtime.api.Framework;
import org.nuxeo.runtime.metrics.MetricsService;

public class ElasticSearchIndexingImpl
implements ElasticSearchIndexing {
    private static final Log log = LogFactory.getLog(ElasticSearchIndexingImpl.class);
    private static final int MAX_CURL_LINE = 8192;
    private static final int DEFAULT_MAX_BULK_SIZE = 0x500000;
    private final ElasticSearchAdminImpl esa;
    private final Timer deleteTimer;
    private final Timer indexTimer;
    private final Timer bulkIndexTimer;
    private final boolean useExternalVersion;
    private JsonESDocumentWriter jsonESDocumentWriter;
    protected static final JsonFactory JSON_FACTORY = new JsonFactory();

    public ElasticSearchIndexingImpl(ElasticSearchAdminImpl esa) {
        this.esa = esa;
        MetricRegistry registry = SharedMetricRegistries.getOrCreate((String)MetricsService.class.getName());
        this.indexTimer = registry.timer(MetricName.build((String[])new String[]{"nuxeo.elasticsearch.service.timer"}).tagged(new String[]{"service", "index"}));
        this.deleteTimer = registry.timer(MetricName.build((String[])new String[]{"nuxeo.elasticsearch.service.timer"}).tagged(new String[]{"service", "delete"}));
        this.bulkIndexTimer = registry.timer(MetricName.build((String[])new String[]{"nuxeo.elasticsearch.service.timer"}).tagged(new String[]{"service", "bulkIndex"}));
        this.jsonESDocumentWriter = new JsonESDocumentWriter();
        this.useExternalVersion = esa.useExternalVersion();
    }

    public ElasticSearchIndexingImpl(ElasticSearchAdminImpl esa, JsonESDocumentWriter jsonESDocumentWriter) {
        this(esa);
        this.jsonESDocumentWriter = jsonESDocumentWriter;
    }

    @Override
    public void runIndexingWorker(List<IndexingCommand> cmds) {
        throw new UnsupportedOperationException("Not implemented");
    }

    @Override
    public void runReindexingWorker(String repositoryName, String nxql, boolean syncAlias) {
        throw new UnsupportedOperationException("Not implemented");
    }

    @Override
    public void reindexRepository(String repositoryName) {
        throw new UnsupportedOperationException("Not implemented");
    }

    @Override
    public void indexNonRecursive(List<IndexingCommand> cmds) {
        int nbCommands = cmds.size();
        if (nbCommands == 1) {
            this.indexNonRecursive(cmds.get(0));
            return;
        }
        this.processBulkDeleteCommands(cmds);
        try (Timer.Context ignored = this.bulkIndexTimer.time();){
            this.processBulkIndexCommands(cmds);
        }
        this.esa.totalCommandProcessed.addAndGet(nbCommands);
        this.refreshIfNeeded(cmds);
    }

    void processBulkDeleteCommands(List<IndexingCommand> cmds) {
        for (IndexingCommand cmd : cmds) {
            if (cmd.getType() != IndexingCommand.Type.DELETE) continue;
            Timer.Context ignored = this.deleteTimer.time();
            try {
                this.processDeleteCommand(cmd);
            }
            finally {
                if (ignored == null) continue;
                ignored.close();
            }
        }
    }

    void processBulkIndexCommands(List<IndexingCommand> cmds) {
        BulkRequest bulkRequest = new BulkRequest();
        HashSet<String> docIds = new HashSet<String>(cmds.size());
        int bulkSize = 0;
        int maxBulkSize = this.getMaxBulkSize();
        for (IndexingCommand cmd : cmds) {
            if (cmd.getType() == IndexingCommand.Type.DELETE || cmd.getType() == IndexingCommand.Type.UPDATE_DIRECT_CHILDREN || !docIds.add(cmd.getTargetDocumentId())) continue;
            try {
                IndexRequest idxRequest = this.buildEsIndexingRequest(cmd);
                if (idxRequest != null) {
                    bulkSize += idxRequest.source().length();
                    bulkRequest.add(idxRequest);
                }
            }
            catch (BlobNotFoundException be) {
                log.info((Object)("Ignore indexing command in bulk, blob does not exists anymore: " + cmd));
            }
            catch (ConcurrentUpdateException e) {
                throw e;
            }
            catch (DocumentNotFoundException e) {
                log.info((Object)("Ignore indexing command in bulk, doc does not exists anymore: " + cmd));
            }
            catch (IllegalArgumentException e) {
                log.error((Object)("Ignore indexing command in bulk, fail to create request: " + cmd), (Throwable)e);
            }
            if (bulkSize <= maxBulkSize) continue;
            log.warn((Object)("Max bulk size reached " + bulkSize + ", sending bulk command"));
            this.sendBulkCommand(bulkRequest, bulkSize);
            bulkRequest = new BulkRequest();
            bulkSize = 0;
        }
        this.sendBulkCommand(bulkRequest, bulkSize);
    }

    int getMaxBulkSize() {
        String value = Framework.getProperty((String)"elasticsearch.index.bulkMaxSize", (String)String.valueOf(0x500000));
        return Integer.parseInt(value);
    }

    void sendBulkCommand(BulkRequest bulkRequest, int bulkSize) {
        if (bulkRequest.numberOfActions() > 0) {
            BulkResponse response;
            if (log.isDebugEnabled()) {
                this.logDebugMessageTruncated(String.format("Index %d docs (%d bytes) in bulk request: curl -XPOST 'http://localhost:9200/_bulk' -d '%s'", bulkRequest.numberOfActions(), bulkSize, bulkRequest.requests().toString()), 8192);
            }
            if ((response = this.esa.getClient().bulk(bulkRequest)).hasFailures()) {
                this.logBulkFailure(response);
            }
        }
    }

    void logBulkFailure(BulkResponse response) {
        boolean isError = false;
        StringBuilder sb = new StringBuilder();
        sb.append("Ignore indexing of some docs more recent versions has already been indexed");
        for (BulkItemResponse item : response.getItems()) {
            if (!item.isFailed()) continue;
            if (item.getFailure().getStatus() == RestStatus.CONFLICT) {
                sb.append("\n  ").append(item.getFailureMessage());
                continue;
            }
            isError = true;
        }
        if (isError) {
            log.error((Object)response.buildFailureMessage());
        } else {
            log.debug((Object)sb);
        }
    }

    void refreshIfNeeded(List<IndexingCommand> cmds) {
        for (IndexingCommand cmd : cmds) {
            if (!this.refreshIfNeeded(cmd)) continue;
            return;
        }
    }

    boolean refreshIfNeeded(IndexingCommand cmd) {
        if (cmd.isSync()) {
            this.esa.refresh();
            return true;
        }
        return false;
    }

    @Override
    public void indexNonRecursive(IndexingCommand cmd) {
        IndexingCommand.Type type = cmd.getType();
        if (type == IndexingCommand.Type.UPDATE_DIRECT_CHILDREN) {
            return;
        }
        if (type == IndexingCommand.Type.DELETE) {
            try (Timer.Context ignored = this.deleteTimer.time();){
                this.processDeleteCommand(cmd);
            }
        }
        try (Timer.Context ignored = this.indexTimer.time();){
            this.processIndexCommand(cmd);
        }
        this.refreshIfNeeded(cmd);
        this.esa.totalCommandProcessed.incrementAndGet();
    }

    void processIndexCommand(IndexingCommand cmd) {
        IndexRequest request;
        try {
            request = this.buildEsIndexingRequest(cmd);
        }
        catch (BlobNotFoundException pe) {
            request = null;
        }
        catch (DocumentNotFoundException e) {
            request = null;
        }
        catch (IllegalStateException e) {
            log.error((Object)("Fail to create request for indexing command: " + cmd), (Throwable)e);
            return;
        }
        if (request == null) {
            log.info((Object)("Cancel indexing command because target document does not exists anymore: " + cmd));
            return;
        }
        if (log.isDebugEnabled()) {
            this.logDebugMessageTruncated(String.format("Index request: curl -XPUT 'http://localhost:9200/%s/%s/%s' -d '%s'", this.getWriteIndexForRepository(cmd.getRepositoryName()), "doc", cmd.getTargetDocumentId(), request.toString()), 8192);
        }
        try {
            this.esa.getClient().index(request);
        }
        catch (ConcurrentUpdateException e) {
            SequenceTracer.addNote((String)("Ignore indexing of doc " + cmd.getTargetDocumentId()));
            log.info((Object)("Ignore indexing of doc " + cmd.getTargetDocumentId() + " a more recent version has already been indexed: " + e.getMessage()));
        }
    }

    void logDebugMessageTruncated(String msg, int maxSize) {
        if (log.isTraceEnabled() || msg.length() < maxSize) {
            log.debug((Object)msg);
        } else {
            log.debug((Object)(msg.substring(0, maxSize) + "..."));
        }
    }

    void processDeleteCommand(IndexingCommand cmd) {
        if (cmd.isRecurse()) {
            this.processDeleteCommandRecursive(cmd);
        } else {
            this.processDeleteCommandNonRecursive(cmd);
        }
    }

    void processDeleteCommandNonRecursive(IndexingCommand cmd) {
        String indexName = this.getWriteIndexForRepository(cmd.getRepositoryName());
        DeleteRequest request = new DeleteRequest(indexName, "doc", cmd.getTargetDocumentId());
        if (log.isDebugEnabled()) {
            log.debug((Object)String.format("Delete request: curl -XDELETE 'http://localhost:9200/%s/%s/%s'", indexName, "doc", cmd.getTargetDocumentId()));
        }
        this.esa.getClient().delete(request);
    }

    void processDeleteCommandRecursive(IndexingCommand cmd) {
        String indexName = this.getWriteIndexForRepository(cmd.getRepositoryName());
        String docPath = this.getPathOfDocFromEs(cmd.getRepositoryName(), cmd.getTargetDocumentId());
        if (docPath == null) {
            if (!Framework.isTestModeSet()) {
                log.warn((Object)("Trying to delete a non existing doc: " + cmd.toString()));
            }
            return;
        }
        this.esa.getClient().refresh(indexName);
        ConstantScoreQueryBuilder query = QueryBuilders.constantScoreQuery((QueryBuilder)QueryBuilders.termQuery((String)"ecm:path.children", (String)docPath));
        TimeValue keepAlive = TimeValue.timeValueMinutes((long)1L);
        SearchSourceBuilder search = new SearchSourceBuilder().size(100).query((QueryBuilder)query).fetchSource(false);
        SearchRequest request = new SearchRequest(new String[]{indexName}).scroll(keepAlive).source(search);
        if (log.isDebugEnabled()) {
            log.debug((Object)String.format("Search with scroll request: curl -XGET 'http://localhost:9200/%s/%s/_search?scroll=%s' -d '%s'", indexName, "doc", keepAlive, query.toString()));
        }
        SearchResponse response = this.esa.getClient().search(request);
        while (response.getHits().getHits().length > 0) {
            BulkRequest bulkRequest = new BulkRequest();
            for (SearchHit hit : response.getHits().getHits()) {
                bulkRequest.add(new DeleteRequest(hit.getIndex(), hit.getType(), hit.getId()));
            }
            if (log.isDebugEnabled()) {
                log.debug((Object)String.format("Bulk delete request on %s elements", bulkRequest.numberOfActions()));
            }
            this.esa.getClient().bulk(bulkRequest);
            response = this.runNextScroll(response, keepAlive);
        }
    }

    SearchResponse runNextScroll(SearchResponse response, TimeValue keepAlive) {
        if (log.isDebugEnabled()) {
            log.debug((Object)String.format("Scroll request: -XGET 'localhost:9200/_search/scroll' -d '{\"scroll\": \"%s\", \"scroll_id\": \"%s\" }'", keepAlive, response.getScrollId()));
        }
        SearchScrollRequest request = new SearchScrollRequest(response.getScrollId()).scroll(keepAlive);
        return this.esa.getClient().searchScroll(request);
    }

    String getPathOfDocFromEs(String repository, String docId) {
        GetResponse ret;
        String indexName = this.getWriteIndexForRepository(repository);
        GetRequest request = new GetRequest(indexName, "doc", docId).fetchSourceContext(new FetchSourceContext(true, new String[]{"ecm:path"}, null));
        if (log.isDebugEnabled()) {
            log.debug((Object)String.format("Get path of doc: curl -XGET 'http://localhost:9200/%s/%s/%s?fields=%s'", indexName, "doc", docId, "ecm:path"));
        }
        if (!(ret = this.esa.getClient().get(request)).isExists() || ret.getSource() == null || ret.getSource().get("ecm:path") == null) {
            return null;
        }
        return ret.getSource().get("ecm:path").toString();
    }

    IndexRequest buildEsIndexingRequest(IndexingCommand cmd) {
        DocumentModel doc = cmd.getTargetDocument();
        if (doc == null) {
            return null;
        }
        try {
            IndexRequest request = new IndexRequest(this.getWriteIndexForRepository(cmd.getRepositoryName()), "doc", cmd.getTargetDocumentId()).source(this.source(doc), XContentType.JSON);
            if (this.useExternalVersion && cmd.getOrder() > 0L) {
                request.versionType(VersionType.EXTERNAL).version(cmd.getOrder());
            }
            return request;
        }
        catch (IOException e) {
            throw new NuxeoException("Unable to create index request for Document " + cmd.getTargetDocumentId(), (Throwable)e);
        }
    }

    protected String getWriteIndexForRepository(String repository) {
        return this.esa.getWriteIndexName(this.esa.getIndexNameForRepository(repository));
    }

    @Override
    public BytesReference source(DocumentModel doc) throws IOException {
        BytesStreamOutput out = new BytesStreamOutput();
        try (JsonGenerator jsonGen = JSON_FACTORY.createGenerator((OutputStream)out);){
            this.jsonESDocumentWriter.writeESDocument(jsonGen, doc, null, null);
            BytesReference bytesReference = out.bytes();
            return bytesReference;
        }
    }
}

