/*
 * Decompiled with CFR 0.152.
 */
package org.nuxeo.ecm.core.blob;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileTime;
import java.time.Clock;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.nuxeo.ecm.core.blob.AbstractBlobStore;
import org.nuxeo.ecm.core.blob.BlobStore;
import org.nuxeo.ecm.core.blob.BlobUpdateContext;
import org.nuxeo.ecm.core.blob.BlobWriteContext;
import org.nuxeo.ecm.core.blob.CachingConfiguration;
import org.nuxeo.ecm.core.blob.LocalBlobStore;
import org.nuxeo.ecm.core.blob.PathStrategyFlat;
import org.nuxeo.ecm.core.blob.binary.BinaryGarbageCollector;
import org.nuxeo.ecm.core.blob.binary.BinaryManagerStatus;

public class CachingBlobStore
extends AbstractBlobStore {
    private static final Logger log = LogManager.getLogger(CachingBlobStore.class);
    protected static final Set<Path> LOCKED_FILES = ConcurrentHashMap.newKeySet();
    protected final BlobStore store;
    protected final CachingConfiguration cacheConfig;
    public final LocalBlobStore cacheStore;
    protected final BinaryGarbageCollector gc;
    protected final Lock clearOldBlobsLock = new ReentrantLock();
    protected long clearOldBlobsLastTime;
    protected long clearOldBlobsInterval = Duration.ofMinutes(1L).toMillis();
    protected Clock clock = Clock.systemUTC();

    @Deprecated
    public CachingBlobStore(String name, BlobStore store, CachingConfiguration config) {
        this(null, name, store, config);
    }

    public CachingBlobStore(String blobProviderId, String name, BlobStore store, CachingConfiguration config) {
        super(blobProviderId, name, store.getKeyStrategy());
        this.store = store;
        this.cacheConfig = config;
        this.cacheStore = new LocalBlobStore(name, store.getKeyStrategy(), new PathStrategyFlat(config.dir));
        this.gc = new CachingBinaryGarbageCollector();
    }

    @Override
    public boolean hasVersioning() {
        return this.store.hasVersioning();
    }

    @Override
    public BlobStore unwrap() {
        return this.store.unwrap();
    }

    protected Path renameCachedBlob(String tmpKey, String key) throws IOException {
        return this.copyOrMoveCachedBlob(key, this.cacheStore, tmpKey, true);
    }

    protected Path copyOrMoveCachedBlob(String destKey, LocalBlobStore sourceStore, String sourceKey, boolean atomicMove) throws IOException {
        this.recordBlobAccess(sourceStore, sourceKey);
        String returnedKey = this.cacheStore.copyBlob(destKey, sourceStore, sourceKey, atomicMove);
        if (returnedKey == null) {
            return null;
        }
        if (!returnedKey.equals(destKey)) {
            throw new IllegalStateException("Expected " + destKey + " but was " + returnedKey);
        }
        BlobStore.OptionalOrUnknown<Path> fileOpt = this.cacheStore.getFile(destKey);
        if (!fileOpt.isPresent()) {
            throw new IllegalStateException("File disappeared after copy/move: " + destKey);
        }
        Path path = fileOpt.get();
        this.recordBlobAccess(path);
        this.clearOldBlobs();
        return path;
    }

    @Override
    protected String writeBlobGeneric(BlobWriteContext blobWriteContext) throws IOException {
        String tmpKey = this.cacheStore.writeBlob(blobWriteContext.copyWithKey(this.randomString()));
        String key = blobWriteContext.getKey();
        if (blobWriteContext.useDeDuplication() && this.getFileFromCache(key, true).isPresent()) {
            this.cacheStore.deleteBlob(tmpKey);
            return key;
        }
        BlobStore.OptionalOrUnknown<Path> fileOpt = this.cacheStore.getFile(tmpKey);
        if (!fileOpt.isPresent()) {
            throw new IllegalStateException("File disappeared after write: " + tmpKey);
        }
        blobWriteContext.setFile(fileOpt.get());
        String returnedKey = this.store.writeBlob(blobWriteContext.copyWithNoWriteObserverAndKey(key));
        this.renameCachedBlob(tmpKey, returnedKey);
        return returnedKey;
    }

    @Override
    public String copyOrMoveBlob(String key, BlobStore sourceStore, String sourceKey, boolean atomicMove) throws IOException {
        String returnedKey;
        LocalBlobStore sourceCacheStore;
        LocalBlobStore localBlobStore = sourceCacheStore = sourceStore instanceof CachingBlobStore ? ((CachingBlobStore)sourceStore).cacheStore : null;
        if ((!atomicMove || this.copyBlobIsOptimized(sourceStore)) && sourceCacheStore != null && key != null) {
            this.copyOrMoveCachedBlob(key, sourceCacheStore, sourceKey, atomicMove);
        }
        if ((returnedKey = this.store.copyOrMoveBlob(key, sourceStore, sourceKey, atomicMove)) != null && atomicMove && sourceCacheStore != null) {
            sourceCacheStore.deleteBlob(sourceKey);
        }
        return returnedKey;
    }

    @Override
    public boolean useAsyncDigest() {
        return ((AbstractBlobStore)this.store).useAsyncDigest();
    }

    @Override
    public BlobStore.OptionalOrUnknown<Path> getFile(String key) {
        BlobStore.OptionalOrUnknown<Path> fileOpt = this.getFileFromCache(key, false);
        return fileOpt.isPresent() ? fileOpt : BlobStore.OptionalOrUnknown.unknown();
    }

    protected BlobStore.OptionalOrUnknown<Path> getFileFromCache(String key, boolean exists) {
        this.recordBlobAccess(this.cacheStore, key);
        BlobStore.OptionalOrUnknown<Path> fileOpt = this.cacheStore.getFile(key);
        if (fileOpt.isPresent()) {
            Path path = fileOpt.get();
            long len = path.toFile().length();
            if (exists) {
                this.logTrace("<--", "exists (" + len + " bytes)");
            } else {
                this.logTrace("<-", "read " + len + " bytes");
            }
            this.logTrace("hnote right: " + key);
        } else {
            this.logTrace("<--", "missing");
            this.logTrace("hnote right: " + key);
        }
        return fileOpt;
    }

    @Override
    public BlobStore.OptionalOrUnknown<InputStream> getStream(String key) throws IOException {
        Path path;
        BlobStore.OptionalOrUnknown<Path> fileOpt = this.getFile(key);
        if (fileOpt.isPresent()) {
            path = fileOpt.get();
        } else {
            String tmpKey = this.cacheStore.copyOrMoveBlob(this.randomString(), this.store, key, false);
            if (tmpKey == null) {
                return BlobStore.OptionalOrUnknown.missing();
            }
            path = this.renameCachedBlob(tmpKey, key);
        }
        return BlobStore.OptionalOrUnknown.of(Files.newInputStream(path, new OpenOption[0]));
    }

    @Override
    public boolean readBlob(String key, Path dest) throws IOException {
        BlobStore.OptionalOrUnknown<InputStream> streamOpt = this.getStream(key);
        if (!streamOpt.isPresent()) {
            return false;
        }
        try (InputStream stream = streamOpt.get();){
            Files.copy(stream, dest, StandardCopyOption.REPLACE_EXISTING);
            boolean bl = true;
            return bl;
        }
    }

    @Override
    public void writeBlobProperties(BlobUpdateContext blobUpdateContext) throws IOException {
        this.store.writeBlobProperties(blobUpdateContext);
    }

    @Override
    public void deleteBlob(String key) {
        this.cacheStore.deleteBlob(key);
        this.store.deleteBlob(key);
    }

    @Override
    public void clear() {
        this.cacheStore.clear();
        this.store.clear();
    }

    protected void clearOldBlobs() {
        if (this.clearOldBlobsLock.tryLock()) {
            try {
                if (this.clock.millis() > this.clearOldBlobsLastTime + this.clearOldBlobsInterval) {
                    this.clearOldBlobsNow();
                    this.clearOldBlobsLastTime = this.clock.millis();
                }
            }
            finally {
                this.clearOldBlobsLock.unlock();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected void clearOldBlobsNow() {
        long maxSize = this.cacheConfig.maxSize;
        long maxCount = this.cacheConfig.maxCount;
        long minAgeMillis = this.cacheConfig.minAge * 1000L;
        long threshold = this.clock.millis() - minAgeMillis;
        log.debug("clearOldBlobs starting, dir={} maxSize={}, maxCount={}, minAge={}s, threshold={}", (Object)this.cacheConfig.dir, (Object)maxSize, (Object)maxCount, (Object)this.cacheConfig.minAge, (Object)threshold);
        ArrayList<PathInfo> files = new ArrayList<PathInfo>();
        try (DirectoryStream<Path> ds = Files.newDirectoryStream(this.cacheConfig.dir);){
            for (Path path : ds) {
                PathInfo pi;
                try {
                    pi = new PathInfo(path);
                }
                catch (NoSuchFileException e) {
                    log.trace("clearOldBlobs ignoring missing file: {}", (Object)path);
                    continue;
                }
                catch (IOException e) {
                    log.warn(e.getMessage());
                    continue;
                }
                if (this.cacheStore.pathStrategy.isTempFile(path)) {
                    log.trace("clearOldBlobs ignoring temporary file: {} (timestamp {})", (Object)path, (Object)pi.time);
                    continue;
                }
                files.add(pi);
            }
        }
        catch (IOException e) {
            log.warn(e.getMessage());
        }
        Collections.sort(files);
        log.debug("clearOldBlobs {} files to check", (Object)files.size());
        long size = 0L;
        long count = 0L;
        long recentCount = 0L;
        for (PathInfo pi : files) {
            if (++count > maxCount || (size += pi.size) > maxSize) {
                if (pi.time < threshold) {
                    if (CachingBlobStore.tryLock(pi.path)) {
                        try {
                            long time = Files.getLastModifiedTime(pi.path, new LinkOption[0]).toMillis();
                            if (time < threshold) {
                                log.trace("clearOldBlobs DELETING file: {} (timestamp {}, cumulative count {}, cumulative size {})", (Object)pi.path, (Object)time, (Object)count, (Object)size);
                                Files.delete(pi.path);
                                size -= pi.size;
                                --count;
                                continue;
                            }
                            ++recentCount;
                            log.trace("clearOldBlobs keeping file: {} because it's recent (timestamp {})", (Object)pi.path, (Object)time);
                            continue;
                        }
                        catch (IOException e) {
                            log.warn(e.getMessage());
                            continue;
                        }
                        finally {
                            CachingBlobStore.unlock(pi.path);
                            continue;
                        }
                    }
                    log.trace("clearOldBlobs skipping file: {} because it's already locked", (Object)pi.path);
                    continue;
                }
                ++recentCount;
                log.trace("clearOldBlobs keeping file: {} because it's recent (timestamp {})", (Object)pi.path, (Object)pi.time);
                continue;
            }
            log.trace("clearOldBlobs keeping file: {}", (Object)pi.path);
        }
        if (log.isDebugEnabled()) {
            if (maxSize == 0L) {
                maxSize = 1L;
            }
            log.debug(String.format("clearOldBlobs done (keeping %d files out of %d (including %d recent), cache fill ratio now %.1f%%)", count, files.size(), recentCount, 100.0 * (double)size / (double)maxSize));
        }
    }

    protected void recordBlobAccess(LocalBlobStore localBlobStore, String key) {
        this.recordBlobAccess(localBlobStore.pathStrategy.getPathForKey(key));
    }

    protected void recordBlobAccess(Path path) {
        if (CachingBlobStore.tryLock(path)) {
            try {
                Files.setLastModifiedTime(path, FileTime.fromMillis(this.clock.millis()));
            }
            catch (NoSuchFileException noSuchFileException) {
            }
            catch (IOException e) {
                log.error((Object)e, (Throwable)e);
            }
            finally {
                CachingBlobStore.unlock(path);
            }
        }
    }

    protected static boolean tryLock(Path path) {
        long millis = 1L;
        for (int i = 0; i < 20; ++i) {
            if (LOCKED_FILES.add(path)) {
                return true;
            }
            if (i >= 10) {
                millis *= 2L;
            }
            try {
                Thread.sleep(millis);
                continue;
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new RuntimeException(e);
            }
        }
        return false;
    }

    protected static void unlock(Path path) {
        LOCKED_FILES.remove(path);
    }

    @Override
    public BinaryGarbageCollector getBinaryGarbageCollector() {
        return this.gc;
    }

    public class CachingBinaryGarbageCollector
    implements BinaryGarbageCollector {
        protected final BinaryGarbageCollector delegate;
        protected final BinaryGarbageCollector cacheDelegate;

        public CachingBinaryGarbageCollector() {
            this.delegate = CachingBlobStore.this.store.getBinaryGarbageCollector();
            this.cacheDelegate = CachingBlobStore.this.cacheStore.getBinaryGarbageCollector();
        }

        @Deprecated
        public CachingBinaryGarbageCollector(BinaryGarbageCollector delegate) {
            this();
        }

        @Override
        public String getId() {
            return this.delegate.getId();
        }

        @Override
        public void start() {
            this.delegate.start();
            this.cacheDelegate.start();
        }

        @Override
        public void mark(String key) {
            this.delegate.mark(key);
            this.cacheDelegate.mark(key);
        }

        @Override
        public void stop(boolean delete) {
            this.delegate.stop(delete);
            this.cacheDelegate.stop(delete);
        }

        @Override
        public BinaryManagerStatus getStatus() {
            return this.delegate.getStatus();
        }

        @Override
        public boolean isInProgress() {
            return this.delegate.isInProgress();
        }
    }

    protected static class PathInfo
    implements Comparable<PathInfo> {
        protected final Path path;
        protected final long time;
        protected final long size;

        public PathInfo(Path path) throws IOException {
            this.path = path;
            this.time = Files.getLastModifiedTime(path, new LinkOption[0]).toMillis();
            this.size = Files.size(path);
        }

        @Override
        public int compareTo(PathInfo other) {
            return Long.compare(other.time, this.time);
        }
    }
}

