/*
 * Decompiled with CFR 0.152.
 */
package org.nuxeo.runtime.kv;

import java.nio.ByteBuffer;
import java.nio.charset.CharacterCodingException;
import java.nio.charset.CharsetDecoder;
import java.nio.charset.CodingErrorAction;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import javax.inject.Inject;
import org.junit.Assert;
import org.junit.Assume;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.nuxeo.runtime.kv.KeyValueService;
import org.nuxeo.runtime.kv.KeyValueStore;
import org.nuxeo.runtime.kv.KeyValueStoreProvider;
import org.nuxeo.runtime.test.runner.Deploy;
import org.nuxeo.runtime.test.runner.Features;
import org.nuxeo.runtime.test.runner.FeaturesRunner;
import org.nuxeo.runtime.test.runner.RuntimeFeature;
import org.nuxeo.runtime.test.runner.TransactionalFeature;
import org.nuxeo.runtime.transaction.TransactionHelper;

@RunWith(value=FeaturesRunner.class)
@Features(value={RuntimeFeature.class, TransactionalFeature.class})
@Deploy(value={"org.nuxeo.runtime.kv"})
public abstract class AbstractKeyValueStoreTest {
    protected static final String BAR = "bar";
    protected static final String GEE = "gee";
    protected static final String MOO = "moo";
    protected static final String ZAP = "zap";
    protected static final byte[] BAR_B = "bar".getBytes();
    protected static final byte[] GEE_B = "gee".getBytes();
    protected static final byte[] MOO_B = "moo".getBytes();
    protected static final byte[] NOT_UTF_8 = new byte[]{-128, 0};
    @Inject
    protected KeyValueService keyValueService;
    protected KeyValueStoreProvider store;

    @Before
    public void setUp() {
        this.store = (KeyValueStoreProvider)this.keyValueService.getKeyValueStore("default");
        this.store.clear();
    }

    protected boolean hasSlowTTLExpiration() {
        return false;
    }

    protected void sleepForTTLExpiration() {
    }

    protected Set<String> storeKeys() {
        return this.store.keyStream().collect(Collectors.toSet());
    }

    @Test
    public void testCopyDoesNotShareData() {
        KeyValueStore otherStore = this.keyValueService.getKeyValueStore("notregistered");
        String key = "mykey";
        this.store.put(key, "foo");
        Assert.assertNotNull((Object)this.store.getString(key));
        Assert.assertNull((Object)otherStore.getString(key));
    }

    @Test
    public void testPutGet() {
        String key = "foo";
        Assert.assertEquals(Collections.emptySet(), this.storeKeys());
        Assert.assertNull((Object)this.store.get(key));
        Assert.assertNull((Object)this.store.getString(key));
        Assert.assertNull((Object)this.store.getLong(key));
        this.store.put(key, (byte[])null);
        Assert.assertNull((Object)this.store.get(key));
        Assert.assertEquals(Collections.emptySet(), this.storeKeys());
        this.store.put(key, BAR_B);
        Assert.assertEquals((Object)BAR, (Object)new String(this.store.get(key)));
        Assert.assertEquals((Object)BAR, (Object)this.store.getString(key));
        Assert.assertEquals(Collections.singleton(key), this.storeKeys());
        this.store.put(key, GEE_B);
        Assert.assertEquals((Object)GEE, (Object)new String(this.store.get(key)));
        Assert.assertEquals((Object)GEE, (Object)this.store.getString(key));
        Assert.assertEquals(Collections.singleton(key), this.storeKeys());
        this.store.put(key, MOO);
        Assert.assertEquals((Object)MOO, (Object)new String(this.store.get(key)));
        Assert.assertEquals((Object)MOO, (Object)this.store.getString(key));
        Assert.assertEquals(Collections.singleton(key), this.storeKeys());
        try {
            this.store.getLong(key);
            Assert.fail((String)"shouldn't allow to get long");
        }
        catch (NumberFormatException numberFormatException) {
            // empty catch block
        }
        byte[] bytes = ZAP.getBytes();
        this.store.put(key, bytes);
        bytes[0] = 99;
        byte[] bytes2 = this.store.get(key);
        Assert.assertEquals((Object)ZAP, (Object)new String(bytes2));
        bytes2[0] = 99;
        Assert.assertEquals((Object)ZAP, (Object)new String(this.store.get(key)));
        this.store.put(key, (String)null);
        Assert.assertNull((Object)this.store.get(key));
        Assert.assertEquals(Collections.emptySet(), this.storeKeys());
        this.store.put(key, Long.valueOf(-123L));
        Assert.assertEquals((Object)-123L, (Object)this.store.getLong(key));
    }

    @Test
    public void testNotTransactional() {
        String key = "foo";
        String value = BAR;
        TransactionHelper.runInTransaction(() -> {
            this.store.put(key, value);
            Assert.assertEquals((Object)value, (Object)this.store.getString(key));
            TransactionHelper.setTransactionRollbackOnly();
        });
        Assert.assertEquals((Object)value, (Object)this.store.getString(key));
        TransactionHelper.runInTransaction(() -> Assert.assertEquals((Object)value, (Object)this.store.getString(key)));
    }

    @Test
    public void testReadFromBytes() {
        String key = "foo";
        this.store.put(key, (byte[])null);
        this.checkReadNull(key);
        this.store.put(key, "123456789".getBytes(StandardCharsets.UTF_8));
        this.checkReadLong(key, 123456789L);
        this.store.put(key, "ABC".getBytes(StandardCharsets.UTF_8));
        this.checkReadString(key, "ABC");
        this.store.put(key, NOT_UTF_8);
        this.checkReadBytes(key, NOT_UTF_8);
    }

    @Test
    public void testReadFromString() {
        String key = "foo";
        this.store.put(key, (String)null);
        this.checkReadNull(key);
        this.store.put(key, "123456789");
        this.checkReadLong(key, 123456789L);
        this.store.put(key, "ABC");
        this.checkReadString(key, "ABC");
    }

    @Test
    public void testReadFromLong() {
        String key = "foo";
        this.store.put(key, (Long)null);
        this.checkReadNull(key);
        this.store.put(key, Long.valueOf(123456789L));
        this.checkReadLong(key, 123456789L);
    }

    protected void checkReadNull(String key) {
        Assert.assertNull((Object)this.store.get(key));
        Assert.assertNull((Object)this.store.getString(key));
        Assert.assertNull((Object)this.store.getLong(key));
        Set<String> keys = Collections.singleton(key);
        Assert.assertTrue((boolean)this.store.get(keys).isEmpty());
        Assert.assertTrue((boolean)this.store.getStrings(keys).isEmpty());
        Assert.assertTrue((boolean)this.store.getLongs(keys).isEmpty());
    }

    protected void checkReadLong(String key, long value) {
        Long longLong = value;
        String longString = longLong.toString();
        Assert.assertArrayEquals((byte[])longString.getBytes(StandardCharsets.UTF_8), (byte[])this.store.get(key));
        Assert.assertEquals((Object)longString, (Object)this.store.getString(key));
        Assert.assertEquals((Object)longLong, (Object)this.store.getLong(key));
        Set<String> keys = Collections.singleton(key);
        Map bytesMap = this.store.get(keys);
        Assert.assertEquals((long)1L, (long)bytesMap.size());
        Assert.assertEquals((Object)key, bytesMap.keySet().iterator().next());
        Assert.assertArrayEquals((byte[])longString.getBytes(StandardCharsets.UTF_8), (byte[])((byte[])bytesMap.values().iterator().next()));
        Assert.assertEquals(Collections.singletonMap(key, longString), (Object)this.store.getStrings(keys));
        Assert.assertEquals(Collections.singletonMap(key, longLong), (Object)this.store.getLongs(keys));
    }

    protected void checkReadString(String key, String string) {
        Assert.assertArrayEquals((byte[])string.getBytes(StandardCharsets.UTF_8), (byte[])this.store.get(key));
        Assert.assertEquals((Object)string, (Object)this.store.getString(key));
        try {
            this.store.getLong(key);
            Assert.fail((String)"should fail reading a non-numeric value");
        }
        catch (NumberFormatException numberFormatException) {
            // empty catch block
        }
        Set<String> keys = Collections.singleton(key);
        Map bytesMap = this.store.get(keys);
        Assert.assertEquals((long)1L, (long)bytesMap.size());
        Assert.assertEquals((Object)key, bytesMap.keySet().iterator().next());
        Assert.assertArrayEquals((byte[])string.getBytes(StandardCharsets.UTF_8), (byte[])((byte[])bytesMap.values().iterator().next()));
        Assert.assertEquals(Collections.singletonMap(key, string), (Object)this.store.getStrings(keys));
        try {
            this.store.getLongs(keys);
            Assert.fail((String)"should fail reading a non-numeric value");
        }
        catch (NumberFormatException numberFormatException) {
            // empty catch block
        }
    }

    protected void checkReadBytes(String key, byte[] bytes) {
        Assert.assertArrayEquals((byte[])bytes, (byte[])this.store.get(key));
        try {
            this.store.getString(key);
            Assert.fail((String)"should fail reading a non-UTF-8 value");
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
        try {
            this.store.getLong(key);
            Assert.fail((String)"should fail reading a non-numeric value");
        }
        catch (NumberFormatException numberFormatException) {
            // empty catch block
        }
        Set<String> keys = Collections.singleton(key);
        Map bytesMap = this.store.get(keys);
        Assert.assertEquals((long)1L, (long)bytesMap.size());
        Assert.assertEquals((Object)key, bytesMap.keySet().iterator().next());
        Assert.assertArrayEquals((byte[])bytes, (byte[])((byte[])bytesMap.values().iterator().next()));
        try {
            this.store.getStrings(keys);
            Assert.fail((String)"should fail reading a non-UTF-8 value");
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
        try {
            this.store.getLongs(keys);
            Assert.fail((String)"should fail reading a non-numeric value");
        }
        catch (NumberFormatException numberFormatException) {
            // empty catch block
        }
    }

    @Test
    public void testGetMany() {
        Assert.assertTrue((boolean)this.store.get(Collections.emptyList()).isEmpty());
        Assert.assertTrue((boolean)this.store.getStrings(Collections.emptyList()).isEmpty());
        String key1 = "foo1";
        String key2 = "foo2";
        String key3 = "foo3";
        String key4 = "foo4";
        String key5 = "foo5";
        HashSet<String> keys = new HashSet<String>(Arrays.asList(key1, key2, key3, key4, key5));
        Assert.assertTrue((boolean)this.store.get(keys).isEmpty());
        HashMap<String, String> map = new HashMap<String, String>();
        map.put(key1, BAR);
        map.put(key2, GEE);
        map.put(key3, MOO);
        this.store.put(key1, BAR);
        this.store.put(key2, GEE);
        this.store.put(key3, MOO);
        this.store.put(key4, (String)null);
        Assert.assertEquals(map, (Object)this.store.getStrings(keys));
        this.store.put(key1, BAR_B);
        this.store.put(key2, GEE_B);
        this.store.put(key3, MOO_B);
        Map storeBMap = this.store.get(keys);
        Assert.assertArrayEquals((byte[])BAR_B, (byte[])((byte[])storeBMap.get(key1)));
        Assert.assertArrayEquals((byte[])GEE_B, (byte[])((byte[])storeBMap.get(key2)));
        Assert.assertArrayEquals((byte[])MOO_B, (byte[])((byte[])storeBMap.get(key3)));
        Assert.assertEquals((long)3L, (long)storeBMap.entrySet().size());
    }

    @Test
    public void testGetManyLong() {
        Assert.assertTrue((boolean)this.store.getLongs(Collections.emptyList()).isEmpty());
        String key1 = "foo1";
        String key2 = "foo2";
        String key3 = "foo3";
        String key4 = "foo4";
        String key5 = "foo5";
        HashSet<String> keys = new HashSet<String>(Arrays.asList(key1, key2, key3, key4, key5));
        Assert.assertTrue((boolean)this.store.get(keys).isEmpty());
        HashMap<String, Long> map = new HashMap<String, Long>();
        map.put(key1, 1L);
        map.put(key2, 2L);
        map.put(key3, 3L);
        this.store.put(key1, Long.valueOf(1L));
        this.store.put(key2, Long.valueOf(2L));
        this.store.put(key3, Long.valueOf(3L));
        this.store.put(key4, (Long)null);
        Assert.assertEquals(map, (Object)this.store.getLongs(keys));
        this.store.put(key1, "notalong");
        try {
            this.store.getLongs(keys);
            Assert.fail((String)"shouldn't allow to get longs");
        }
        catch (NumberFormatException numberFormatException) {
            // empty catch block
        }
    }

    @Test
    public void testCompareAndSet() {
        String key = "foo";
        Assert.assertFalse((boolean)this.store.compareAndSet(key, BAR_B, GEE_B));
        Assert.assertNull((Object)this.store.get(key));
        Assert.assertFalse((boolean)this.store.compareAndSet(key, BAR_B, null));
        Assert.assertNull((Object)this.store.get(key));
        Assert.assertTrue((boolean)this.store.compareAndSet(key, (byte[])null, (byte[])null));
        Assert.assertNull((Object)this.store.get(key));
        Assert.assertTrue((boolean)this.store.compareAndSet(key, null, BAR_B));
        Assert.assertEquals((Object)BAR, (Object)new String(this.store.get(key)));
        Assert.assertFalse((boolean)this.store.compareAndSet(key, GEE_B, MOO_B));
        Assert.assertEquals((Object)BAR, (Object)new String(this.store.get(key)));
        Assert.assertFalse((boolean)this.store.compareAndSet(key, null, GEE));
        Assert.assertEquals((Object)BAR, (Object)new String(this.store.get(key)));
        Assert.assertFalse((boolean)this.store.compareAndSet(key, (String)null, (String)null));
        Assert.assertEquals((Object)BAR, (Object)new String(this.store.get(key)));
        Assert.assertTrue((boolean)this.store.compareAndSet(key, BAR, GEE));
        Assert.assertEquals((Object)GEE, (Object)new String(this.store.get(key)));
        byte[] bytes = ZAP.getBytes();
        Assert.assertTrue((boolean)this.store.compareAndSet(key, GEE_B, bytes));
        bytes[0] = 99;
        Assert.assertEquals((Object)ZAP, (Object)new String(this.store.get(key)));
        Assert.assertTrue((boolean)this.store.compareAndSet(key, ZAP.getBytes(), null));
        Assert.assertNull((Object)this.store.get(key));
        bytes = ZAP.getBytes();
        Assert.assertTrue((boolean)this.store.compareAndSet(key, null, bytes));
        bytes[0] = 99;
        Assert.assertEquals((Object)ZAP, (Object)new String(this.store.get(key)));
        this.store.put(key, "0");
        Assert.assertTrue((boolean)this.store.compareAndSet(key, "0", "1"));
        Assert.assertTrue((boolean)this.store.compareAndSet(key, "1".getBytes(), "2".getBytes()));
    }

    @Test
    public void testBinaryValuesAreAccepted() {
        String key = "foo";
        byte[] value = new byte[256];
        for (int i = 0; i < value.length; ++i) {
            value[i] = (byte)i;
        }
        try {
            CharsetDecoder cd = StandardCharsets.UTF_8.newDecoder().onMalformedInput(CodingErrorAction.REPORT).onUnmappableCharacter(CodingErrorAction.REPORT);
            cd.decode(ByteBuffer.wrap(value));
            Assert.fail((String)"bytes should be invalid UTF-8");
        }
        catch (CharacterCodingException cd) {
            // empty catch block
        }
        this.store.put(key, value);
        Assert.assertArrayEquals((byte[])value, (byte[])this.store.get(key));
        try {
            this.store.getString(key);
            Assert.fail((String)"Shoudl fail to get value as a String");
        }
        catch (IllegalArgumentException e) {
            Assert.assertEquals((Object)"Value is not a String for key: foo", (Object)e.getMessage());
        }
    }

    @Test
    public void testClear() {
        String key = "foo";
        this.store.put(key, BAR_B);
        Assert.assertEquals((Object)BAR, (Object)new String(this.store.get(key)));
        Assert.assertEquals(Collections.singleton(key), this.storeKeys());
        this.store.clear();
        Assert.assertNull((Object)this.store.get(key));
        Assert.assertEquals(Collections.emptySet(), this.storeKeys());
    }

    @Test
    public void testTTL() throws Exception {
        Assume.assumeFalse((String)"Ignored because of slow TTL expiration", (boolean)this.hasSlowTTLExpiration());
        int longTTL = 30;
        int shortTTL = 3;
        String key = "foo";
        Assert.assertFalse((boolean)this.store.setTTL(key, 0L));
        Assert.assertFalse((boolean)this.store.setTTL(key, (long)longTTL));
        this.store.put(key, BAR_B, (long)longTTL);
        Assert.assertEquals((Object)BAR, (Object)new String(this.store.get(key)));
        Assert.assertTrue((boolean)this.store.setTTL(key, (long)shortTTL));
        Thread.sleep((shortTTL + 2) * 1000);
        this.sleepForTTLExpiration();
        Assert.assertNull((Object)this.store.get(key));
        this.store.put(key, BAR, (long)shortTTL);
        this.store.setTTL(key, 0L);
        Thread.sleep((shortTTL + 2) * 1000);
        this.sleepForTTLExpiration();
        Assert.assertEquals((Object)BAR, (Object)new String(this.store.get(key)));
        Assert.assertTrue((boolean)this.store.compareAndSet(key, BAR, GEE, (long)shortTTL));
        Assert.assertEquals((Object)GEE, (Object)this.store.getString(key));
        Thread.sleep((shortTTL + 2) * 1000);
        this.sleepForTTLExpiration();
        Assert.assertNull((Object)this.store.get(key));
        Assert.assertTrue((boolean)this.store.compareAndSet(key, null, MOO, (long)shortTTL));
        Assert.assertEquals((Object)MOO, (Object)this.store.getString(key));
        Thread.sleep((shortTTL + 2) * 1000);
        this.sleepForTTLExpiration();
        Assert.assertNull((Object)this.store.get(key));
    }

    @Test
    public void testAddAndGet() throws Exception {
        String key = "foo";
        Assert.assertNull((Object)this.store.get(key));
        Assert.assertEquals((long)123L, (long)this.store.addAndGet(key, 123L));
        Assert.assertEquals((long)579L, (long)this.store.addAndGet(key, 456L));
        Assert.assertEquals((long)-421L, (long)this.store.addAndGet(key, -1000L));
        Assert.assertEquals((long)0L, (long)this.store.addAndGet(key, 421L));
        this.store.put(key, "123");
        Assert.assertEquals((long)456L, (long)this.store.addAndGet(key, 333L));
        Assert.assertEquals((Object)"456", (Object)this.store.getString(key));
        Assert.assertEquals((Object)456L, (Object)this.store.getLong(key));
        this.store.put(key, "");
        try {
            this.store.addAndGet(key, 1L);
        }
        catch (NumberFormatException numberFormatException) {
            // empty catch block
        }
        this.store.put(key, "ABC");
        try {
            this.store.addAndGet(key, 1L);
            Assert.fail((String)"shouldn't allow incrementing a non-numeric string");
        }
        catch (NumberFormatException numberFormatException) {
            // empty catch block
        }
    }

    @Test
    public void testKeyStream() throws Exception {
        this.store.put("foo", "test");
        this.store.put("foox", "test");
        this.store.put("foo?", "test");
        this.store.put("foo?a", "test");
        this.store.put("foo*", "test");
        this.store.put("foo*b", "test");
        this.store.put(BAR, "test");
        this.store.put("barx", "test");
        this.store.put("bar.", "test");
        this.store.put("bar.1", (String)null);
        this.store.put("bar.2", (byte[])null);
        this.store.put("bar.3", (Long)null);
        this.store.put("bar.4", "test");
        this.store.put("bar.5", "test".getBytes(StandardCharsets.UTF_8));
        this.store.put("bar.6", Long.valueOf(123L));
        this.store.put("baz", "test");
        this.store.put("bazx", "test");
        this.store.put("baz%", "test");
        this.store.put("baz%7", "test");
        String prefix = "foo?";
        List<String> expected = Arrays.asList("foo?", "foo?a");
        Assert.assertEquals(new HashSet<String>(expected), this.store.keyStream(prefix).collect(Collectors.toSet()));
        prefix = "foo*";
        expected = Arrays.asList("foo*", "foo*b");
        Assert.assertEquals(new HashSet<String>(expected), this.store.keyStream(prefix).collect(Collectors.toSet()));
        prefix = "bar.";
        expected = Arrays.asList("bar.", "bar.4", "bar.5", "bar.6");
        Assert.assertEquals(new HashSet<String>(expected), this.store.keyStream(prefix).collect(Collectors.toSet()));
        prefix = "baz%";
        expected = Arrays.asList("baz%", "baz%7");
        Assert.assertEquals(new HashSet<String>(expected), this.store.keyStream(prefix).collect(Collectors.toSet()));
    }
}

