/**
 * Copyright sp42 frank@ajaxjs.com
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.ajaxjs.framework.mock;

import java.sql.Connection;

import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.sql.DataSource;

import org.sqlite.SQLiteDataSource;
import org.sqlite.SQLiteJDBCLoader;

import com.ajaxjs.jdbc.JdbcConnection;
import com.ajaxjs.util.io.resource.Scanner;
import com.mysql.jdbc.jdbc2.optional.MysqlDataSource;

/**
 * Mock data source object.
 * @author Sp42 frank@ajaxjs.com
 *
 */
public class MockDataSource {
	/**
	 * 测试用数据库（SQLite）
	 */
	public static final String testUsed_sqlite = Scanner.getResourceFilePath(News.class, "foo.sqlite");

	/**
	 * JDNI 是一个为 Java 应用程序提供命名服务的应用程序接口
	 * https://www.cnblogs.com/zhchoutai/p/7389089.html
	 * 
	 * @return
	 */
	private static InitialContext initIc() {
		System.setProperty(Context.INITIAL_CONTEXT_FACTORY, "org.apache.naming.java.javaURLContextFactory");
		System.setProperty(Context.URL_PKG_PREFIXES, "org.apache.naming");

		try {
			InitialContext ic = new InitialContext();
			ic.createSubcontext("java:");
			ic.createSubcontext("java:/comp");
			ic.createSubcontext("java:/comp/env");
			ic.createSubcontext("java:/comp/env/jdbc");
			return ic;
		} catch (NamingException e) {
			e.printStackTrace();
			return null;
		}
	}

	/**
	 * Return a specified data source object by JDNI path.
	 * 
	 * @param db_filePath
	 *            SQLite 磁盘路径
	 * @return data source object
	 */
	public static DataSource getDataSource(String db_filePath) {
		try {
			SQLiteJDBCLoader.initialize();
		} catch (Exception e) {
			e.printStackTrace();
		}

		SQLiteDataSource dataSource = new SQLiteDataSource();
		dataSource.setUrl("jdbc:sqlite:" + db_filePath);

		return dataSource;
	}

	/**
	 * Return test-uesd data source
	 * 
	 * @return data source object
	 */
	public static DataSource getDataSource() {
		return getDataSource(testUsed_sqlite);
	}

	public static Connection getTestConnection() {
		return JdbcConnection.getConnection(getDataSource(testUsed_sqlite));
	}

	public static Connection getTestMySqlConnection(String url, String user, String password) {
		return JdbcConnection.getConnection(getMySqlDataSource(url, user, password));
	}

	/**
	 * 模拟数据库链接的配置 需要加入 tomcat-juli.jar 这个包，tomcat7 此包位于 tomcat 根目录的 bin 下。
	 */
	public static void initSqliteDBConnection(String db_filePath) {
		try {
			initIc().bind("java:/comp/env/jdbc/sqlite", getDataSource(db_filePath));
		} catch (NamingException e) {
			e.printStackTrace();
		}
	}

	/**
	 * 
	 * @param url
	 *            "jdbc:mysql://localhost:3306/databaseName"
	 * @param user
	 * @param password
	 */
	public static DataSource getMySqlDataSource(String url, String user, String password) {
		MysqlDataSource dataSource = new MysqlDataSource();
		dataSource.setURL(url);
		dataSource.setUser(user);
		dataSource.setPassword(password);
		return dataSource;
		//		try {
		//			initIc().bind("java:/comp/env/jdbc/mysql_deploy", dataSource);
		//			return dataSource;
		//		} catch (NamingException e) {
		//			e.printStackTrace();
		//			return null;
		//		}
	}
}
