/*
 * Copyright (C) Jerry Huxtable 1998
 */
package com.alkacon.simapi.filter;

import com.alkacon.simapi.filter.math.*;

import java.awt.image.*;
import java.awt.*;
import java.io.*;
import java.util.*;

interface ElevationMap {
	int getHeightAt(int x, int y);
}

public class LightFilter extends WholeImageFilter implements Serializable {
	
	public final static int COLORS_FROM_IMAGE = 0;
	public final static int COLORS_CONSTANT = 1;
	public final static int COLORS_FROM_ENVIRONMENT = 2;

	public final static int BUMPS_FROM_IMAGE = 0;
	public final static int BUMPS_FROM_MAP = 1;
	public final static int BUMPS_FROM_BEVEL = 2;

	private float bumpHeight;
	private float viewDistance = 10000.0f;
	Material material;
	private Vector lights;
	int diffuseColor;
	int specularColor;
	private int colorSource = COLORS_FROM_IMAGE;
	private int bumpSource = BUMPS_FROM_IMAGE;
	private Function2D bumpFunction;
	private Image environmentMap;
	private int[] envPixels;
	private int envWidth = 1, envHeight = 1;
	private Vector3D l;
	private Vector3D v;
	private Vector3D n;
	private ARGB shadedColor;
	private ARGB diffuse_color;
	private ARGB specular_color;
	private Vector3D tmpv, tmpv2;
	public NormalEvaluator normalEvaluator = new NormalEvaluator();//FIXME

	public LightFilter() {
		lights = new Vector();//fixme
//		addLight(new AmbientLight());//fixme
		addLight(new DistantLight());
		bumpHeight= 1.0f;
		material = new Material();
		diffuseColor = -1;
		specularColor = -1;
		l = new Vector3D();
		v = new Vector3D();
		n = new Vector3D();
		shadedColor = new ARGB();
		diffuse_color = new ARGB();
		specular_color = new ARGB();
		tmpv = new Vector3D();
		tmpv2 = new Vector3D();
	}

	public void setBumpFunction(Function2D bumpFunction) {
		this.bumpFunction = bumpFunction;
	}

	public Function2D getBumpFunction() {
		return bumpFunction;
	}

	public void setBumpHeight(float bumpHeight) {
		this.bumpHeight= bumpHeight;
	}

	public float getBumpHeight() {
		return bumpHeight;
	}

	public void setViewDistance(float viewDistance) {
		this.viewDistance = viewDistance;
	}

	public float getViewDistance() {
		return viewDistance;
	}

	public void setDiffuseColor(int diffuseColor) {
		this.diffuseColor = diffuseColor;
	}

	public int getDiffuseColor() {
		return diffuseColor;
	}

	public void setEnvironmentMap(Image environmentMap) {
		this.environmentMap = environmentMap;
		if (environmentMap != null) {
			PixelGrabber pg = new PixelGrabber(environmentMap, 0, 0, -1, -1, null, 0, -1);
			try {
				pg.grabPixels();
			} catch (InterruptedException e) {
				throw new RuntimeException("interrupted waiting for pixels!");
			}
			if ((pg.status() & ImageObserver.ABORT) != 0) {
				throw new RuntimeException("image fetch aborted");
			}
			envPixels = (int[])pg.getPixels();
			envWidth = pg.getWidth();
			envHeight = pg.getHeight();
		} else {
			envWidth = envHeight = 1;
			envPixels = null;
		}
	}

	public Image getEnvironmentMap() {
		return environmentMap;
	}

	public void setColorSource(int colorSource) {
		this.colorSource = colorSource;
	}

	public int getColorSource() {
		return colorSource;
	}

	public void setBumpSource(int bumpSource) {
		this.bumpSource = bumpSource;
	}

	public int getBumpSource() {
		return bumpSource;
	}

	public void addLight(Light light) {
		lights.addElement(light);
	}
	
	public void removeLight(Light light) {
		lights.removeElement(light);
	}
	
	public Vector getLights() {
		return lights;
	}
	
	public void imageComplete(int status) {
		if (status == 1 || status == 4) {
			consumer.imageComplete(status);
			return;
		}
		int width = transformedSpace.width;
		int height = transformedSpace.height;
		int index = 0;
		int[] outPixels = new int[width * height];
		float width45 = Math.abs(6.0f * bumpHeight);
		boolean invertBumps = bumpHeight < 0;
		float Nz = 1530.0f / width45;
		Vector3D position = new Vector3D(0.0f, 0.0f, 0.0f);
		Vector3D viewpoint = new Vector3D((float)width / 2.0f, (float)height / 2.0f, viewDistance);
		Vector3D normal = new Vector3D(0.0f, 0.0f, Nz);
		ARGB diffuseColor = new ARGB(this.diffuseColor);
		ARGB specularColor = new ARGB(this.specularColor);
		Function2D bump = bumpFunction;
		if (bumpSource == BUMPS_FROM_IMAGE || bump == null)//FIXME-creates image function for bevels
			bump = new ImageFunction2D(inPixels, width, height, ImageFunction2D.CLAMP);

		Vector3D v1 = new Vector3D();
		Vector3D v2 = new Vector3D();
		Vector3D n = new Vector3D();
		Light[] lightsArray = new Light[lights.size()];
		lights.copyInto(lightsArray);
		for (int i = 0; i < lightsArray.length; i++)
			lightsArray[i].prepare(width, height);

		// Loop through each source pixel
		for (int y = 0; y < height; y++) {
			float ny = y;
			position.y = y;
			for (int x = 0; x < width; x++) {
				float nx = x;
				
				// Calculate the normal at this point
				if (bumpSource != BUMPS_FROM_BEVEL) {
					// Complicated and slower method
					// Calculate four normals using the gradients in +/- X/Y directions
					int count = 0;
					normal.x = normal.y = normal.z = 0;
					float m0 = width45*bump.evaluate(nx, ny);
					float m1 = x > 0 ? width45*bump.evaluate(nx - 1.0f, ny)-m0 : -2;
					float m2 = y > 0 ? width45*bump.evaluate(nx, ny - 1.0f)-m0 : -2;
					float m3 = x < width-1 ? width45*bump.evaluate(nx + 1.0f, ny)-m0 : -2;
					float m4 = y < height-1 ? width45*bump.evaluate(nx, ny + 1.0f)-m0 : -2;
					
					if (m1 != -2 && m4 != -2) {
						v1.x = -1.0f; v1.y = 0.0f; v1.z = m1;
						v2.x = 0.0f; v2.y = 1.0f; v2.z = m4;
						v1.crossProduct(v2, n);
						n.normalize();
						if (n.z < 0.0)
							n.z = -n.z;
						normal.add(n);
						count++;
					}

					if (m1 != -2 && m2 != -2) {
						v1.x = -1.0f; v1.y = 0.0f; v1.z = m1;
						v2.x = 0.0f; v2.y = -1.0f; v2.z = m2;
						v1.crossProduct(v2, n);
						n.normalize();
						if (n.z < 0.0)
							n.z = -n.z;
						normal.add(n);
						count++;
					}

					if (m2 != -2 && m3 != -2) {
						v1.x = 0.0f; v1.y = -1.0f; v1.z = m2;
						v2.x = 1.0f; v2.y = 0.0f; v2.z = m3;
						v1.crossProduct(v2, n);
						n.normalize();
						if (n.z < 0.0)
							n.z = -n.z;
						normal.add(n);
						count++;
					}

					if (m3 != -2 && m4 != -2) {
						v1.x = 1.0f; v1.y = 0.0f; v1.z = m3;
						v2.x = 0.0f; v2.y = 1.0f; v2.z = m4;
						v1.crossProduct(v2, n);
						n.normalize();
						if (n.z < 0.0)
							n.z = -n.z;
						normal.add(n);
						count++;
					}

					// Average the four normals
					normal.x /= count;
					normal.y /= count;
					normal.z /= count;
				} else {
					if (normalEvaluator != null)
						normalEvaluator.getNormalAt(x, y, width, height, normal);
				}
				if (invertBumps) {
					normal.x = -normal.x;
					normal.y = -normal.y;
				}
				position.x = x;

				if (normal.z >= 0) {
					// Get the material colour at this point
					if (colorSource == COLORS_FROM_IMAGE)
						diffuseColor.setColor(inPixels[index]);
					else if (colorSource == COLORS_FROM_ENVIRONMENT && environmentMap != null) {
						//FIXME-too much normalizing going on here
						tmpv2.set(viewpoint);
						tmpv2.z = 100.0f;//FIXME
						tmpv2.subtract(position);
						tmpv2.normalize();
						tmpv.set(normal);
						tmpv.normalize();
						tmpv.reflect(tmpv2);
						tmpv.normalize();
						diffuseColor.setColor(getEnvironmentMap(tmpv, inPixels, width, height));
					}
					// Shade the pixel
					ARGB c = phongShade(position, viewpoint, normal, diffuseColor, specularColor, material, lightsArray);
					int alpha = inPixels[index] & 0xff000000;
					int rgb = c.argbValue() & 0x00ffffff;
					outPixels[index++] = alpha | rgb;
				} else
					outPixels[index++] = 0;
			}
		}
		consumer.setPixels(0, 0, width, height, defaultRGBModel, outPixels, 0, width);
		consumer.imageComplete(status);
		inPixels = null;
	}

	public ARGB phongShade(Vector3D position, Vector3D viewpoint, Vector3D normal, ARGB diffuseColor, ARGB specularColor, Material material, Light[] lightsArray) {
		shadedColor.setColor(diffuseColor);
		shadedColor.multiply(material.ambientIntensity);

		for (int i = 0; i < lightsArray.length; i++) {
			Light light = lightsArray[i];
			//FIXME-normalize outside loop
			n.set(normal);
			n.normalize();
			l.set(light.position);
			if (light.type != DISTANT)
				l.subtract(position);
			l.normalize();
			float nDotL = n.innerProduct(l);
			if (nDotL >= 0.0) {
				float dDotL = 0;
				
				v.set(viewpoint);
				v.subtract(position);
				v.normalize();

				// Spotlight
				if (light.type == SPOT) {
					dDotL = light.direction.innerProduct(l);
					if (dDotL < light.cosConeAngle)
						continue;
				}

				n.multiply(2.0f * nDotL);
				n.subtract(l);
				float rDotV = n.innerProduct(v);

				float rv;
				if (rDotV < 0.0)
					rv = 0.0f;
				else
					rv = (float)Math.pow(rDotV, material.highlight);

				// Spotlight
				if (light.type == SPOT) {
					dDotL = light.cosConeAngle/dDotL;
					float e = dDotL;
					e *= e;
					e *= e;
					e *= e;
					e = (float)Math.pow(dDotL, light.focus*10)*(1 - e);
//					e = (float)Math.pow(e, light.focus*10)*(1 - dDotL);
					rv *= e;
					nDotL *= e;
				}
				
				diffuse_color.setColor(diffuseColor);
				diffuse_color.multiply(material.diffuseReflectivity);
				diffuse_color.multiply(light.realColor);
				diffuse_color.multiply(nDotL);
				specular_color.setColor(specularColor);
				specular_color.multiply(material.specularReflectivity);
				specular_color.multiply(light.realColor);
				specular_color.multiply(rv);
				diffuse_color.add(specular_color);
				diffuse_color.clamp();
				shadedColor.add(diffuse_color);
			}
		}
		shadedColor.clamp();
		return shadedColor;
	}

	private int[] rgb = new int[4];

	private int getEnvironmentMap(Vector3D normal, int[] inPixels, int width, int height) {
		if (environmentMap != null) {
			float angle = (float)Math.acos(-normal.y);

			float x, y;
			y = angle/ImageMath.PI;

			if (y == 0.0f || y == 1.0f)
				x = 0.0f;
			else {
				float f = normal.x/(float)Math.sin(angle);

				if (f > 1.0f)
					f = 1.0f;
				else if (f < -1.0f) 
					f = -1.0f;

				x = (float)Math.acos(f)/ImageMath.PI;
			}
			// A bit of emprirical scaling....
			x = (x-0.5f)*1.2f+0.5f;
			y = (y-0.5f)*1.2f+0.5f;
			x = ImageMath.clamp(x * envWidth, 0, envWidth-1);
			y = ImageMath.clamp(y * envHeight, 0, envHeight-1);
			int ix = (int)x;
			int iy = (int)y;

			float xWeight = x-ix;
			float yWeight = y-iy;
			int i = envWidth*iy + ix;
			int dx = ix == envWidth-1 ? 0 : 1;
			int dy = iy == envHeight-1 ? 0 : envWidth;
			rgb[0] = envPixels[i];
			rgb[1] = envPixels[i+dx];
			rgb[2] = envPixels[i+dy];
			rgb[3] = envPixels[i+dx+dy];
			return ImageMath.bilinearInterpolate(xWeight, yWeight, rgb);
		}
		return 0;
	}

	public class NormalEvaluator {
		public final static int RECTANGLE = 0;
		public final static int ROUNDRECT = 1;
		public final static int ELLIPSE = 2;
		
		public final static int LINEAR = 0;
		public final static int SIN = 1;
		public final static int CIRCLE_UP = 2;
		public final static int CIRCLE_DOWN = 3;
		public final static int SMOOTH = 4;
		public final static int PULSE = 5;
		public final static int SMOOTH_PULSE = 6;
		public final static int THING = 7;

		private int margin = 10;
		private int shape = RECTANGLE;
		private int bevel = LINEAR;
		private int cornerRadius = 15;

		public void setMargin(int margin) {
			this.margin = margin;
		}

		public int getMargin() {
			return margin;
		}

		public void setCornerRadius(int cornerRadius) {
			this.cornerRadius = cornerRadius;
		}

		public int getCornerRadius() {
			return cornerRadius;
		}

		public void setShape(int shape) {
			this.shape = shape;
		}

		public int getShape() {
			return shape;
		}

		public void setBevel(int bevel) {
			this.bevel = bevel;
		}

		public int getBevel() {
			return bevel;
		}

		public  void getNormalAt(int x, int y, int width, int height, Vector3D normal) {
			float distance = 0;
			normal.x = normal.y = 0;
			normal.z = 0.707f;
			switch (shape) {
			case RECTANGLE:
				if (x < margin) {
					if (x < y && x < height-y)
						normal.x = -1;
				} else if (width-x <= margin) {
					if (width-x-1 < y && width-x <= height-y)
						normal.x = 1;
				}
				if (normal.x == 0) {
					if (y < margin) {
						normal.y = -1;
					} else if (height-y <= margin)
						normal.y = 1;
				}
				distance = Math.min(Math.min(x, y), Math.min(width-x-1, height-y-1));
				break;
			case ELLIPSE:
				float a = width/2;
				float b = height/2;
				float a2 = a*a;
				float b2 = b*b;
				float dx = x-a;
				float dy = y-b;
				float x2 = dx*dx;
				float y2 = dy*dy;
				distance = (b2 - (b2*x2)/a2) - y2;
				float radius = (float)Math.sqrt(x2+y2);
				distance = 0.5f*distance/((a+b)/2);//FIXME
				if (radius != 0) {
					normal.x = dx/radius;
					normal.y = dy/radius;
				}
				break;
			case ROUNDRECT:
				distance = Math.min(Math.min(x, y), Math.min(width-x-1, height-y-1));
				float c = Math.min(cornerRadius, Math.min(width/2, height/2));
				if ((x < c || width-x <= c) && (y < c || height-y <= c)) {
					if (width-x <= c)
						x -= width-c-c-1;
					if (height-y <= c)
						y -= height-c-c-1;
					dx = x-c;
					dy = y-c;
					x2 = dx*dx;
					y2 = dy*dy;
					radius = (float)Math.sqrt(x2+y2);
					distance = c-radius;
					normal.x = dx/radius;
					normal.y = dy/radius;
				} else if (x < margin) {
					normal.x = -1;
				} else if (width-x <= margin) {
					normal.x = 1;
				} else if (y < margin) {
					normal.y = -1;
				} else if (height-y <= margin)
					normal.y = 1;
				break;
			}
			distance /= margin;
if (distance < 0) {
	normal.z = -1;
	normal.normalize();
	return;
}

			float dx = 1.0f/margin;
			float z1 = bevelFunction(distance);
			float z2 = bevelFunction(distance+dx);
			float dz = z2-z1;
			normal.z = dx;
			normal.x *= dz;
			normal.y *= dz;
/*
			if (dz == 0)
				normal.z = 1e10;
			else {
				float f = dz/(1.0/margin);
				normal.x /= f;
				normal.y /= f;
				normal.z *= f;
			}
*/

			normal.normalize();
		}
		
		private float bevelFunction(float x) {
			x = ImageMath.clamp(x, 0.0f, 1.0f);
			switch (bevel) {
			case LINEAR:
				return ImageMath.clamp(x, 0.0f, 1.0f);
			case SIN:
				return (float)Math.sin(x*Math.PI/2);
			case CIRCLE_UP:
				return ImageMath.circleUp(x);
			case CIRCLE_DOWN:
				return ImageMath.circleDown(x);
			case SMOOTH:
				return ImageMath.smoothStep(0.1f, 0.9f, x);
			case PULSE:
				return ImageMath.pulse(0.0f, 1.0f, x);
			case SMOOTH_PULSE:
				return ImageMath.smoothPulse(0.0f, 0.1f, 0.5f, 1.0f, x);
			case THING:
				return (float)(x < 0.2 ? Math.sin(x/0.2*Math.PI/2) : 0.5+0.5*Math.sin(1+x/0.6*Math.PI/2));
			}
			return x;
		}
	}
	
	public String toString() {
		return "Stylize/Light Effects...";
	}

	static class ARGB {
		public float a;
		public float r;
		public float g;
		public float b;

		public ARGB() {
			this(0, 0, 0, 0);
		}
		
		public ARGB(int a, int r, int g, int b) {
			this.a = (float)a / 255.0f;
			this.r = (float)r / 255.0f;
			this.g = (float)g / 255.0f;
			this.b = (float)b / 255.0f;
		}

		public ARGB(float a, float r, float g, float b) {
			this.a = a;
			this.r = r;
			this.g = g;
			this.b = b;
		}

		public ARGB(ARGB v) {
			a = v.a;
			r = v.r;
			g = v.g;
			b = v.b;
		}

		public ARGB(int v) {
			this(v >> 24 & 255, v >> 16 & 255, v >> 8 & 255, v & 255);
		}

		public ARGB(Color c) {
			this(c.getRGB());
		}

		public void setColor(int a, int r, int g, int b) {
			this.a = (float)a / 255.0f;
			this.r = (float)r / 255.0f;
			this.g = (float)g / 255.0f;
			this.b = (float)b / 255.0f;
		}

		public void setColor(float a, float r, float g, float b) {
			this.a = a;
			this.r = r;
			this.g = g;
			this.b = b;
		}

		public void setColor(ARGB v) {
			a = v.a;
			r = v.r;
			g = v.g;
			b = v.b;
		}

		public void setColor(int v) {
			setColor(v >> 24 & 255, v >> 16 & 255, v >> 8 & 255, v & 255);
		}

		public int argbValue() {
			a = 1.0f;
			int ia = (int)(255.0 * a);
			int ir = (int)(255.0 * r);
			int ig = (int)(255.0 * g);
			int ib = (int)(255.0 * b);
			return ia << 24 | ir << 16 | ig << 8 | ib;
		}

		public float length() {
			return (float)Math.sqrt(r * r + g * g + b * b);
		}

		public void normalize() {
			float l = length();
			if (l != 0.0)
				multiply(1.0f / l);
		}

		public void clamp() {
			if (a < 0.0f)
				a = 0.0f;
			else if (a > 1.0f)
				a = 1.0f;
			if (r < 0.0f)
				r = 0.0f;
			else if (r > 1.0f)
				r = 1.0f;
			if (g < 0.0)
				g = 0.0f;
			else if (g > 1.0)
				g = 1.0f;
			if (b < 0.0f)
				b = 0.0f;
			else if (b > 1.0f)
				b = 1.0f;
		}

		public void multiply(float f) {
			r *= f;
			g *= f;
			b *= f;
		}

		public void add(ARGB v) {
			r += v.r;
			g += v.g;
			b += v.b;
		}

		public void subtract(ARGB v) {
			r -= v.r;
			g -= v.g;
			b -= v.b;
		}

		public void multiply(ARGB v) {
			r *= v.r;
			g *= v.g;
			b *= v.b;
		}

		public String toString() {
			return a + " " + r + " " + g + " " + b + ")";
		}
	}

	static class Vector3D {
		public float x;
		public float y;
		public float z;

		public Vector3D() {
	   	}
	   	
		public Vector3D(float x, float y, float z) {
			this.x = x;
			this.y = y;
			this.z = z;
		}

		public Vector3D(Vector3D v) {
			x = v.x;
			y = v.y;
			z = v.z;
		}

		public void set(float x, float y, float z) {
			this.x = x;
			this.y = y;
			this.z = z;
		}

		public void set(Vector3D v) {
			x = v.x;
			y = v.y;
			z = v.z;
		}

		public float length() {
			return (float)Math.sqrt(x * x + y * y + z * z);
		}

		public void normalize() {
			float l = length();
			if (l != 0.0f)
				multiply(1.0f / l);
		}

		public void multiply(float f) {
			x *= f;
			y *= f;
			z *= f;
		}

		public void add(Vector3D v) {
			x += v.x;
			y += v.y;
			z += v.z;
		}

		public void subtract(Vector3D v) {
			x -= v.x;
			y -= v.y;
			z -= v.z;
		}

		public void multiply(Vector3D v) {
			x *= v.x;
			y *= v.y;
			z *= v.z;
		}

		public float innerProduct(Vector3D v) {
			return x * v.x + y * v.y + z * v.z;
		}

		public void crossProduct(Vector3D v, Vector3D result) {
			result.x = y * v.z - z * v.y;
			result.y = z * v.x - x * v.z;
			result.z = x * v.y - y * v.x;
		}

		// Reflects v about "this" (the normal)
		public void reflect(Vector3D v) {
			float n = 2.0f*innerProduct(v);
			multiply(n);
			subtract(v);
		}

		public String toString() {
			return "(" + x + " " + y + " " + z + ")";
		}
	}

	public static class Material {
		float ambientIntensity;
		float diffuseReflectivity;
		float specularReflectivity;
		float highlight;
		float reflectivity;

		public Material() {
			ambientIntensity = 0.5f;
			diffuseReflectivity = 0.8f;
			specularReflectivity = 0.9f;
			highlight = 3.0f;
			reflectivity = 0.0f;
		}
	}

	public final static int AMBIENT = 0;
	public final static int DISTANT = 1;
	public final static int POINT = 2;
	public final static int SPOT = 3;

	public static class Light implements Cloneable {

		int type = AMBIENT;
		Vector3D position;
		Vector3D direction;
		ARGB realColor;
		int color = 0xffffffff;
		float intensity;
		float azimuth;
		float elevation;
		float focus = 0.5f;
		float centreX = 0.5f, centreY = 0.5f;
		float coneAngle = ImageMath.PI/6;
		float cosConeAngle;
		float distance = 100.0f;

		public Light() {
			this(135*ImageMath.PI/180.0f, 0.5235987755982988f, 1.0f);
		}
		
		public Light(float azimuth, float elevation, float intensity) {
			this.azimuth = azimuth;
			this.elevation = elevation;
			this.intensity = intensity;
		}
		
		public void setAzimuth(float azimuth) {
			this.azimuth = azimuth;
		}

		public float getAzimuth() {
			return azimuth;
		}

		public void setElevation(float elevation) {
			this.elevation = elevation;
		}

		public float getElevation() {
			return elevation;
		}

		public void setDistance(float distance) {
			this.distance = distance;
		}

		public float getDistance() {
			return distance;
		}

		public void setIntensity(float intensity) {
			this.intensity = intensity;
		}

		public float getIntensity() {
			return intensity;
		}

		public void setConeAngle(float coneAngle) {
			this.coneAngle = coneAngle;
		}

		public float getConeAngle() {
			return coneAngle;
		}

		public void setFocus(float focus) {
			this.focus = focus;
		}

		public float getFocus() {
			return focus;
		}

		public void setColor(int color) {
			this.color = color;
		}

		public int getColor() {
			return color;
		}

		public void setCentreX(float x) {
			centreX = x;
		}
		
		public float getCentreX() {
			return centreX;
		}

		public void setCentreY(float y) {
			centreY = y;
		}
		
		public float getCentreY() {
			return centreY;
		}

		public void prepare(int width, int height) {
			float lx = (float)(Math.cos(azimuth) * Math.cos(elevation));
			float ly = (float)(Math.sin(azimuth) * Math.cos(elevation));
			float lz = (float)Math.sin(elevation);
			direction = new Vector3D(lx, ly, lz);
			direction.normalize();
			if (type != DISTANT) {
				lx *= distance;
				ly *= distance;
				lz *= distance;
				lx += width * centreX;
				ly += height * (1-centreY);
			}
			position = new Vector3D(lx, ly, lz);
			realColor = new ARGB(color);
			realColor.multiply(intensity);
			cosConeAngle = (float)Math.cos(coneAngle);
		}
		
		public Object clone() {
			try {
				Light copy = (Light)super.clone();
				return copy;
			}
			catch (CloneNotSupportedException e) {
				return null;
			}
		}

		public String toString() {
			return "Light";
		}

	}

	public class AmbientLight extends Light {
		public String toString() {
			return "Ambient Light";
		}
	}

	public class PointLight extends Light {
		public PointLight() {
			type = POINT;
		}

		public String toString() {
			return "Point Light";
		}
	}

	public class DistantLight extends Light {
		public DistantLight() {
			type = DISTANT;
		}

		public String toString() {
			return "Distant Light";
		}
	}

	public class SpotLight extends Light {
		public SpotLight() {
			type = SPOT;
		}

		public String toString() {
			return "Spotlight";
		}
	}
}
