package com.atlassian.application.host;

import com.atlassian.application.api.Application;
import com.atlassian.application.api.ApplicationAccess;
import com.atlassian.application.api.ApplicationKey;
import com.atlassian.application.api.ApplicationManager;
import com.atlassian.application.api.PluginApplication;
import com.atlassian.application.host.i18n.I18nResolver;
import com.atlassian.application.host.license.LicenseLocator;
import com.atlassian.application.host.plugin.PluginApplicationImpl;
import com.atlassian.application.host.plugin.PluginApplicationMetaData;
import com.atlassian.application.host.plugin.PluginApplicationMetaDataManager;
import com.atlassian.sal.api.license.SingleProductLicenseDetailsView;
import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.base.Supplier;
import com.google.common.collect.Iterables;
import io.atlassian.fugue.Option;
import org.joda.time.DateTime;

import java.util.Collections;
import javax.annotation.Nullable;

import static com.atlassian.application.host.util.Arguments.checkArgumentNotNull;
import static com.google.common.base.Suppliers.memoize;
import static com.google.common.collect.Iterables.transform;
import static io.atlassian.fugue.Iterables.findFirst;

/**
 * Provides an a template for hosts (i.e. JIRA) to implement {@link ApplicationManager}.
 * <p>
 * The host must implement {@link #getPlatform()} to return the host specific {@link
 * com.atlassian.application.api.PlatformApplication}.
 *
 * @since v1.0
 */
public abstract class AbstractApplicationManager implements ApplicationManager
{
    private final I18nResolver resolver;
    private final PluginApplicationMetaDataManager pluginApplications;
    private final LicenseLocator licenseLocator;
    private final ApplicationAccessFactory applicationAccessFactory;
    private final ApplicationConfigurationManager appConfigManager;

    protected AbstractApplicationManager(final PluginApplicationMetaDataManager pluginApplications,
        final I18nResolver i18nResolver, final LicenseLocator licenseLocator,
        final ApplicationAccessFactory applicationAccessFactory,
        final ApplicationConfigurationManager appConfigManager)
    {
        this.resolver = checkArgumentNotNull(i18nResolver, "i18n");
        this.pluginApplications = checkArgumentNotNull(pluginApplications, "pluginApplications");
        this.licenseLocator = checkArgumentNotNull(licenseLocator, "licenseLocator");
        this.applicationAccessFactory = checkArgumentNotNull(applicationAccessFactory, "accessFactory");
        this.appConfigManager = checkArgumentNotNull(appConfigManager, "appConfigManager");
    }

    @Override
    public final Iterable<Application> getApplications()
    {
        return Iterables.concat(Collections.singleton(getPlatform()), pluginApplications());
    }

    @Override
    public final Option<Application> getApplication(final ApplicationKey key)
    {
        checkArgumentNotNull(key, "key");

        return findFirst(getApplications(), new Predicate<Application>()
        {
            @Override
            public boolean apply(@Nullable final Application input)
            {
                assert input != null;

                return input.getKey().equals(key);
            }
        });
    }

    @Override
    public final <A extends Application> Option<A> getApplication(final ApplicationKey key, final Class<A> type)
    {
        checkArgumentNotNull(key, "key");
        checkArgumentNotNull(type, "type");

        return getApplication(key).flatMap(new Function<Application, Option<A>>()
        {
            @Override
            public Option<A> apply(@Nullable final Application input)
            {
                if (type.isInstance(input))
                {
                    return Option.some(type.cast(input));
                }
                else
                {
                    return Option.none(type);
                }
            }
        });
    }

    @Override
    public final Option<ApplicationAccess> getAccess(final ApplicationKey key)
    {
        return getApplication(key).map(new Function<Application, ApplicationAccess>()
        {
            @Override
            public ApplicationAccess apply(@Nullable final Application input)
            {
                assert input != null;
                return input.getAccess();
            }
        });
    }

    private Supplier<Option<SingleProductLicenseDetailsView>> viewSupplier(final ApplicationKey key)
    {
        return new Supplier<Option<SingleProductLicenseDetailsView>>()
        {
            @Override
            public Option<SingleProductLicenseDetailsView> get()
            {
                return licenseLocator.apply(key);
            }
        };
    }

    private Supplier<ApplicationAccess> accessSuppler(final ApplicationKey key, final DateTime buildDate)
    {
        return new Supplier<ApplicationAccess>()
        {
            @Override
            public ApplicationAccess get()
            {
                return applicationAccessFactory.access(key, buildDate);
            }
        };
    }

    private Iterable<PluginApplication> pluginApplications()
    {
        return transform(pluginApplications.getApplications(),
            new Function<PluginApplicationMetaData, PluginApplication>()
            {
                @Override
                public PluginApplication apply(@Nullable final PluginApplicationMetaData input)
                {
                    assert input != null;
                    return new PluginApplicationImpl(input,
                            resolver,
                            memoize(viewSupplier(input.getKey())),
                            memoize(accessSuppler(input.getKey(), input.buildDate())),
                            appConfigManager);
                }
            });
    }
}
