package com.atlassian.application.host.plugin;

import com.atlassian.application.api.ApplicationKey;
import com.atlassian.application.api.ApplicationPlugin;
import io.atlassian.fugue.Option;
import org.joda.time.DateTime;

import java.net.URI;

/**
 * The meta-data about an Application provided by a set of plugins. There are three types of
 * plugins:
 *
 * <ul>
 *     <li>
 *          Primary Plugin: The plugin that defines the application. Disabling this plugin will will disable the
 *          application. There is only one primary plugin per application. The primary plugin may also provide
 *          application functionality.
 *      </li>
 *      <li>
 *          Application Plugin: A plugin providing fundamental features for the application. All application plugins
 *          must be enabled and working for the application to be considered working. The application plugin is
 *          <strong>owned</strong> by the application (aka the plugin will not work without the primary of the
 *          application).
 *      </li>
 *      <li>
 *          Utility Plugin: A plugin that provides utilities to the primary and/or application plugins. A utility plugin
 *          is not owned by the application (i.e. the plugin will work without the primary of the application) and may
 *          be used by multiple applications or other plugins
 *      </li>
 * </ul>
 *
 * Unless otherwise noted none of the parameters or return values can be {@code null}.
 *
 * @since v1.0
 */
public interface PluginApplicationMetaData
{
    /**
     * Return the {@link com.atlassian.application.api.ApplicationKey} associated with the Application.
     *
     * @return the {@link com.atlassian.application.api.ApplicationKey} associated with the Application.
     */
    ApplicationKey getKey();

    /**
     * Get the name of the application.
     *
     * @return the name of the application. Must not be internationalised.
     */
    String getName();

    /**
     * Get the version of the application.
     *
     * <p>In the case of plugin applications, this will be the version of the primary plugin.
     *
     * @return the version of the application.
     */
    String getVersion();

    /**
     * Get the description of the application as an i18n key.
     *
     * @return the description of the application as an i18n key.
     */
    String getDescriptionKey();

    /**
     * Returns a key can be used to generate internationalised description for a passed number of users.
     * For example, JIRA Service Desk might return &quot;agent&quot; for a count of 1 or &quot;agents&quot;
     * for a count of 2 or more. A value of -1 is passed to try and find the "unlimited" users.
     *
     * @return the internationalised name of the passed number of users.
     */
    String getUserCountKey();

    /**
     * Get the relative URI for the configuration of the application. The context path must not be included.
     * <p>
     * {@link io.atlassian.fugue.Option#none()} can be returned to indicate that there is no configuration path.
     *
     * @return the relative URI to the configuration page minus the context path or {@link
     * io.atlassian.fugue.Option#none()} if no such path exists.
     */
    Option<URI> getConfigurationURI();

    /**
     * Get the relative URI to show after the application is installed. The context path must not be included.
     * <p>
     * {@link io.atlassian.fugue.Option#none()} can be returned to indicate that there is no path.
     *
     * @return the relative URI to the post install page minus the context path or {@link
     * io.atlassian.fugue.Option#none()} if no such path exists.
     */
    Option<URI> getPostInstallURI();

    /**
     * Get the relative URI to show after the application is updated. The context path must not be included.
     * <p>
     * {@link io.atlassian.fugue.Option#none()} can be returned to indicate that there is no path.
     *
     * @return the relative URI to the post update page minus the context path or {@link
     * io.atlassian.fugue.Option#none()} if no such path exists.
     */
    Option<URI> getPostUpdateURI();

    /**
     * Return the build date of the application.
     *
     * @return the build date of the application. Epoch (i.e. {@code new DateTime(0)}) will be returned if the
     * build date is unknown.
     */
    DateTime buildDate();

    /**
     * Return the module key for the module in the Primary plugin that defines the application. The plugin key must
     * be included.
     *
     * @return the module key to the module in the Primary plugin that defines the application.
     */
    String getDefinitionModuleKey();

    /**
     * Get the primary, application and utility plugins associated with the application.
     *
     * @return the primary, application and utility plugins associated with the application.
     */
    Iterable<ApplicationPlugin> getPlugins();

    /**
     * Get the primary plugin associated in the application.
     *
     * @return the primary plugin for the application.
     */
    ApplicationPlugin getPrimaryPlugin();

    /**
     * Return all the application plugins in the application.
     *
     * @return the application plugins in the application. Can be empty if the primary provides all the functionality
     * of the application.
     */
    Iterable<ApplicationPlugin> getApplicationPlugins();

    /**
     * Return all the utility plugins in the application.
     *
     * @return the utility plugins in the application. Can be empty.
     */
    Iterable<ApplicationPlugin> getUtilityPlugins();

    /**
     * Get the default user group defined for the application.
     *
     * @return the default user group of the application.
     */
    String getDefaultGroup();

    /**
     * Returns relative URI to use as target space for product user helplinks.
     *
     * <p>{@link io.atlassian.fugue.Option#none()} can be returned to indicate that there is no server help space.
     *
     * @return the server help space to point product user helplinks to
     * {@link io.atlassian.fugue.Option#none()} if no such space exists.
     */
    Option<URI> getProductHelpServerSpaceURI();

    /**
     * Returns relative URI to use as target space for product user helplinks in cloud.
     *
     * <p>{@link io.atlassian.fugue.Option#none()} can be returned to indicate that there is no cloud help space.
     *
     * @return the cloud help space to point product user helplinks to
     * {@link io.atlassian.fugue.Option#none()} if no such space exists.
     */
    Option<URI> getProductHelpCloudSpaceURI();
}
