package com.atlassian.application.host.util;

import io.atlassian.fugue.Option;
import org.apache.commons.lang3.StringUtils;

import javax.annotation.Nullable;

/**
 * Some pre-conditions for validating arguments and state.
 *
 * @since v1.0
 */
public final class Arguments
{
    /**
     * Cannot be created.
     */
    private Arguments()
    {
    }

    /**
     * Check that the passed argument is not blank.
     *
     * @param argument the argument to check.
     * @param argumentName the name of the argument to check.
     * @return argument un-changed
     * @throws java.lang.IllegalArgumentException if argument is blank.
     */
    public static String checkArgumentNotBlank(final String argument, final String argumentName)
    {
        checkArgumentNotNull(argument, argumentName);
        if (StringUtils.isBlank(argument))
        {
            throw new IllegalArgumentException(String.format("Argument '%s' must not be blank.", argumentName));
        }
        return argument;
    }

    /**
     * Check that the passed argument is {@code null}.
     *
     * @param argument the argument to check.
     * @param argumentName the name of the argument to check.
     * @param <T> the type of argument
     * @return argument un-changed
     * @throws java.lang.NullPointerException if argument is {@code null}.
     */
    public static <T> T checkArgumentNotNull(@Nullable final T argument, final String argumentName)
    {
        if (argument == null)
        {
            throw new NullPointerException(String.format("Argument '%s' must not be null.", argumentName));
        }
        else
        {
            return argument;
        }
    }

    /**
     * Check that the passed argument is not blank provided it is defined.
     *
     * @param argument the argument to check.
     * @param argumentName the name of the argument to check.
     * @return argument un-changed
     * @throws java.lang.IllegalArgumentException if argument is blank.
     */
    public static Option<String> checkArgumentNotBlank(final Option<String> argument, final String argumentName)
    {
        checkArgumentNotNull(argument, argumentName);
        if (argument.isDefined() && StringUtils.isBlank(argument.get()))
        {
            throw new IllegalArgumentException(String.format("Argument '%s' must not be blank.", argumentName));
        }
        return argument;
    }
}
