package com.atlassian.logging.log4j;

import org.apache.commons.lang3.StringUtils;

public class LogMessageUtil {
    public static final char LINE_FEED = '\n';
    public static final char CARRIAGE_RETURN = '\r';
    public static final char VERTICAL_TAB = '\u000B';
    public static final char FORM_FEED = '\u000C';
    public static final char NEXT_LINE = '\u0085';
    public static final char LINE_SEPARATOR = '\u2028';
    public static final char PARAGRAPH_SEPARATOR = '\u2029';

    private LogMessageUtil() {
    }

    /**
     * Appends line indent after each newline character.
     *
     * @param messageText input text
     * @param lineIndent  line indent
     * @return if input text contains no newline characters it's returned unchanged, otherwise a new String object is
     * returned with each newline followed by a lineIndent
     */
    public static String appendLineIndent(final String messageText, final String lineIndent) {
        if (StringUtils.isEmpty(messageText)) {
            return messageText;
        }

        final char[] chars = messageText.toCharArray();
        int i = 0;
        int lastPos = 0;
        StringBuilder builder = null;

        while (i < chars.length) {
            if (chars[i] == CARRIAGE_RETURN || chars[i] == LINE_FEED || chars[i] == VERTICAL_TAB
                    || chars[i] == FORM_FEED || chars[i] == NEXT_LINE || chars[i] == LINE_SEPARATOR || chars[i] == PARAGRAPH_SEPARATOR) {
                if (builder == null) {
                    builder = new StringBuilder(messageText.length());
                }
                // peekahead to see if the next char is part of the new line sequence this only matters for Windows
                // where the current char is a CARRIAGE_RETURN and the next is a LINE_FEED
                if (i + 1 < chars.length && chars[i] == CARRIAGE_RETURN && chars[i + 1] == LINE_FEED) {
                    i++;
                }
                //copy up to the next character after the escape sequence
                builder.append(chars, lastPos, i + 1 - lastPos)
                        .append(lineIndent);
                lastPos = i + 1;
            }
            i++;
        }
        // messageText contains no newlines, return it unchanged
        if (builder == null) {
            return messageText;
        }
        // Copy everything after the last recorded position if there was a new line sequence in the middle of the
        // message string and the last time we copied was NOT at the end of the string
        if (lastPos < chars.length) {
            builder.append(chars, lastPos, i - lastPos);
        }
        return builder.toString();
    }
}
