package com.atlassian.logging.log4j.appender;

import com.atlassian.logging.log4j.appender.fluentd.FluentdAppenderHelper;
import org.apache.logging.log4j.core.Appender;
import org.apache.logging.log4j.core.Core;
import org.apache.logging.log4j.core.Filter;
import org.apache.logging.log4j.core.Layout;
import org.apache.logging.log4j.core.LogEvent;
import org.apache.logging.log4j.core.appender.AbstractAppender;
import org.apache.logging.log4j.core.config.Property;
import org.apache.logging.log4j.core.config.plugins.Plugin;
import org.apache.logging.log4j.core.config.plugins.PluginBuilderAttribute;
import org.apache.logging.log4j.core.config.plugins.PluginBuilderFactory;

import java.io.Serializable;

/**
 * A Log4J 2 {@link org.apache.logging.log4j.core.Appender} which logs to FluentD.
 * <p>
 * Delegates to {@link FluentdAppenderHelper} to do the real work.
 * <p>
 * This appender is async appender by nature (benchmarks show that wrapping this in AsyncAppender only slows it down).
 */
@Plugin(name = "FluentdAppender", category = Core.CATEGORY_NAME, elementType = Appender.ELEMENT_TYPE, printObject = true)
public class FluentdAppender extends AbstractAppender {

    private final FluentdAppenderHelper<LogEvent> helper;

    protected FluentdAppender(String name,
                              Layout<? extends Serializable> layout,
                              Filter filter,
                              boolean ignoreExceptions,
                              final Property[] properties,
                              FluentdAppenderHelper<LogEvent> helper) {
        super(name, filter, layout, ignoreExceptions, properties);
        this.helper = helper;
    }

    /**
     * Stops and restarts the appender, picking up any configuration options that have been changed since the appender
     * was last started. If the appender has not first been enabled, then this will do nothing.
     */
    public void restart() {
        helper.restart();
    }

    /**
     * Enables the appender so that it can be started.
     */
    public void enable() {
        helper.enable();
    }

    /**
     * Disables the appender.
     */
    public void disable() {
        helper.disable();
    }


    @Override
    public void append(final LogEvent loggingEvent) {
        if (helper.isEnabled()) {
            // Read the following attributes, because they are only populated on first read, and so we need to read them
            // before handing the event to the background thread. Typically this doesn't matter as the event is logged
            // elsewhere first, but it matters if the events are sent just to this appender.
            loggingEvent.getThreadName();
            loggingEvent.getContextData();
            loggingEvent.getSource();

            helper.append(loggingEvent.toImmutable());
        }
    }

    @Override
    public void start() {
        setStarting();
        super.start();
        helper.initialise();
        setStarted();
    }

    @Override
    public void stop() {
        setStopping();
        helper.close();
        super.stop();
        setStopped();
    }

    @PluginBuilderFactory
    public static <B extends Builder<B>> B newBuilder() {
        return new Builder<B>().asBuilder();
    }

    public static class Builder<B extends Builder<B>> extends AbstractAppender.Builder<B>
            implements org.apache.logging.log4j.core.util.Builder<FluentdAppender> {
        @PluginBuilderAttribute
        private String fluentdEndpoint;
        @PluginBuilderAttribute
        private Long batchPeriodMs;
        @PluginBuilderAttribute
        private Long maxNumLogEvents;
        @PluginBuilderAttribute
        private Integer maxRetryPeriodMs;
        @PluginBuilderAttribute
        private Integer backoffMultiplier;
        @PluginBuilderAttribute
        private Integer maxBackoffMinutes;

        private final FluentdAppenderHelper<LogEvent> helper = new FluentdAppenderHelper<>();

        @Override
        public B setLayout(Layout<? extends Serializable> layout) {
            return super.setLayout(layout);
        }


        /**
         * The fluentd endpoint to send the logs to - including the tag.  Ex http://hostname:8888/test.tag.here
         */
        public B setFluentdEndpoint(String fluentdEndpoint) {
            this.fluentdEndpoint = fluentdEndpoint;
            return asBuilder();
        }

        /**
         * The appender will send the logs to fluentd in a batch at every batchPeriodMs milliseconds
         */
        public B setBatchPeriodMs(long batchPeriodMs) {
            this.batchPeriodMs = batchPeriodMs;
            return asBuilder();
        }

        /**
         * The max length of the sum of all logs in the queue before we start dropping them. This will prevent OOMEs.
         * The amount of space in bytes is roughly to maxChars * 2
         */
        public B setMaxNumLogEvents(long maxNumLogEvents) {
            this.maxNumLogEvents = maxNumLogEvents;
            return asBuilder();
        }

        /**
         * The maximum amount of time that the (async) log sender will retry for before giving up.
         */
        public B setMaxRetryPeriodMs(int maxRetryPeriodMs) {
            this.maxRetryPeriodMs = maxRetryPeriodMs;
            return asBuilder();
        }
        /**
         * The backoff multiplier Backoff formula is: `backoffMultiplier * 2^x + up to 20% jitter` where x is the attempt
         * number.
         */
        public B setBackoffMultiplier(int backoffMultiplier) {
            this.backoffMultiplier = backoffMultiplier;
            return asBuilder();
        }

        /**
         * The maximum amount of backoff for a given retry (not including 20% jitter)
         */
        public B setMaxBackoffMinutes(int maxBackoffMinutes) {
            this.maxBackoffMinutes = maxBackoffMinutes;
            return asBuilder();
        }

        @Override
        public FluentdAppender build() {
            if (fluentdEndpoint != null) {
                this.helper.setFluentdEndpoint(fluentdEndpoint);
            }
            if (batchPeriodMs != null) {
                this.helper.setBatchPeriodMs(batchPeriodMs);
            }
            if (maxNumLogEvents != null) {
                this.helper.setMaxNumEvents(maxNumLogEvents);
            }
            if (maxRetryPeriodMs != null) {
                this.helper.setMaxRetryPeriodMs(maxRetryPeriodMs);
            }
            if (backoffMultiplier != null) {
                this.helper.setBackoffMultiplier(backoffMultiplier);
            }
            if (maxBackoffMinutes != null) {
                this.helper.setMaxBackoffMinutes(maxBackoffMinutes);
            }
            if (getLayout() != null) {
                helper.setLayout(getLayout()::toSerializable);
            }
            return new FluentdAppender(getName(), getLayout(), getFilter(), isIgnoreExceptions(), getPropertyArray(), helper);
        }
    }
}
