package com.atlassian.logging.log4j.util;

import com.atlassian.annotations.VisibleForTesting;

import javax.annotation.Nullable;
import java.io.PrintStream;
import java.util.concurrent.TimeUnit;

public class TimeSleeper {
    private final Object lock = new Object();
    private volatile boolean sleeping = false;

    // this counter is used only for test purposes
    private volatile int waitInvocationsCounter = 0;
    private PrintStream logOutputStream;

    public TimeSleeper() {
        this(System.err);
    }

    @VisibleForTesting
    TimeSleeper(@Nullable final PrintStream logOutputStream) {
        this.logOutputStream = logOutputStream;
    }

    public void wakeup() {
        if (!sleeping) {
            return;
        }
        synchronized (lock) {
            sleeping = false;
            lock.notifyAll();
        }
    }

    /**
     * Sleeps up to given timeout. The sleeping thread can be waken up by {@link TimeSleeper#wakeup()} method.
     *
     * @param time     Time period to sleep for
     * @param timeUnit Time unit for the given period
     * @return Information if we reached the desired sleep time (true) or we were waken up earlier (false).
     */
    public boolean sleep(final long time, final TimeUnit timeUnit) {
        synchronized (lock) {
            if (sleeping) {
                throw new IllegalStateException("Got sleep request while already sleeping - this is not supported!");
            }

            try {
                sleeping = true;
                waitInvocationsCounter = 0;

                // a bit of time logic to guard us from spurious wakeup
                long remainingSleepTimeMs = timeUnit.toMillis(time);
                while (sleeping && remainingSleepTimeMs > 0) {
                    final long sleepStartTime = System.nanoTime();

                    // counter for test only - be careful when changing code around!
                    waitInvocationsCounter++;

                    lock.wait(remainingSleepTimeMs);
                    remainingSleepTimeMs -= TimeUnit.NANOSECONDS.toMillis(
                            System.nanoTime() - sleepStartTime);
                }
                return sleeping;
            } catch (final Exception e) {
                logException("Failed to keep thread asleep", e);

                // well, this was some unexpected wakeup
                return false;
            } finally {
                sleeping = false;
            }
        }
    }

    @VisibleForTesting
    void logException(final String message, final Exception exception) {
        if (logOutputStream != null) {
            logOutputStream.println(getClass().getName() + " - ERROR - " + message);
            exception.printStackTrace(logOutputStream);
        }
    }

    @VisibleForTesting
    void simulateSpuriousWakeUp() {
        synchronized (lock) {
            lock.notifyAll();
        }
    }

    @VisibleForTesting
    boolean isSleeping() {
        synchronized (lock) {
            return sleeping;
        }
    }

    @VisibleForTesting
    int getWaitInvocationsCounter() {
        synchronized (lock) {
            return waitInvocationsCounter;
        }
    }


}