package com.atlassian.bonnie;

import java.io.IOException;

import org.apache.lucene.document.Document;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.store.Directory;

/**
 * A connection to a Lucene index. Supports reads, writes, searches, batch updates, and truncates.
 */
public interface ILuceneConnection
{
    public interface SearcherAction
    {
        /**
         * Perform search. The searcher is managed which means you must not
         * close the searcher.
         */
        public void perform(IndexSearcher searcher) throws IOException;
    }

    public interface SearcherWithTokenAction<T>
    {
        /**
         * Perform search. The searcher is managed which means you must not
         * close the searcher.
         */
        public T perform(IndexSearcher searcher, long searchToken) throws IOException;
    }

    public interface ReaderAction
    {
        /**
         * Perform index reading.
         */
        public Object perform(IndexReader reader) throws IOException;
    }

    public interface WriterAction
    {
        /**
         * Perform index writing.
         */
        public void perform(IndexWriter writer) throws IOException;
    }

    public interface BatchUpdateAction
    {
        /**
         * Perform batch update operation. Use this action for multi-step
         * operations where you want to acquire a write lock for the duration of
         * the operation.
         */
        public void perform() throws Exception;
    }

    void withSearch(SearcherAction action) throws LuceneException;

    /**
     * <p>Perform a search that can be later continued via the search token passed to {@link ILuceneConnection.SearcherWithTokenAction#perform(org.apache.lucene.search.IndexSearcher, long)}.</p>
     *
     * <p>You may want to continue the search by requesting the next page of results, or refining the search with
     * additional parameters. The token guarantees that the search will be conducted on a particular version of the index.</p>
     *
     * @param action the search action to perform
     * @param <T> the return type of the searcher action callback
     *
     * @return result of the search
     */
    <T> T withSearcher(SearcherWithTokenAction<T> action);

    /**
     * Perform a search with the specified search token.
     *
     * @param searchToken a search token that identifies a version of the index to search.
     * @param action the search action to perform
     * @param <T> the return type of the searcher action callback
     *
     * @return result of the search
     * @throws SearchTokenExpiredException if the specified search token has expired. Clients should report the error to the user and / or retry the search with a new token.
     */
    <T> T withSearcher(long searchToken, SearcherWithTokenAction<T> action) throws SearchTokenExpiredException;

    /**
     * Idempotent operation. Just for querying, do not delete documents with
     * this action. Use {@link #withWriter(WriterAction)} to perform
     * index deletes.
     */
    Object withReader(ReaderAction action) throws LuceneException;

    /**
     * Add (write) documents to the index
     */
    void withWriter(WriterAction action) throws LuceneException;

    /**
     * Perform multiple writes to the index. Holds a writeLock on the index for
     * the whole time, and will use the {@link Configuration batch
     * configuration} settings.
     * <p></p>
     * Update actions performed within a batch update won't be visible to other
     * readers or searchers until the batch is complete. Be aware this also
     * applies to actions within a batch! That is, a read operation inside a
     * batch will not see the changes made by earlier updates in that batch.
     */
    void withBatchUpdate(BatchUpdateAction action);

    /**
     * Perform an optimize on the index. Holds a writeLock and can take a long
     * time (depending on the size of the index, how much optimization needs to
     * be done... and whether a virus-checker is installed :) ).
     */
    void optimize() throws LuceneException;

    /**
     * Closes the reader and the writer. Calling any method on the API after
     * closing the connection will throw {@link LuceneConnectionClosedException}.
     */
    void close() throws LuceneException;

    /**
     * Closes the writer. Calling {@link #withWriter(WriterAction)} or {@link #withBatchUpdate(BatchUpdateAction)}
     * after closing the writer will throw {@link IndexWriterClosedException}.
     */
    void closeWriter() throws LuceneException;

    /**
     * Returns the number of {@link Document documents} in the index.
     */
    int getNumDocs();

    /**
     * Removes all documents from the index.
     *
     * @throws LuceneException if there was a problem removing the index
     */
    void truncateIndex() throws LuceneException;

    /**
     * Provide configuration for the index writers used by implementations of this interface.
     * 
     * @see DefaultConfiguration
     */
    public interface Configuration
    {
        int getInteractiveMergeFactor();
        int getInteractiveMaxMergeDocs();
        int getInteractiveMaxBufferedDocs();
        int getBatchMergeFactor();
        int getBatchMaxMergeDocs();
        int getBatchMaxBufferedDocs();
        int getMaxFieldLength();
        boolean isCompoundIndexFileFormat();

        /**
         * The max age of an IndexSearcher in seconds.
         *
         * <tt>IndexSearcher</tt>s that are older than this are eligible for pruning by a scheduled job so that it's resources are reclaimed.
         *
         * @return max age of an IndexSearcher in seconds.
         */
        long getIndexSearcherMaxAge();

        /**
         * The delay in seconds that the a periodic should wait before attempting to prune <tt>IndexSearcher</tt>s that have exceeded their max age.
         *
         * @return delay in seconds
         */
        long getIndexSearcherPruneDelay();
    }

    /**
     * The Bonnie default configuration. Preserves backward compatibility for old Confluence settings.
     */
    static final Configuration DEFAULT_CONFIGURATION = new DefaultConfiguration();

    /**
     * Allows taking a snapshot of the index.
     *
     * @param destDir directory in which the snapshot should be saved
     * @since 6.2
     */
    void snapshot(Directory destDir) throws IOException;

    /**
     * Closes the reader and the writer and create new ones.
     *
     *
     * @param resetAction action that should be executed before the reset
     * @since 6.5
     */
    void reset(Runnable resetAction) throws LuceneException;
}