package com.atlassian.bonnie;

import com.atlassian.bonnie.search.SearcherInitialisation;
import com.atlassian.bonnie.upgrader.LuceneIndexUpgrader;
import io.atlassian.util.concurrent.ThreadFactories;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.index.DirectoryReader;
import org.apache.lucene.index.IndexCommit;
import org.apache.lucene.index.IndexFormatTooOldException;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.IndexUpgrader;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.index.IndexWriterConfig;
import org.apache.lucene.index.KeepOnlyLastCommitDeletionPolicy;
import org.apache.lucene.index.LiveIndexWriterConfig;
import org.apache.lucene.index.SnapshotDeletionPolicy;
import org.apache.lucene.search.DelayCloseIndexSearcher;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.ReferenceManager;
import org.apache.lucene.search.SearcherFactory;
import org.apache.lucene.search.SearcherLifetimeManager;
import org.apache.lucene.search.SearcherManager;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.FSDirectory;
import org.apache.lucene.store.IOContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.IOException;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
/**
 * ILuceneConnection implementation that allows concurrent searching/reading and
 * writing/deleting. Concurrent writes and deletes block each other.
 * <p></p>
 * This class optimizes use of Lucene reader instances by holding a common
 * IndexReader that is shared by idempotent operations on an unmodified index.
 * Any mutative operations cause the current IndexReader to be cleared, and
 * subsequent reads will see the results of the previous index change.
 * <p></p>
 */
public class LuceneConnection implements ILuceneConnection
{
    private static final Logger log = LoggerFactory.getLogger(LuceneConnection.class);

    private static final SearcherInitialisation NOOP_SEARCHER_INITIALISATION = new SearcherInitialisation() {
        public void initialise(IndexSearcher searcher) {}
    };

    private final Analyzer analyzerForIndexing;

    private final Configuration configuration;

    private final SearcherInitialisation searcherInitialisation;

    private final Lock indexWriteLock = new LoggingReentrantLock("indexWriteLock");

    private final Lock searcherRefreshLock = new LoggingReentrantLock("searcherRefreshLock");

    /**
     * As per the documentation in {@link SearcherLifetimeManager#close()},
     * we must ensure we don't permit record() to be called when close() is called (and vice-versa).
     *
     * Use this lock to enforce this.
     *
     * @see SearcherLifetimeManager#close()
     */
    private final ReadWriteLock searcherLifetimeManagerLock = new ReentrantReadWriteLock();

    private final AtomicBoolean isClosed = new AtomicBoolean(false);
    private final AtomicBoolean writerIsClosed = new AtomicBoolean(false);

    /**
     * Used to indicate that we are currently in batchMode and need to use the batch configuration when using an
     * IndexWriter
     */
    private final AtomicBoolean batchMode = new AtomicBoolean(false);

    private IndexWriter writer;
    private SnapshotDeletionPolicy snapshotter;

	private SearcherManager searcherManager;
    private SearcherLifetimeManager searcherLifetimeManager;

    private static final String TRACKED_SEARCHERS_PRUNE_DELAY = "atlassian.indexing.tracked.searchers.prune.delay";
    private static final String TRACKED_SEARCHERS_MAX_AGE = "atlassian.indexing.tracked.searchers.max.age";

    private volatile DelayCloseIndexSearcher searcher;
    private ScheduledExecutorService scheduledExecutorService;

    public LuceneConnection(Directory directory, Analyzer analyzer, final Configuration configuration, final SearcherInitialisation searcherInitialisation)
    {
        this.analyzerForIndexing = analyzer;
        this.configuration = configuration;
        this.searcherInitialisation = searcherInitialisation;

        setUpWriterAndSearcher(directory);

        scheduleIndexSearcherPruneJob(configuration);
    }

    public LuceneConnection(Directory directory, Analyzer analyzer, Configuration configuration)
    {
        this(directory, analyzer, configuration, NOOP_SEARCHER_INITIALISATION);
    }

    public LuceneConnection(Directory directory, Analyzer analyzer)
    {
        this(directory, analyzer, DEFAULT_CONFIGURATION);
    }

    public LuceneConnection(File path, Analyzer analyzer, Configuration configuration, SearcherInitialisation searcherInitialisation)
    {
        this(DirectoryUtil.getDirectory(path), analyzer, configuration, searcherInitialisation);
    }

    public LuceneConnection(File path, Analyzer analyzer, Configuration configuration)
    {
        this(DirectoryUtil.getDirectory(path), analyzer, configuration);
    }

    public LuceneConnection(File path, Analyzer analyzer)
    {
        this(path, analyzer, DEFAULT_CONFIGURATION);
    }

    public int getNumDocs()
    {
        return (Integer) withReader(new ReaderAction() {
            public Object perform(IndexReader reader)
            {
                return reader.numDocs();
            }
        });
    }

    private SearcherManager newSearcherManager() throws IOException
    {
        final SearcherManager newSearchManager = new SearcherManager(writer, true, new SearcherFactory() {
            @Override
            public IndexSearcher newSearcher(IndexReader reader) throws IOException
            {
                IndexSearcher indexSearcher = super.newSearcher(reader);
                searcherInitialisation.initialise(indexSearcher);
                return indexSearcher;
            }
        });

        /**
         * Listen for a refresh of the current <tt>IndexSearcher</tt> (happens after an index change via {@link withWriter()});
         *
         * When this event happens, we must make {@link SearcherLifetimeManager} aware of the new searcher by calling {@link #record(org.apache.lucene.search.IndexSearcher)}.
         * The contract of <tt>SearchLifetimeManager</tt> is that it tracks the current and old <tt>IndexSearcher</tt>s still in use. Therefore, when we
         * have a 'new current searcher', we must make it aware of it.
         *
         * Failing to do so will mean that an <tt>IndexSearcher</tt> over an old version of the index will continue to be
         * tracked by <tt>SearcherLifetimeManager</tt> as the "current" searcher and kept around for longer than it should.
         * Holding onto an older version of a search index does not consume too much memory if it isn't too different from
         * the current version of the index. However, in the rare event that a merge happens, this will no longer be true,
         * and holding two disparate indexes in memory is undesirable.
         */
        newSearchManager.addListener(new ReferenceManager.RefreshListener()
        {
            @Override
            public void beforeRefresh() throws IOException
            {
            }

            @Override
            public void afterRefresh(boolean didRefresh) throws IOException
            {
                if (didRefresh)
                {
                    IndexSearcher indexSearcher = newSearchManager.acquire();

                    try
                    {
                        record(indexSearcher);
                    }
                    finally
                    {
                        newSearchManager.release(indexSearcher);
                    }
                }
            }
        });

        return newSearchManager;
    }

    private void scheduleIndexSearcherPruneJob(final Configuration configuration)
    {
        long indexSearcherPruneDelay = Long.getLong(TRACKED_SEARCHERS_PRUNE_DELAY, configuration.getIndexSearcherPruneDelay());
        long indexSearcherMaxAge = Long.getLong(TRACKED_SEARCHERS_MAX_AGE, configuration.getIndexSearcherMaxAge());

        log.info("Starting the scheduled service for the prune job of {}", writer.getDirectory());

        scheduledExecutorService = Executors.newSingleThreadScheduledExecutor(
                ThreadFactories.named("lucene-searchers-pruner-" + writer.getDirectory())
                        .type(ThreadFactories.Type.DAEMON)
                        .uncaughtExceptionHandler((t, e) ->
                                log.error("Error thread {} abruptly terminated due to an uncaught exception", t.getName(), e))
                        .build());

        scheduledExecutorService.scheduleAtFixedRate(() -> {
            try
            {
                searcherLifetimeManager.prune(createPruneByAge(configuration.getIndexSearcherMaxAge()));
            }
            catch (Throwable e) // prevent errors and exceptions from escaping and halting any future executions of this task
            {
                log.error("Error pruning IndexSearchers.", e); // log at error level as administrators need to respond.
            }
        }, indexSearcherMaxAge, indexSearcherPruneDelay, TimeUnit.SECONDS);
    }

    private static SearcherLifetimeManager.PruneByAge createPruneByAge(double maxAgeSecs) {
        return new SearcherLifetimeManager.PruneByAge(maxAgeSecs);
    }

    /**
     * Blocks and waits until all write operations to the index complete. Optimizes the index while holding the write
     * lock to ensure no concurrent modifications. Optimize is done using interactive mode configuration.
     */
    public void optimize() throws LuceneException
    {
        withWriter(new WriterAction()
        {
            public void perform(IndexWriter writer) throws IOException
            {
				writer.forceMerge(1, true); // contract of this method says to block until completion
            }
        });
    }

    public void close() throws LuceneException
    {
        assertNotClosed();

        try
        {
            log.info("Closing connection to {}", writer.getDirectory());

            closeWriter();
			searcherManager.close();
            isClosed.set(true);
        }
        catch (IOException e)
        {
            throw new LuceneException(e);
        }

        shutdownScheduledExecutorService();

        closeSearcherLifetimeManager();
    }

    private void shutdownScheduledExecutorService()
    {
        scheduledExecutorService.shutdown();
        try
        {
            if (!scheduledExecutorService.awaitTermination(60, TimeUnit.SECONDS))
            {

                scheduledExecutorService.shutdownNow();
                if (!scheduledExecutorService.awaitTermination(60, TimeUnit.SECONDS))
                    log.debug("Pool did not terminate");
            }
        }
        catch (InterruptedException ie)
        {
            scheduledExecutorService.shutdownNow();
            Thread.currentThread().interrupt();
        }
    }

    private void closeSearcherLifetimeManager()
    {
        searcherLifetimeManagerLock.writeLock().lock();
        try
        {
            searcherLifetimeManager.close();
        }
        catch (IOException e)
        {
            log.debug("Error closing searcherLifetimeManager", e);
        }
        finally
        {
            searcherLifetimeManagerLock.writeLock().unlock();
        }
    }

    @Override
    public void closeWriter() throws LuceneException
    {
        assertNotClosed();
        assertWriterNotClosed();

        try
        {
            writer.close();
            writerIsClosed.set(true);
        }
        catch (IOException e)
        {
            throw new LuceneException(e);
        }
    }

    private void assertNotClosed() throws LuceneException
    {
        if (isClosed.get())
            throw new LuceneConnectionClosedException("Cannot operate on closed " + getClass().getSimpleName());
    }

    private void assertWriterNotClosed() throws LuceneException
    {
        if (writerIsClosed.get())
            throw new LuceneConnectionClosedException("Cannot operate on closed IndexWriter");
    }

    /**
     * This implementation does not respect the boolean return of the of the
     * {@link ILuceneConnection.SearcherAction#perform(org.apache.lucene.search.IndexSearcher)} method
     */
    public void withSearch(SearcherAction action) throws LuceneException
    {
        assertNotClosed();

        // keep a reference to the searcher in case it changes before the end of the search
        IndexSearcher indexSearcher = null;
        try
        {
		    indexSearcher = searcherManager.acquire();
            action.perform(indexSearcher);
        }
        catch (IOException e)
        {
            throw new LuceneException(e);
        }
        finally
        {
			try
			{
                if (indexSearcher != null)
                {
				    searcherManager.release(indexSearcher);
                }
			}
			catch (IOException e)
			{
				throw new LuceneException(e);
			}
		}
    }

    @Override
    public <T> T withSearcher(SearcherWithTokenAction<T> action) throws LuceneException
    {
        assertNotClosed();

        IndexSearcher indexSearcher = null;

        try
        {
            indexSearcher = searcherManager.acquire();

            long searchToken = record(indexSearcher);

            return action.perform(indexSearcher, searchToken);
        }
        catch (IOException e)
        {
            throw new LuceneException(e);
        }
        finally
        {
            try
            {
                if (indexSearcher != null)
                {
                    searcherManager.release(indexSearcher);
                }
            }
            catch (IOException e)
            {
                throw new LuceneException(e);
            }
        }
    }

    /**
     * Returns a token that provides a means for clients to re-use the passed in indexSearcher for further searching.
     *
     * Underneath the covers, the token is just the version number of the search index. With this version number we are
     * able to specify an older index to continue searching over.
     *
     * @param indexSearcher the index searcher
     * @return a token that can be used to retrieve an IndexSearcher over the same index
     * @throws IOException
     */
    private long record(IndexSearcher indexSearcher) throws LuceneException
    {
        searcherLifetimeManagerLock.readLock().lock();
        long searchToken;
        try
        {
            searchToken = searcherLifetimeManager.record(indexSearcher);
        }
        catch (IOException e)
        {
            throw new LuceneException(e);
        }
        finally
        {
            searcherLifetimeManagerLock.readLock().unlock();
        }

        return searchToken;
    }

    @Override
    public <T> T withSearcher(long searchToken, SearcherWithTokenAction<T> action) throws SearchTokenExpiredException
    {
        assertNotClosed();

        if (searchToken <= 0)
            throw new IllegalArgumentException("searchToken must be greater than 0");

        IndexSearcher indexSearcher = searcherLifetimeManager.acquire(searchToken);

        if (indexSearcher == null)
            throw new SearchTokenExpiredException(searchToken);

        try
        {
            return action.perform(indexSearcher, searchToken);
        }
        catch (IOException e)
        {
            throw new LuceneException(e);
        }
        finally
        {
            try
            {
                searcherLifetimeManager.release(indexSearcher);
            }
            catch (IOException e)
            {
                throw new LuceneException(e);
            }
        }
    }

    public Object withReader(final ReaderAction action) throws LuceneException
    {
        final AtomicReference<Object> result = new AtomicReference<Object>();
        withSearch(new SearcherAction()
        {
            public void perform(IndexSearcher searcher) throws IOException
            {
                result.set(action.perform(searcher.getIndexReader()));
            }
        });
        return result.get();
    }

    /**
     * Blocks and waits until all write operations to the index complete. Executes the WriterAction while holding the
     * write lock to ensure no concurrent modifications.
     */
    public void withWriter(WriterAction action) throws LuceneException
    {
        assertWriterNotClosed();
        indexWriteLock.lock();
        try
        {
            configureIndexWriter(writer, configuration);
            try
            {
                action.perform(writer);
            }
            catch (IOException e)
            {
                throw new LuceneException(e);
            }
            // Only refresh the current searcher if the index update succeeded
            commitAndRefreshSearcher();
        }
        finally
        {
            // Release the lock, no matter what.
            indexWriteLock.unlock();
        }
    }

    /**
     * Blocks and waits until all write operations to the index complete. Executes the BatchUpdateAction while holding
     * the write lock to ensure no concurrent modifications. <p></p> Note: This method holds the writeLock for the whole
     * operation, so is used to ensure a set of deletes and writes are effectively executed atomically.
     * <p></p>
     * Refreshes the searcher only once, at the end of the batch update action.
     */
    public void withBatchUpdate(BatchUpdateAction action)
    {
        assertNotClosed();
        assertWriterNotClosed();
        indexWriteLock.lock();
        try
        {
            batchMode.set(true);
            try
            {
                action.perform();
            }
            catch (Exception e) // unfortunately, the API requires us to catch Exception here
            {
                throw new LuceneException(e);
            }
            finally
            {
                batchMode.set(false);
            }
            commitAndRefreshSearcher();
        }
        finally
        {
            indexWriteLock.unlock();
        }
    }

    /**
     * Marks the current searcher to be closed once all searches are finished,
     * and creates a new one for later searches.
     * <p></p>
     * Doesn't refresh the searcher if a batch operation is in progress (i.e.
     * {@link #batchMode} is <tt>true</tt>.
     */
    private void commitAndRefreshSearcher()
    {
        // don't refresh searcher during a batch operation
        if (batchMode.get())
            return;

        searcherRefreshLock.lock();
        try
        {
            writer.commit();
			searcherManager.maybeRefreshBlocking();
        }
        catch (IOException e)
        {
            throw new LuceneException("Error refreshing index searcher", e);
        }
        finally
        {
            searcherRefreshLock.unlock();
        }
    }

    /**
     * Closes the provided reader and logs any exceptions.
     */
    private void closeReader(IndexReader reader)
    {
        if (reader == null)
            return;

        if (log.isDebugEnabled())
		{
			if (reader instanceof DirectoryReader)
			{
				log.debug("Closing index reader: " + ((DirectoryReader) reader).directory());
			}
		}
        try
        {
            reader.close();
        }
        catch (IOException e)
        {
			if (reader instanceof DirectoryReader)
			{
				log.error("Error closing reader: " + ((DirectoryReader) reader).directory(), e);
			}
        }
    }

    private void configureIndexWriter(final IndexWriter indexWriter, Configuration configuration)
    {
		LiveIndexWriterConfig config = indexWriter.getConfig();
        config.setUseCompoundFile(configuration.isCompoundIndexFileFormat());

        if (batchMode.get())
        {
            config.setMaxBufferedDocs(configuration.getBatchMaxBufferedDocs());
        }
        else
        {
            config.setMaxBufferedDocs(configuration.getInteractiveMaxBufferedDocs());
        }
    }

	public void truncateIndex() throws LuceneException
    {
    	withWriter(new ILuceneConnection.WriterAction()
        {
            public void perform(IndexWriter writer) throws IOException
            {
                writer.deleteAll();
            }
        });
    }

    @Override
    public void snapshot(Directory destDir) throws IOException
    {
        assertWriterNotClosed();
        final IndexCommit snapshot = snapshotter.snapshot();
        try
        {
            for (String fileName : snapshot.getFileNames())
            {
                snapshot.getDirectory().copy(destDir, fileName, fileName, IOContext.DEFAULT);
            }
        }
        finally
        {
            snapshotter.release(snapshot);
        }
    }

    @Override
    public void reset(Runnable resetAction) throws LuceneException
    {
        assertWriterNotClosed();

        indexWriteLock.lock();
        try
        {
            log.info("Resetting the connection to {}", writer.getDirectory());

            Directory indexDirectory = writer.getDirectory();
            close();

            if (resetAction != null)
                resetAction.run();

            setUpWriterAndSearcher(indexDirectory);
            scheduleIndexSearcherPruneJob(configuration);

        }
        finally
        {
            // Release the lock, no matter what.
            indexWriteLock.unlock();
        }
    }

    private void setUpWriterAndSearcher(Directory directory) throws LuceneException
    {
        log.info("Setting up the writer and searcher of {}", directory);

        try
        {
            ensureLockOnDirectory(directory);
            ensureCorrectIndexFormat(directory);
            ensureIndexExists(directory);

            IndexWriterConfig config = new IndexWriterConfig(BonnieConstants.LUCENE_VERSION, analyzerForIndexing);
            config.setIndexDeletionPolicy(new SnapshotDeletionPolicy(new KeepOnlyLastCommitDeletionPolicy()));
            writer = new IndexWriter(directory, config);
            snapshotter = (SnapshotDeletionPolicy) writer.getConfig().getIndexDeletionPolicy();
            searcherLifetimeManager = new SearcherLifetimeManager();
            searcherManager = newSearcherManager();

            isClosed.set(false);
            writerIsClosed.set(false);
        }
        catch (IOException e)
        {
            throw new LuceneException(e);
        }
    }

    /**
     * Ensures the index directory exists, contains a Lucene index, and is available
     * for writing.
     */
    private void ensureIndexExists(Directory directory) throws IOException
    {
        indexWriteLock.lock();
        try
        {
            if (!DirectoryReader.indexExists(directory))
            {
                new IndexWriter(directory, new IndexWriterConfig(BonnieConstants.LUCENE_VERSION, null)).close();
            }
        }
        finally
        {
            indexWriteLock.unlock();
        }
    }

    /**
     * At the moment, this only supports upgrading an File based directory to the latest lucene format.
     */
    private void ensureCorrectIndexFormat(Directory directory) throws IOException
    {
        if (!(directory instanceof FSDirectory))
        {
            log.info("Expect FSDirectory. Skip index format check");
            return;
        }

        indexWriteLock.lock();

        try
        {
            upgradeIndexIfNecessary(((FSDirectory) directory).getDirectory(), directory);
        }
        finally
        {
            indexWriteLock.unlock();
        }
    }

    private void ensureLockOnDirectory(Directory directory) throws IOException
    {
        indexWriteLock.lock();

        try
        {
            if (IndexWriter.isLocked(directory))
            {
                // happens if the index was locked by a process which then died (or is still running -- hence the warning)
                log.warn("Forcing unlock of locked index directory: " + directory);
                IndexWriter.unlock(directory);
            }
        }
        finally
        {
            indexWriteLock.unlock();
        }
    }

    private static void upgradeIndexIfNecessary(File directoryPath, Directory directory) throws IOException
    {
        try
        {
            new IndexWriter(directory, new IndexWriterConfig(BonnieConstants.LUCENE_VERSION, null)).close();
        }
        catch (IndexFormatTooOldException e)
        {
            log.info("Detected old index format. Attempting an upgrade.");
            upgradeIndexToLucene36(directoryPath);
            upgradeIndexToCurrentLuceneVersion(directory);
        }
    }

    private static void upgradeIndexToCurrentLuceneVersion(Directory directory) throws IOException
    {
        log.info("Upgrading index to " + BonnieConstants.LUCENE_VERSION.name());
        IndexUpgrader indexUpgrader = new IndexUpgrader(directory, BonnieConstants.LUCENE_VERSION);
        indexUpgrader.upgrade();
        log.info("Index upgraded to " + BonnieConstants.LUCENE_VERSION.name());
    }

    private static void upgradeIndexToLucene36(File directory) throws IOException
    {
        log.info("Upgrading index to Lucene 3.6");
        LuceneIndexUpgrader indexUpgrader = LuceneIndexUpgrader.create(directory);
        indexUpgrader.upgrade();
        log.info("Index upgraded to Lucene 3.6");
    }

    ScheduledExecutorService getScheduledExecutorService() {
        return scheduledExecutorService;
    }
}