package com.atlassian.bonnie;

import org.apache.lucene.document.DateTools;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.text.ParseException;
import java.util.*;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * Utilites for Lucene-related functionality in applications. Be sure to deprecate before removing anything here,
 * as it may be used in applications and plugins.
 */
public class LuceneUtils
{
    private static final Logger log = LoggerFactory.getLogger(LuceneUtils.class);
    private static final AtomicInteger invalidDateWarningCount = new AtomicInteger();
    private static final int INVALID_DATE_MAX_WARNINGS = 20;

    /**
     * Converts a Date into a String using the Lucene index format, with a millisecond resolution.
     *
     * @see DateTools#dateToString
     */
    public static String dateToString(Date date)
    {
        return DateTools.dateToString(date, DateTools.Resolution.MILLISECOND);
    }

    /**
     * Converts a String from the Lucene index format into a Date. Attempts conversion using both
     * new and old date formats. If the String is blank or null, returns a new Date object.
     * <p></p>
     * If the string is in the old format, this method will log a warning a limited number of times
     * per application run to recommend that the user upgrades the index format.
     */
    public static Date stringToDate(String s)
    {
        if (s != null && s.trim().length() > 0)
        {
            try
            {
                return DateTools.stringToDate(s);
            }
            catch (ParseException e)
            {
                int currentErrorCount = invalidDateWarningCount.get();
                if (currentErrorCount <= INVALID_DATE_MAX_WARNINGS) {
                    invalidDateWarningCount.incrementAndGet();
                    log.warn("Unable to parse a date found in the index because it uses an invalid encoding. Rebuilding the search index is recommended.");
                    if (currentErrorCount == INVALID_DATE_MAX_WARNINGS) {
                        log.warn("Suppressing more warnings about invalid dates until the application is restarted.");
                    }
                }

				return new Date(Long.parseLong(s, Character.MAX_RADIX));
            }

        }
        return new Date();
    }

    public static Query buildSingleFieldSingleValueTermQuery(String field, String query)
    {
        return buildSingleFieldMultiValueTermQuery(field, Collections.singletonList(query), true);
    }

    /**
     * Builds a query that checks that the field contains all or some of the values specified.
     * You can set this behaviour by adjusting the value of the addQuery parameter
     *
     * @param andQuery - set to true if you require the value of the field to match ALL values
     */
    public static Query buildSingleFieldMultiValueTermQuery(String field, Collection values, boolean andQuery)
    {
        BooleanQuery query = new BooleanQuery();

        BooleanClause.Occur occur = andQuery ? BooleanClause.Occur.MUST : BooleanClause.Occur.SHOULD;

        for (Iterator iterator = values.iterator(); iterator.hasNext();)
        {
            String value = (String) iterator.next();
            query.add(new TermQuery(new Term(field, value)), occur);
        }

        return query;
    }

    public static Query buildSingleFieldMultiValuePrefixQuery(String field, Collection values, boolean andQuery)
    {
        BooleanQuery query = new BooleanQuery();

        BooleanClause.Occur occur = andQuery ? BooleanClause.Occur.MUST : BooleanClause.Occur.SHOULD;

        for (Iterator iterator = values.iterator(); iterator.hasNext();)
        {
            String value = (String) iterator.next();
            query.add(new PrefixQuery(new Term(field, value)), occur);
        }

        return query;
    }

    /**
     * Builds a Map representation of a Document.
     * This allows dot-style notation in Velocity to be retained for both object- and map-backed
     * representations.
     * <p></p>
     * Dotted fields are split into maps that will lead to the eventual value.
     * For example, content.space.name will resolve to map, to map, to string.
     * This approach also assumes there is no clash between dotted keys and top-level keys,
     * i.e. there aren't both content=foo and content.space.name=bar.
     *
     * @param doc
     * @return
     */
    public static Map buildMapFromDocument(Document doc)
    {
        Map result = new HashMap();
        for (Iterator iter = doc.getFields().iterator(); iter.hasNext();)
        {
            Field f = (Field) iter.next();
            String fieldname = f.name();
            if (fieldname.indexOf('.') > -1)
            {
                String[] split = fieldname.split("\\.");
                Map last = result;
                for (int i = 0; i < split.length; ++i)
                {
                    String key = split[i];
                    if (i == split.length - 1)
                    {
                        last.put(key, f.stringValue()); // terminal case
                    }
                    else
                    {
                        Object temp = last.get(key);
                        if (temp != null && !(temp instanceof Map))
                        {
                            break;
                        }

                        if (temp == null)
                        {
                            temp = new HashMap(5);
                            last.put(key, temp);
                        }
                        last = (Map) temp;
                    }
                }
            }
            else
            {
                result.put(fieldname, f.stringValue());      // assume there's only one value/field
            }
        }
        return result;
    }
}
