/*
 * Copyright (c) 2003 by Atlassian Software Systems Pty. Ltd.
 * All rights reserved.
 */
package com.atlassian.bonnie;

import org.apache.lucene.search.Filter;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.Sort;

import java.util.List;
import java.util.Set;

/**
 * Searches through all objects that have been indexed using a free text query.
 * To index an object allowing it to show up in the search, use Indexer
 * @see com.atlassian.bonnie.Searcher
 *
 * @author   Armond Avanes (armond555@yahoo.com)
 */
public interface Searcher
{
    //~ Methods --------------------------------------------------------------------------------------------------------

    /**
     * Performs the search operation and returns a List of found items.
     */
    public List search(Query query);

    public List search(Query myquery, Sort sort);

    /**
     * Performs a search over multiple fields
     * @param searchFields a sting array of fields to search accross
     * @param query the qury to execute
     * @return a list of Hit results
     */
    public Query buildStandardQuery(String[] searchFields, String query);

    public Query buildStandardQuery(String field, String query);

    /**
     * Search an index and return not only the item handles, but also pull arbitrary fields from the document and return them too.
     * <p></p>
     * Extracting fields from all documents could be expensive and potentially unnecessary, so the startIndex and numItems parameters
     * allow you to control where the extraction will start from, and how many items it will continue for.
     * <p></p>
     *
     * @param myquery The query to run
     * @param fieldsToExtract A set of field names which will be extracted from the document and returned - if the fields are null, or the set is empty, BaseDocumentBuilder.FieldName.HANDLE_FIELD_NAME will be used.
     * @param startIndex The index to start extracting content from, or 0 for the start of the list
     * @param numItems The number of items to be extracted in total, or Integer.MAX_VALUE for all items
     * @return A list containing maps of the fields you wish to extract. For results outside of the startIndex/numItems range, this list will contain null values.
     */
    List searchForFields(Query myquery, Set fieldsToExtract, int startIndex, int numItems);
    List searchForFields(Query myquery, Set fieldsToExtract, int startIndex, int numItems, Filter filter, int[] filteredcount);
    List searchForFields(Query myquery, Set fieldsToExtract, int startIndex, int numItems, Filter filter, Sort sorter, int[] filteredcount);

    List getAllFieldValues(String fieldName);

    /**
     * Run a search, but just give me back the result as a #.
     */
    int searchCount(Query query);

    int searchCount(Query query, Filter filter);

    String explain(Query query, int docid);

    Query rewrite(Query query);
}
