package com.atlassian.bonnie.index;

import java.io.IOException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.atlassian.bonnie.ILuceneConnection;
import com.atlassian.bonnie.search.IndexerThreadFactory;
import com.atlassian.core.util.ProgressMeter;
import com.atlassian.core.util.ProgressWrapper;

/**
 * Multiple-thread implementation of a BatchOpIndexer
 */
public abstract class BaseMultiThreadedIndexer implements BatchOpIndexer
{
    private static Logger log = LoggerFactory.getLogger(BaseMultiThreadedIndexer.class);
    protected ILuceneConnection luceneConnection;
    private boolean reindexing;

    public final void reindex(ObjectQueue queue, DocumentWritingScheme documentWritingScheme,
                              ProgressMeter meter, boolean truncate)
    {
        try
        {
            reindexing = true;
            int numThreads = calculateNumberOfThreads(queue.size());
            ProgressWrapper progress = meter != null ?
                    new ProgressWrapper(meter, queue.size()) : NoOpProgressWrapper.INSTANCE;
            documentWritingScheme.setProgressWrapper(progress);

            if (log.isDebugEnabled())
                log.debug("Starting reindexing thread pool with " + numThreads + " threads");

            ExecutorService executor = Executors.newFixedThreadPool(numThreads,
                new IndexerThreadFactory(getClass().getName()));

            Runnable r = getQueueProcessingRunnable(queue, documentWritingScheme);
            for (int i = 0; i < numThreads; i++)
            {
                executor.execute(r);
            }

            executor.shutdown();
            if (log.isDebugEnabled())
                log.debug("Waiting for queue to shutdown...");
            try
            {
                while (!executor.awaitTermination(60L, TimeUnit.SECONDS))
                {
                    log.warn("Timed out while waiting for reindexing threads to terminate. " +
                        "Continuing to wait...");
                }
            }
            catch (InterruptedException e)
            {
                log.error("Problem in parallelising indexing? " + e, e);
            }

            progress.setPercentage(99);
            progress.setStatus("Merging indices.");

            allThreadsComplete(documentWritingScheme, truncate, progress);

            progress.setPercentage(100);
            progress.setStatus("Finished reindexing " + progress.getTotal() + " objects.");
        }
        finally
        {
            reindexing = false;
        }
    }

    protected Runnable getQueueProcessingRunnable(ObjectQueue queue, DocumentWritingScheme documentWritingScheme)
    {
        return new QueueProcessingRunnableImpl(queue, documentWritingScheme);
    }

    public void truncateIndex() throws IOException
    {
        luceneConnection.truncateIndex();
    }

    /**
     * Perform clean-up operations such as closing writers, optimizing, merging with original index, etc.
     *
     * @param scheme   scheme
     * @param truncate should the original index be truncated
     */
    protected abstract void allThreadsComplete(DocumentWritingScheme scheme, boolean truncate, ProgressWrapper progress);

    /**
     * Determine how many threads to use.
     *
     * @param numObjects the number of objects to index
     * @return number of threads
     */
    protected int calculateNumberOfThreads(int numObjects)
    {
        if (numObjects < 10)
            return 1;
        else if (numObjects < 100)
            return 3;
        else if (numObjects < 500)
            return 5;
        else
            return 10;
    }

    /**
     * Is reindexing currently underway?
     *
     * @return true if the reindex method is currently running.
     */
    protected boolean isReindexing()
    {
        return reindexing;
    }

    public void setReindexing(boolean reindexing)
    {
        this.reindexing = reindexing;
    }
}
